# -*- coding: utf-8 -*-
import json

from flask import (
    request,
    Response,
)
from passport.backend.core.logging_utils.helpers import mask_sensitive_fields


class JsonResponse(Response):
    def __init__(self, *args, **kwargs):
        self.dict_ = dict(*args, **kwargs)
        data = self._build_json(self.dict_)
        super(JsonResponse, self).__init__(data, mimetype='application/json')

    def _build_json(self, *args, **kwargs):
        indent = None
        if not request.is_xhr:
            indent = 2
        return json.dumps(
            dict(*args, **kwargs),
            indent=indent,
        )


class JsonLoggedResponse(JsonResponse):
    def __init__(self, *args, **kwargs):
        sensitive_fields = kwargs.pop('sensitive_fields', None)
        super(JsonLoggedResponse, self).__init__(*args, **kwargs)

        self._logdata = self._build_json(
            mask_sensitive_fields(self.dict_, sensitive_fields)
        )

    def logdata(self):
        return self._logdata


def ok_response(*args, **kwargs):
    return _json_response(200, *args, status='ok', **kwargs)


def simple_error_response(errors, code=200, sensitive_fields=None, **kwargs):
    return _json_response(code, status='error', errors=errors,
                          sensitive_fields=sensitive_fields, **kwargs)


def _json_response(code, *args, **kwargs):
    response = JsonLoggedResponse(*args, **kwargs)
    response.status_code = code
    return response


def format_error(code, message, field=None):
    return {'code': code.lower(), 'message': message, 'field': field}


def format_errors(exc, field=None):
    errors = []

    for field, e in (exc.error_dict or {}).items():
        errors.extend(format_errors(e, field))
    for e in exc.error_list or []:
        errors.extend(format_errors(e, field))

    if not errors:
        errors.append(format_error(exc.code, exc.msg, field))

    return errors


__all__ = (
    'format_error',
    'format_errors',
    'ok_response',
    'simple_error_response',
    '_json_response',
)
