# -*- coding: utf-8 -*-

import json
import logging
import time

from passport.backend.adm_api.common.exceptions import (
    AccessDeniedError,
    InternalError,
)
from passport.backend.core.conf import settings
from passport.backend.core.lazy_loader import (
    lazy_loadable,
    LazyLoader,
)


log = logging.getLogger('passport_adm_api.common.grants')

ROLES_UPDATE_PERIOD = 60  # Обновляем файл с ролями не чаще одного раза в 60 секунд


@lazy_loadable('GrantsLoader')
class GrantsLoader(object):

    def __init__(self, roles_path=None):
        self.roles = None
        self.roles_path = roles_path or settings.ROLES_PATH
        self._load_roles()

    def _build_login_to_grant_ids_mapping(self):
        role_id_to_grants = dict([(role['id'], set(role['grants'])) for role in self.roles['roles']])
        login_to_grant_ids = {}
        for user in self.roles['users']:
            login = user['username']
            grant_set = set()
            for role_id in user['roles']:
                grant_set.update(role_id_to_grants[role_id])
            login_to_grant_ids[login] = grant_set
        self.login_to_grant_ids = login_to_grant_ids

    def _roles_json(self):
        with open(self.roles_path, 'r') as roles_file:
            return json.loads(roles_file.read())

    def _read_roles_file(self):
        try:
            return self._roles_json()
        except (IOError, ValueError) as e:
            log.error('Failed to read roles file: %s', str(e))

    def _load_roles(self):
        roles = self._read_roles_file()
        if roles is not None:
            self.roles = roles
            self.grant_name_to_id = dict([(grant['name'], grant['id']) for grant in self.roles['grants']])
            self.grant_id_to_name = dict([(grant['id'], grant['name']) for grant in self.roles['grants']])
            self._build_login_to_grant_ids_mapping()
            self.update_time = time.time()
            log.debug('Roles successfully loaded from %s', self.roles_path)
        elif not self.roles:
            # нет загруженных ранее данных
            log.error('Roles file not loaded, cannot proceed')
            raise InternalError()

    def _update_roles_if_required(self):
        if not self.roles or time.time() - self.update_time > ROLES_UPDATE_PERIOD:
            self._load_roles()

    def _login_grants(self, login):
        return self.login_to_grant_ids.get(login, set())

    def grants_to_ids(self, grant_names):
        self._update_roles_if_required()
        return set([self.grant_name_to_id[name] for name in grant_names])

    def has_grants(self, login, grants):
        self._update_roles_if_required()
        return grants <= self._login_grants(login)

    def check_grants(self, login, required_grants):
        self._update_roles_if_required()
        if not self.has_grants(login, required_grants):
            grants_delta = [self.grant_id_to_name[grant] for grant in required_grants - self._login_grants(login)]
            log.info('Missing grants for "%s": "%s"', login, grants_delta)
            raise AccessDeniedError()


def get_grants_loader():
    return LazyLoader.get_instance('GrantsLoader')
