# -*- coding: utf-8 -*-
import logging
import os
import time

from flask import request
from passport.backend.core.host.host import get_current_host
from passport.backend.core.logging_utils.helpers import (
    mask_sensitive_fields,
    trim_message,
)
from passport.backend.core.logging_utils.request_id import RequestIdManager


log = logging.getLogger('passport_adm_api.common')
error_log = logging.getLogger('passport_adm_api.error')


def setup_log_prefix():
    request_id = (
        ('%x' % get_current_host().get_id()).upper(),
        os.getpid(),
        time.time(),
    )
    RequestIdManager.clear_request_id()
    RequestIdManager.push_request_id(*request_id)


def log_request():
    log.info(u'Request started: %s %s', request.method, request.path)
    log.info(u'Request args: %s', str(mask_sensitive_fields(request.args.to_dict())))
    log.info(u'Request form: %s', str(mask_sensitive_fields(request.form.to_dict())))


def log_response(response):
    response_data = response.data
    if hasattr(response, 'logdata'):
        response_data = response.logdata()
    response_data = trim_message(response_data)
    log.info(u'Response status: %s', response.status)
    log.info(u'Response sent: "%s"', response_data)

    return response


def log_internal_error(err):
    log.error(u'Request failed with "%s", traceback written to error log.', err)
    error_log.error(
        u'%s on %s %s' % (err.__class__.__name__, request.method, request.path),
        exc_info=err,
        extra={
            'request_info': str(request),
            'request_headers': dict(request.headers),
            'request_values': request.values.to_dict(),
        },
    )


__all__ = (
    'setup_log_prefix',
    'log_request',
    'log_response',
    'log_internal_error',
)
