# -*- coding: utf-8 -*-
import base64
from datetime import datetime
import json
import logging

from passport.backend.adm_api.common.exceptions import (
    AccountInvalidTypeError,
    AccountNotFoundError,
    PublicIdNotAvailable,
    PublicIdNotFoundError,
    UfoUnavailableError,
    YtDtUnavailableError,
)
from passport.backend.adm_api.common.yt_dt import get_yt_dt
from passport.backend.adm_api.views.account.forms import (
    AuthsForm,
    IsVerifiedForm,
    PublicIdForm,
    Sms2FAForm,
    TakeoutDeleteSubscriptionForm,
    TakeoutSubscriptionForm,
    UidForm,
    UidFormWithComment,
)
from passport.backend.adm_api.views.base import BaseView
from passport.backend.core import validators
from passport.backend.core.builders.blackbox import get_blackbox
from passport.backend.core.builders.blackbox.utils import add_phone_arguments
from passport.backend.core.builders.historydb_api import get_historydb_api
from passport.backend.core.builders.passport import get_passport
from passport.backend.core.builders.ufo_api import (
    BaseUfoApiError,
    get_ufo_api,
)
from passport.backend.core.env_profile.profiles import UfoProfile
from passport.backend.core.models.account import (
    Account,
    UnknownUid,
)
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.types.display_name import DisplayName
from passport.backend.core.types.login.login import normalize_login
from passport.backend.utils.time import unixtime_to_datetime
from yt.wrapper import YtError


log = logging.getLogger('passport_adm_api.views.account.controllers')


class AuthsView(BaseView):
    basic_form = AuthsForm

    required_grants = (
        'show_history',
    )

    def process_request(self):
        self.process_basic_form()
        auths, _ = get_historydb_api().auths_aggregated(
            uid=self.form_values['uid'],
            password_auths=self.form_values['password_auths'],
        )

        self.response_values['auths'] = auths


class BotnetiaStatsView(BaseView):
    basic_form = UidForm

    required_grants = (
        'show_person',
    )

    def process_request(self):
        self.process_basic_form()
        try:
            stats = get_yt_dt().get_by_uid(
                uid=self.form_values['uid'],
                columns=[
                    'asname',
                    'bad_karma_regs_to_finished_last_day_asname',
                    'bad_karma_regs_to_finished_last_day_ip',
                    'finished_regs_last_month_phone_number',
                    'finished_regs_last_month_yandexuid',
                    'begin_unixtime',
                    'bt_counter_91',
                    'bt_counter_92',
                    'bt_counter_93',
                    'bt_counter_94',
                    'cookie_age',
                    'different_countries',
                    'duration',
                    'in_field_mean',
                    'print_rate',
                    'yandexuid',
                    'z_score_asname',
                ],
            )
            self.response_values['stats'] = stats
            if not stats:
                return

            stats['bt_counter_92_diff'] = stats['begin_unixtime'] - (
                stats.get('bt_counter_92') or stats['begin_unixtime']
            )
            stats['bt_counter_94_diff'] = stats['begin_unixtime'] - (
                stats.get('bt_counter_94') or stats['begin_unixtime']
            )

        except YtError as exc:
            raise YtDtUnavailableError(exc)


class ProfileView(BaseView):

    basic_form = UidForm

    required_grants = (
        'show_history',
    )

    def process_request(self):
        self.process_basic_form()
        try:
            raw_profile_items = get_ufo_api().profile(self.form_values['uid'])
        except BaseUfoApiError:
            raise UfoUnavailableError()

        items = []
        for raw_item in raw_profile_items:
            data_decoded = json.loads(base64.b64decode(raw_item['data']))
            items.append({'id': raw_item['id'], 'data': data_decoded})

        profile = None
        if items and items[0]['id'] == UfoProfile.PROFILE_FAKE_UUID:
            profile = items.pop(0)['data']

        self.response_values.update(
            profile=profile,
            fresh=items,
        )


class AccountGetterMixin:
    def get_account(self, need_phones=False, **blackbox_kwargs):
        if need_phones:
            blackbox_kwargs = add_phone_arguments(**blackbox_kwargs)
        try:
            return Account().parse(
                get_blackbox().userinfo(
                    uid=self.form_values['uid'],
                    **blackbox_kwargs
                )
            )
        except UnknownUid:
            raise AccountNotFoundError()


class RemoveAllPublicIdView(BaseView, AccountGetterMixin):
    required_grants = (
        'remove_all_public_id',
    )
    basic_form = UidFormWithComment

    def process_request(self):
        self.process_basic_form()
        account = self.get_account(get_hidden_aliases=True)
        with UPDATE(account, self.request.env, self.get_events('remove_all_public_id')):
            if account.public_id_alias:
                account.public_id_alias.alias = None
            if account.user_defined_public_id:
                account.user_defined_public_id = None
            account.public_id_alias.old_public_ids.clear()


class RemovePublicIdView(BaseView, AccountGetterMixin):
    basic_form = PublicIdForm
    required_grants = (
        'remove_public_id',
    )

    def process_request(self):
        self.process_basic_form()
        public_id = self.form_values['public_id']
        normalized_public_id = normalize_login(public_id)

        account = self.get_account(get_hidden_aliases=True)
        with UPDATE(account, self.request.env, self.get_events('remove_public_id')):
            if account.public_id_alias.alias == normalized_public_id:
                account.public_id_alias.alias = None
                account.user_defined_public_id = None
            elif account.public_id_alias.has_old_public_id(normalized_public_id):
                account.public_id_alias.remove_old_public_id(normalized_public_id)
            else:
                raise PublicIdNotFoundError()


class SetPublicIdView(BaseView, AccountGetterMixin):
    basic_form = PublicIdForm
    required_grants = (
        'set_public_id',
    )

    def process_request(self):
        self.process_basic_form()
        public_id = self.form_values['public_id']
        normalized_public_id = normalize_login(public_id)

        account = self.get_account(get_hidden_aliases=True)
        with UPDATE(account, self.request.env, self.get_events('set_public_id')):
            if public_id != account.user_defined_public_id:

                if account.public_id_alias.alias != normalized_public_id:
                    try:
                        availability_validator = validators.PublicIdAvailability(uid=account.uid)
                        availability_validator.to_python(self.form_values, validators.State(self.request.env))
                    except validators.Invalid:
                        raise PublicIdNotAvailable()
                    if account.public_id_alias.alias and not account.public_id_alias.has_old_public_id(account.public_id_alias.alias):
                        account.public_id_alias.add_old_public_id(account.public_id_alias.alias)
                    account.public_id_alias.alias = normalized_public_id

                account.user_defined_public_id = public_id


class IsVerifiedView(BaseView, AccountGetterMixin):
    basic_form = IsVerifiedForm
    required_grants = (
        'set_is_verified',
    )

    def process_request(self):
        self.process_basic_form()
        account = self.get_account()
        with UPDATE(account, self.request.env, self.get_events('set_is_verified')):
            account.is_verified = self.form_values['is_verified']

            if self.form_values['is_verified']:
                account.person.display_name = DisplayName(
                    name=account.person.display_name.public_name,
                )


class AccountPhonishDisableAuth(BaseView, AccountGetterMixin):
    basic_form = UidFormWithComment
    required_grants = (
        'phonish.disable_auth',
    )

    def process_request(self):
        self.process_basic_form()
        account = self.get_account(need_phones=True)

        if not account.is_phonish:
            raise AccountInvalidTypeError()

        with UPDATE(account, self.request.env, self.get_events('disable_phonish_auth')):
            account.global_logout_datetime = datetime.now()
            # У фониша должен быть ровно один номер. Но перестрахуемся на случай всяких ненатуралов.
            for phone in account.phones.all().values():
                # Логически это время "нулевое", то есть меньше него времён не ожидаем. И номер, подтверждённый в это время,
                # гарантированно будет достаточно старым, чтобы фониш не мог по нему авторизоваться.
                # Но технически тут нужно нечто >=1, чтобы не спутать с отсутствием значения.
                phone.confirmed = unixtime_to_datetime(1)


class TakeoutSubscriptionView(BaseView):
    basic_form = TakeoutSubscriptionForm
    required_grants = (
        'set_takeout_subscription',
    )

    def process_request(self):
        self.process_basic_form()
        get_passport().account_options(
            uid=self.form_values['uid'],
            takeout_subscription=self.form_values['takeout_subscription'],
            admin=self.support_account.login,
            comment=self.form_values['comment'] or 'no comment',
        )


class TakeoutDeleteSubscriptionView(BaseView):
    basic_form = TakeoutDeleteSubscriptionForm
    required_grants = (
        'set_takeout_subscription',
    )

    def process_request(self):
        self.process_basic_form()
        subscription = self.form_values['takeout_delete_subscription']
        get_passport().account_options(
            uid=self.form_values['uid'],
            takeout_delete_subscription=subscription,
            admin=self.support_account.login,
            comment=self.form_values['comment'] or 'no comment',
        )


class Sms2FAToggleView(BaseView):
    basic_form = Sms2FAForm
    required_grants = (
        'set_sms_2fa',
    )

    def process_request(self):
        self.process_basic_form()
        get_passport().account_options(
            uid=self.form_values['uid'],
            sms_2fa_on=self.form_values['sms_2fa'],
            forbid_disabling_sms_2fa=self.form_values['forbid_disabling_sms_2fa'],
            admin=self.support_account.login,
            comment=self.form_values['comment'] or 'no comment',
        )
