# -*- coding: utf-8 -*-

from collections import OrderedDict
import logging

from flask.globals import request
from flask.views import View
from passport.backend.adm_api.common import (
    exceptions,
    get_request_values,
    log_internal_error,
    ok_response,
    simple_error_response,
)
from passport.backend.adm_api.common.blackbox import get_intranet_blackbox
from passport.backend.adm_api.common.grants import get_grants_loader
from passport.backend.adm_api.views.headers import (
    HEADER_CLIENT_COOKIE,
    HEADER_CLIENT_HOST,
    HEADER_CONSUMER_AUTHORIZATION,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.core import validators
from passport.backend.core.builders import blackbox
from passport.backend.core.builders.blackbox import (
    BLACKBOX_OAUTH_DISABLED_STATUS,
    BLACKBOX_OAUTH_VALID_STATUS,
    BlackboxInvalidResponseError,
    BlackboxTemporaryError,
)
from passport.backend.core.builders.historydb_api import (
    HistoryDBApiPermanentError,
    HistoryDBApiTemporaryError,
)
from passport.backend.core.builders.meltingpot_api import (
    MeltingpotApiInvalidResponseError,
    MeltingpotApiTemporaryError,
)
from passport.backend.core.builders.passport.exceptions import (
    PassportAccountDisabledError,
    PassportAccountNotFoundError,
    PassportPermanentError,
    PassportTemporaryError,
)
from passport.backend.core.conf import settings
from passport.backend.core.models.account import Account
from passport.backend.core.utils.decorators import cached_property


log = logging.getLogger('passport_adm_api.views.base')

OAUTH_HEADER_PREFIX = 'oauth '
ADM_API_OAUTH_SCOPE = 'passport:adm_api.token_api'


class BaseView(View):
    """
    Базовый класс для всех bundle-вьюшек. Содержит основные свойства и методы,
    полезные всем потомкам. Для непосредственной обработки запроса, класс-потомок
    обязан переопределить метод process_request().
    """

    # Класс основной валидационной формы, с данными которой будет
    # работать потомок. Используется в process_basic_form().
    basic_form = None

    # Список необходимых хедеров
    required_headers = [HEADER_CONSUMER_CLIENT_IP]

    # Список необходимых грантов (выданных посредством ролей в Управляторе)
    required_grants = None

    # По умолчанию идентифицируем пользователя по сессии
    session_required = True
    token_required = False

    exceptions_mapping = OrderedDict([

        (BlackboxTemporaryError, exceptions.BlackboxUnavailableError),
        (BlackboxInvalidResponseError, exceptions.BlackboxPermanentError),

        (HistoryDBApiPermanentError, exceptions.HistoryDBPermanentError),
        (HistoryDBApiTemporaryError, exceptions.HistoryDBApiUnavailableError),

        (MeltingpotApiTemporaryError, exceptions.MeltingpotUnavailableError),
        (MeltingpotApiInvalidResponseError, exceptions.MeltingpotUnavailableError),

        (PassportAccountDisabledError, exceptions.AccountDisabledError),
        (PassportAccountNotFoundError, exceptions.AccountNotFoundError),
        (PassportTemporaryError, exceptions.PassportUnavailableError),
        (PassportPermanentError, exceptions.PassportPermanentError),
    ])

    # Объединённый словарь с данными из request_values и path_values.
    @cached_property
    def all_values(self):
        values = self.request_values
        values.update(self.path_values)
        return values

    # Возвращает словарь с данными из запроса: для GET-запросов выдаёт данные из
    # query-части, для POST-запросов - из body-части.
    @cached_property
    def request_values(self):
        return get_request_values()

    @cached_property
    def grants_loader(self):
        return get_grants_loader()

    def required_grant_ids(self, required_grants):
        return self.grants_loader.grants_to_ids(required_grants or [])

    @property
    def request(self):
        return request

    @property
    def headers(self):
        return request.headers

    @property
    def cookies(self):
        return request.env.cookies

    @property
    def consumer_ip(self):
        return request.env.consumer_ip

    @property
    def client_ip(self):
        return request.env.user_ip

    @property
    def host(self):
        return request.env.host

    @property
    def user_agent(self):
        return request.env.user_agent

    @property
    def authorization(self):
        return request.env.authorization

    @cached_property
    def oauth_token(self):
        auth_header = self.authorization
        if not auth_header.lower().startswith(OAUTH_HEADER_PREFIX):
            raise exceptions.AuthorizationHeaderError()
        return auth_header[len(OAUTH_HEADER_PREFIX):].strip()

    @classmethod
    def as_view(cls, name=None, *args, **kwargs):
        name = name or cls.__name__
        return super(BaseView, cls).as_view(name, *args, **kwargs)

    def __init__(self):
        # Аккаунт пользователя админки
        self.support_account = None

        # Словарь с данными, полученными при валидации основной формы в process_basic_form().
        self.form_values = {}

        # Словарь с данными, которые вытащил роутер из path-части урла запроса.
        self.path_values = {}

        # Словарь для хранения данных, которые будут выданы в http-ответе как json.
        self.response_values = {}

        if self.session_required:
            self.required_headers = self.required_headers + [HEADER_CLIENT_COOKIE, HEADER_CLIENT_HOST]

        if self.token_required:
            self.required_headers = self.required_headers + [HEADER_CONSUMER_AUTHORIZATION]

    def dispatch_request(self, **kwargs):
        """
        Первоочерёдный метод вьюшки, обрабатывающий пришедший запрос.

        Запоминает входящие параметры как path_values, то есть данные из path-части
        урла запроса, полученные роутером. Вызывает process_request(), переопределённый
        в классе-потомке. В случае выброшенного исключения, включает его обработку
        и выдачу ошибочного ответа. Если исключения не было, выдаёт успешный ответ.

        @param kwargs: Словарь с данными из path-части урла.
        """
        self.path_values = kwargs

        try:
            if self.required_headers:
                self.check_headers(self.required_headers)

            if self.session_required:
                self.check_session()

            if self.token_required:
                self.check_oauth_token()

            if self.required_grants:
                self.check_grants(self.required_grants)

            self.process_request()
        except Exception as e:
            return self.respond_error(e)

        return self.respond_success()

    def process_request(self):
        """
        Основной метод вьюшки, обрабатывающий пришедший запрос. Должен быть
        переопределен в классе-потомке.
        """
        raise NotImplementedError()

    def process_error(self, exception):
        # Небольшая оптимизация. Однако отсутствие класса в отображении не отменяет
        # полного перебора классов и проверки на isinstance, так как в exceptions_mapping
        # может быть "родитель" текущего исключения.
        response_exc = self.exceptions_mapping.get(exception.__class__)
        if response_exc is not None:
            return response_exc()

        for original_exc, response_exc in self.exceptions_mapping.items():
            if isinstance(exception, original_exc):
                return response_exc()

        if isinstance(exception, exceptions.BaseAdmError):
            return exception

        return exceptions.UnhandledError()

    def respond_error(self, exception):
        """
        Генерирует и отдаёт ошибочный json-ответ с данными из response_values и
        дополнительными полями status=error и errors=[коды-ошибок].

        Указанное исключение должно быть обработано в методе process_error
        и преобразовано к потомку BaseBundleError в котором переопределены свойства error и errors
        - именно они используются как диагностика ошибки в ответе. Если исключение не отдаёт
        эти коды ошибок, process_error должно вернуть UnhandledBundleError
        со стандартным кодом ошибки - 'exception.unhandled', а само неизвестное исключение
        будет залогировано.

        Есть специфичные случаи:
        - ошибки при недоступности ЧЯ, YaSMS, DB, Redis преобразуются в
        backend.%s_failed.

        @param exception: Исключение для выдачи правильной диагностики полученной ошибки.
        @return: Объект JsonLoggedResponse.
        """

        # Обработаем ошибки, получив предусмотренное view-исключение
        processed_error = self.process_error(exception)

        errors = processed_error.errors

        # Запишем в лог необработанное исключение
        if isinstance(processed_error, exceptions.UnhandledError):
            log_internal_error(exception)

        return simple_error_response(errors, **self.response_values)

    def respond_success(self):
        """
        Генерирует и отдаёт успешный json-ответ с данными из response_values и
        дополнительным полем status=ok.

        @return: Объект JsonLoggedResponse.
        """
        return ok_response(**self.response_values)

    def check_header(self, header):
        """
        Проверяет наличие обязательного хедера в запросе. Если хедера нет,
        кидает исключение HeadersEmptyError.

        @param header: хедер из .headers
        @raise: HeadersEmptyError
        """
        self.check_headers([header])

    def check_headers(self, headers):
        """
        Проверяет наличие обязательных хедеров в запросе. Кидает исключение
        HeadersEmptyError с перечислением отсутствующих хедеров.

        @param headers: Список хедеров из .headers
        @raise: HeadersEmptyError
        """
        codes = [
            header.code_for_error for header in headers
            if (
                header.name not in self.headers or
                (not self.headers.get(header.name) and not header.allow_empty_value)
            )
        ]

        if codes:
            raise exceptions.HeadersEmptyError(codes)

    def check_session(self):
        """
        Проверяет сессию пользователя во внутреннем ЧЯ.
        """
        sessionid = self.cookies.get('Session_id')
        ssl_sessionid = self.cookies.get('sessionid2')
        if not sessionid:
            raise exceptions.SessionidInvalidError()
        bb_response = get_intranet_blackbox().sessionid(
            sessionid=sessionid,
            ip=self.client_ip,
            host=self.host,
            sslsessionid=ssl_sessionid,
            dbfields=settings.BLACKBOX_LOGIN_FIELDS,
            attributes=settings.BLACKBOX_LOGIN_ATTRIBUTES,
            need_public_name=False,
        )
        status = bb_response['cookie_status']
        if status not in (
            blackbox.BLACKBOX_SESSIONID_VALID_STATUS,
            blackbox.BLACKBOX_SESSIONID_NEED_RESET_STATUS,
        ) or settings.HTTPS_ONLY and not bb_response['auth']['secure']:
            raise exceptions.SessionidInvalidError()

        self.support_account = Account().parse(bb_response)

    def check_oauth_token(self):
        """
        Проверяет oauth-токен пользователя во внутреннем ЧЯ.
        """
        bb_response = get_intranet_blackbox().oauth(
            self.oauth_token,
            ip=self.client_ip,
            dbfields=settings.BLACKBOX_LOGIN_FIELDS,
            attributes=settings.BLACKBOX_LOGIN_ATTRIBUTES,
            need_public_name=False,
        )
        if bb_response['status'] == BLACKBOX_OAUTH_DISABLED_STATUS:
            raise exceptions.AccountDisabledError()

        if (bb_response['status'] != BLACKBOX_OAUTH_VALID_STATUS or
                ADM_API_OAUTH_SCOPE not in bb_response['oauth']['scope']):
            raise exceptions.OAuthTokenValidationError()

        self.support_account = Account().parse(bb_response)

    def check_grants(self, required_grants):
        required_grant_ids = self.required_grant_ids(required_grants)
        login = self.support_account.login
        self.grants_loader.check_grants(login, required_grant_ids)

    def is_grant_available(self, grant):
        grant_ids = self.required_grant_ids([grant])
        login = self.support_account.login
        return self.grants_loader.has_grants(login, grant_ids)

    def process_form(self, form, values):
        """
        Валидирует данные с помощью указанной формы.

        Если валидация пройдена успешно, возвращает результат обработки. Обычно,
        это те же данные, но слегка обработанные: у полей убраны лишние пробелы,
        некоторые могут быть преобразованы в сложные объекты и так далее.

        Если валидация пройдена с ошибками, кидает исключение ValidationFailedError.
        С помощью него можно получить список всех найденных ошибок.

        @param form: Объект формы, с помощью которой будет проводиться
        валидация. Форма должна быть отнаследована от passport.backend.core.validators.Schema
        (formencode.Scheme только с нашими патчами).
        @param values: Словарь с данными, которые нужно прогнать через форму.
        @return: Обработанный словарь с данными.
        @raise: ValidationFailedError
        """
        state = validators.State(self.request.env)

        try:
            result = form.to_python(values, state)
        except validators.Invalid as e:
            exception = exceptions.ValidationFailedError(e)
            log.info('Form validation: status=error form=%s errors=%s',
                     form.__class__.__name__, ','.join(exception.errors))
            raise exception

        log.info('Form validation: status=ok form=%s', form.__class__.__name__)

        return result

    def process_basic_form(self):
        """
        Валидирует весь запрос с помощью основной формы, определённой в потомке,
        и сохраняет результат в form_values.
        """
        self.form_values = self.process_form(self.basic_form(), self.all_values)

    def get_events(self, action):
        events = {
            'action': action,
            'admin': self.support_account.login,
        }
        if 'comment' in self.form_values and self.form_values['comment']:
            events['comment'] = self.form_values['comment']
        return events
