# -*- coding: utf-8 -*-

import re

from passport.backend.adm_api.common.exceptions import (
    MeltingpotHandlerNotFound,
    MeltingpotUnavailableError,
    ProxyError,
)
from passport.backend.adm_api.views.base import BaseView
from passport.backend.core.builders.meltingpot_api import get_meltingpot_api


class MeltingpotProxyView(BaseView):

    required_grants = []
    session_required = True

    ENDPOINTS = list(map(
        lambda xs: (xs[0], re.compile('^%s$' % xs[1]), xs[2]),
        [
            # method, url, grants
            ('POST', '1/users/', ('allow_meltingpot_user_add', )),
            ('POST', '1/users/many/', ('allow_meltingpot_user_add', )),
            ('GET', '1/users/updates/', ('show_meltingpot_users', )),
            ('GET', '1/users/\d+/', ('show_meltingpot_users', )),

            ('GET', '1/statistics/', ('show_meltingpot_statistics', )),
            ('GET', '1/statistics/daily/', ('show_meltingpot_statistics', )),
            ('GET', '1/queue/stats/', ('show_meltingpot_statistics', )),

            ('GET', '1/groups/', ('show_meltingpot_group', )),
            ('POST', '1/groups/', ('allow_meltingpot_group', )),

            ('GET', '1/groups/\d+/', ('show_meltingpot_group', )),
            ('POST', '1/groups/\d+/', ('allow_meltingpot_group', )),
            ('DELETE', '1/groups/\d+/', ('allow_meltingpot_group', )),

            ('GET', '1/groups/\d+/stats/', ('show_meltingpot_group', )),
            ('POST', '1/groups/\d+/drain/load/', ('allow_meltingpot_group', )),
            ('POST', '1/groups/\d+/drain/check/', ('allow_meltingpot_group', )),
            ('POST', '1/groups/\d+/drain/move/', ('allow_meltingpot_group', )),
            ('POST', '1/groups/\d+/drain/state/', ('allow_meltingpot_group', )),

            ('GET', '1/schedule/basic/', ('show_meltingpot_schedule', )),
            ('POST', '1/schedule/basic/', ('allow_meltingpot_schedule', )),
            ('GET', '1/schedule/day/', ('show_meltingpot_schedule', )),
            ('GET', '1/schedule/day/range/', ('show_meltingpot_schedule', )),
            ('GET', '1/schedule/exceptional/', ('show_meltingpot_schedule', )),
            ('GET', '1/schedule/exceptional/range/', ('show_meltingpot_schedule', )),
            ('POST', '1/schedule/exceptional/', ('allow_meltingpot_schedule', )),
            ('DELETE', '1/schedule/exceptional/', ('allow_meltingpot_schedule', )),
        ],
    ))

    def check_endpoint_exists(self, method, path):
        for endpoint_method, endpoint_url, required_grants in self.ENDPOINTS:
            if endpoint_method == method and endpoint_url.match(path):
                return True, required_grants
        return False, None

    def process_request(self):
        path = self.path_values['path']
        endpoint_exists, required_grants = self.check_endpoint_exists(self.request.method, path)
        if not endpoint_exists:
            raise MeltingpotHandlerNotFound()

        self.check_grants(required_grants)

        files = {
            name: (file.filename, file.stream, file.content_type)
            for name, file in self.request.files.items()
        }

        response = get_meltingpot_api().proxy(
            self.request.method,
            path,
            self.request.args.to_dict(),
            self.request.form.to_dict(),
            files,
        )

        if response['status'] == 'error':
            if 'exception.unhandled' in response['errors']:
                raise MeltingpotUnavailableError()
            else:
                raise ProxyError(response['errors'])

        response.pop('status')
        self.response_values.update(response)
