# -*- coding: utf-8 -*-
import json
import logging
from time import time

from passport.backend.adm_api.common.blackbox import get_blackbox
from passport.backend.adm_api.common.exceptions import AccountNotFoundError
from passport.backend.adm_api.common.historydb_api import get_historydb_events_info
from passport.backend.adm_api.views.base import BaseView
from passport.backend.adm_api.views.restore.forms import RestoreAccountStateForm
from passport.backend.adm_api.views.restore.helpers import (
    format_timestamp,
    get_restore_events,
)
from passport.backend.core.builders.historydb_api import get_historydb_api
from passport.backend.core.db.read_api.aliases import (
    find_removed_aliases_by_uid,
    find_uids_by_removed_aliases,
)
from passport.backend.core.eav_type_mapping import ALIAS_NAME_TO_TYPE
from passport.backend.core.historydb.analyzer.event_handlers.account_state import (
    ACCOUNT_STATUS_DELETED,
    ACCOUNT_STATUS_DELETED_BY_SUPPORT,
    ACCOUNT_STATUS_DISABLED,
    ACCOUNT_STATUS_DISABLED_ON_DELETE,
    ACCOUNT_STATUS_LIVE,
    ACCOUNT_STATUS_LIVE_UNBLOCKED,
)
from passport.backend.core.models.account import (
    Account,
    UnknownUid,
)
from passport.backend.core.serializers.eav.subscription import SID_TO_ALIAS_NAME
from passport.backend.core.services import get_service
from passport.backend.core.subscription import get_blocking_sids
from passport.backend.core.types.account.account import (
    ACCOUNT_TYPE_LITE,
    ACCOUNT_TYPE_NORMAL,
    ACCOUNT_TYPE_PDD,
    ACCOUNT_TYPE_PHONISH,
    ACCOUNT_TYPE_SOCIAL,
)
from passport.backend.utils.time import DEFAULT_FORMAT


log = logging.getLogger('passport_adm_api.views.restore.account_state')


ACCOUNT_TYPE_TO_STRING = {
    ACCOUNT_TYPE_NORMAL: 'normal',
    ACCOUNT_TYPE_PDD: 'pdd',
    ACCOUNT_TYPE_LITE: 'lite',
    ACCOUNT_TYPE_SOCIAL: 'social',
    ACCOUNT_TYPE_PHONISH: 'phonish',
}


def _get_ts_and_ip_from_event(event):
    if event:
        return dict(
            timestamp=event['timestamp'],
            datetime=format_timestamp(event['timestamp']),
            user_ip=event.get('user_ip'),
        )


def _get_create_delete_history(uid):
    result = {'uid': uid}
    events_info = get_historydb_events_info(uid, account_create_delete_events=True)
    result['created'] = _get_ts_and_ip_from_event(events_info.account_create_event)
    result['deleted'] = _get_ts_and_ip_from_event(events_info.account_delete_event)
    return result


def _get_login_from_removed_aliases(removed_aliases):
    """
    Ищем основной логин в удаленных алиасах.
    """
    for alias_name in ('portal', 'pdd', 'lite', 'social', 'phonish'):
        logins = removed_aliases.get(alias_name)
        if logins:
            return logins[0], alias_name
    raise AccountNotFoundError()


def _get_non_natural_aliases(userinfo_response, login):
    """
    Ищем алиасы типов mail, narodmail в аккаунте.
    """
    aliases = userinfo_response.get('aliases', {})
    non_natural_aliases = {}
    for alias_name in SID_TO_ALIAS_NAME.values():
        alias_type = str(ALIAS_NAME_TO_TYPE[alias_name])
        alias = aliases.get(alias_type)
        if alias and alias != login:
            non_natural_aliases[alias_name] = alias
    return non_natural_aliases


def _get_non_natural_removed_aliases(removed_aliases):
    """
    Ищем алиасы типов mail, narodmail в удаленных алиасах.
    """
    non_natural_aliases = {}
    for alias_name in SID_TO_ALIAS_NAME.values():
        logins = removed_aliases.get(alias_name)
        if logins:
            non_natural_aliases[alias_name] = logins
    return non_natural_aliases


def _get_incomplete_password_changing(password_change_requests, password_changes):
    if password_change_requests:
        last_change_request = password_change_requests[-1]
        if not last_change_request['change_required']:
            # Запрос на смену пароля был отменен
            return
        ts = last_change_request['origin_info']['timestamp']
        password_change_timestamps = [change['origin_info']['timestamp'] for change in password_changes]
        if not password_change_timestamps or password_change_timestamps[-1] < ts:
            # Не было смены пароля после требования сменить пароль
            last_change_request.pop('origin_info')
            return dict(last_change_request, datetime=format_timestamp(ts), timestamp=ts)


class RestoreAccountStateView(BaseView):

    basic_form = RestoreAccountStateForm

    required_grants = (
        'allow_search',
        'show_person',
        'show_history',
    )

    def _find_account(self, uid):
        """
        Пытаемся найти аккаунт по UID'у в ЧЯ и в удаленных алиасах.
        """
        self.userinfo_response = get_blackbox().userinfo(uid, need_public_name=False)
        self.removed_aliases = find_removed_aliases_by_uid(uid)
        self.account = None
        try:
            self.account = Account().parse(self.userinfo_response)
            self.login = self.account.login
            self.account_type = ACCOUNT_TYPE_TO_STRING[self.account.type]
            self.response_values['login'] = self.account.display_login or self.account.login
        except UnknownUid:
            self.login, alias_type = _get_login_from_removed_aliases(self.removed_aliases)
            self.account_type = alias_type if alias_type != 'portal' else 'normal'
            self.response_values['login'] = self.login
        self.response_values['type'] = self.account_type

    def _fill_non_natural_aliases(self):
        """
        Проверяем ненатуральность аккаунта.
        """
        non_natural_aliases = {}
        if self.account is not None:
            non_natural_aliases = _get_non_natural_aliases(self.userinfo_response, self.account.login)
        self.response_values['non_natural_aliases'] = non_natural_aliases
        self.response_values['non_natural_removed_aliases'] = _get_non_natural_removed_aliases(self.removed_aliases)

    def _check_history_and_blackbox_consistency(self, status_from_history, uid):
        """
        Предосторожность на случай лага HistoryDB. Например, аккаунт разблокировали, до истории это
        событие еще не доехало, в ЧЯ уже есть.
        """
        account_status = ACCOUNT_STATUS_DELETED if self.account is None else (
            ACCOUNT_STATUS_LIVE if self.account.is_enabled else ACCOUNT_STATUS_DISABLED
        )
        allowed_history_statuses = {
            ACCOUNT_STATUS_LIVE: {ACCOUNT_STATUS_LIVE, ACCOUNT_STATUS_LIVE_UNBLOCKED},
            ACCOUNT_STATUS_DISABLED: {ACCOUNT_STATUS_DISABLED_ON_DELETE, ACCOUNT_STATUS_DISABLED},
            ACCOUNT_STATUS_DELETED: {ACCOUNT_STATUS_DELETED, ACCOUNT_STATUS_DELETED_BY_SUPPORT},
        }[account_status]
        self.response_values['account_status_consistent'] = True
        if status_from_history not in allowed_history_statuses:
            log.warning(
                'Account status "%s" is inconsistent with history status "%s" for UID %s',
                account_status,
                status_from_history,
                uid,
            )
            self.response_values['account_status_consistent'] = False

    def _fill_account_options(self):
        """
        Если аккаунт не удален, то заполним данные, получаемые из ЧЯ
        """
        bb_info = {}
        if self.account:
            if self.account.yandexoid_alias.alias:
                bb_info['yastaff_login'] = self.account.yandexoid_alias.alias
            bb_info['have_password'] = self.account.have_password
            bb_info['is_2fa_enabled'] = bool(self.account.totp_secret.is_set)
            bb_info['karma_status'] = self.account.karma.value
            bb_info['blocking_sids'] = [
                '%s (%s)' % (sid, get_service(sid).slug)
                for sid in get_blocking_sids(self.account)
                if self.account.is_subscribed(get_service(sid))
            ]
            bb_info['status'] = ACCOUNT_STATUS_LIVE if self.account.is_enabled else ACCOUNT_STATUS_DISABLED
        else:
            bb_info['status'] = ACCOUNT_STATUS_DELETED
        self.response_values['account_options'] = bb_info

    def _fill_same_login_history(self, uid):
        # ищем в removed_aliases удаленные уиды по основному логину
        removed_uids = find_uids_by_removed_aliases(self.login)
        # по удаленным уидам в historydb ищем события регистрации и удаления
        same_login_history = []
        for removed_uid in removed_uids:
            if uid == removed_uid:
                continue
            same_login_history.append(_get_create_delete_history(removed_uid))
        self.response_values['same_login_history'] = same_login_history

    def _fill_incomplete_password_changing(self, events_info):
        self.response_values['incomplete_password_changing'] = _get_incomplete_password_changing(
            events_info.password_change_requests,
            events_info.password_changes,
        )

    def _fill_account_status_from_history(self, events_info, uid):
        history_status = events_info.account_enabled_status
        # добавим в вывод событий отформатированное представление даты
        history_status['karma_events'] = [
            dict(event, datetime=format_timestamp(event['timestamp'])) for event in history_status['karma_events']
        ]
        if 'timestamp' in history_status:
            history_status['datetime'] = format_timestamp(history_status['timestamp'])
        self.response_values['account_history_status'] = history_status
        account_status_from_history = self.response_values['account_history_status']['status']
        self._check_history_and_blackbox_consistency(account_status_from_history, uid)

    def fill_restore_events(self, uid, current_restore_id=None):
        result = []
        events = get_restore_events(uid)
        for event in events:
            restore_id = event['restore_id']
            if restore_id == current_restore_id:
                continue
            ts = event['timestamp']
            event_data = json.loads(event['data_json'])
            result.append({
                'restore_id': restore_id,
                'contact_email': event_data.get('request_info', {}).get('contact_email'),
                'timestamp': ts,
                'datetime': format_timestamp(ts, format=DEFAULT_FORMAT)
            })
        self.response_values['other_restore_events'] = result

    def fill_challenge_info(self, uid, current_restore_ts=None):
        challenges = get_historydb_api().auths_failed(
            uid=uid,
            from_ts=0,
            to_ts=int(float(current_restore_ts or time())),
            limit=1,
            status='challenge_shown',
        )
        challenge_info = None
        if challenges:
            challenge_info = _get_ts_and_ip_from_event(challenges[0])
        self.response_values['challenge_shown'] = challenge_info

    def process_request(self):
        self.process_basic_form()
        restore_id = self.form_values['restore_id']
        uid = self.form_values['uid']
        if uid is None:
            uid = restore_id.uid
        self.response_values['uid'] = uid

        self._find_account(uid)

        self._fill_same_login_history(uid)
        self._fill_non_natural_aliases()

        # В historydb ищем карму, статус блокировки и удаления, тип аккаунта, принуждалки к смене пароля
        events_info = get_historydb_events_info(
            uid,
            account=self.account,
            account_enabled_status=True,
            password_changes=True,
        )

        self._fill_incomplete_password_changing(events_info)
        self._fill_account_status_from_history(events_info, uid)
        self._fill_account_options()
        self.fill_restore_events(uid=uid, current_restore_id=restore_id.to_string() if restore_id else None)
        self.fill_challenge_info(uid=uid, current_restore_ts=restore_id.timestamp if restore_id else None)
