# -*- coding: utf-8 -*-
import json

from passport.backend.adm_api.common.blackbox import get_blackbox
from passport.backend.adm_api.common.exceptions import (
    AccountNotFoundError,
    ActionNotAllowedError,
    PassportPermanentError,
    ProxyError,
    RestoreIdNotFoundError,
    RestoreIdUnknownVersionError,
)
from passport.backend.adm_api.common.historydb_api import get_historydb_events_info
from passport.backend.adm_api.common.statbox import AdmStatboxLogger
from passport.backend.adm_api.views.base import BaseView
from passport.backend.core.builders.passport import get_passport
from passport.backend.core.historydb.events import (
    ACTION_RESTORE_SEMI_AUTO_DECISION,
    EVENT_ACTION,
    EVENT_INFO_RESTORE_ID,
    EVENT_INFO_RESTORE_STATUS,
    RESTORE_STATUS_PASSED,
    RESTORE_STATUS_PENDING,
    RESTORE_STATUS_REJECTED,
)
from passport.backend.core.host.host import get_current_host
from passport.backend.core.models.account import (
    Account,
    UnknownUid,
)
from passport.backend.core.serializers.logs.historydb.runner import HistorydbActionRunner
from passport.backend.core.support_link_types import SUPPORT_LINK_TYPE_CHANGE_PASSWORD_SET_METHOD
from passport.backend.core.types.restore_id import RestoreId
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.time import DEFAULT_FORMAT

from .forms import (
    RestoreSemiAutoAttemptForm,
    RestoreSemiAutoAttemptsForm,
    RestoreSemiAutoSupportDecisionForm,
)
from .helpers import (
    format_restore_attempt,
    format_timestamp,
    get_extra_info_from_attempt,
    get_restore_events,
)


def _format_attempt_info_timestamps(attempt):
    attempt['datetime'] = format_timestamp(attempt['timestamp'])
    for decision in attempt['support_decisions']:
        decision['datetime'] = format_timestamp(decision['timestamp'])
    return attempt


def _get_restore_attempt_info(restore_id):
    attempts = get_historydb_events_info(
        uid=restore_id.uid,
        restore_semi_auto_attempts=True,
    ).restore_semi_auto_attempts
    attempts = [attempt for attempt in attempts if RestoreId.from_string(attempt['restore_id']) == restore_id]
    return _format_attempt_info_timestamps(attempts[0]) if attempts else None


class RestoreSemiAutoAttemptView(BaseView):

    basic_form = RestoreSemiAutoAttemptForm

    required_grants = (
        'allow_search',
        'show_history',
        'show_phones',
        'show_emails',
        'show_restoration_form',
    )

    def process_request(self):
        self.process_basic_form()
        restore_id = self.form_values['restore_id']
        restore_events = get_restore_events(restore_id.uid)
        can_show_answers = self.is_grant_available('show_hint_answer')

        for event in restore_events:
            if RestoreId.from_string(event['restore_id']) == restore_id:
                raw_data = json.loads(event.pop('data_json'))
                event['datetime'] = format_timestamp(event.pop('timestamp'), format=DEFAULT_FORMAT)
                attempt_data = format_restore_attempt(raw_data, can_show_answers)
                if attempt_data is None:
                    raise RestoreIdUnknownVersionError()
                event.update(attempt_data)
                self.response_values['restore_attempt'] = event

                # заполним информацию о возможном решении саппорта
                attempt_info = _get_restore_attempt_info(restore_id)
                initial_support_decision = support_decision = None
                if attempt_info and attempt_info.get('support_decisions'):
                    # разрешается максимум два решения саппорта; в данной ручке интересно самое
                    # последнее решение и самое первое.
                    support_decision = attempt_info['support_decisions'][-1]
                    initial_support_decision = attempt_info['support_decisions'][0]
                self.response_values['restore_attempt'].update(
                    initial_support_decision=initial_support_decision,
                    support_decision=support_decision,
                )
                return

        raise RestoreIdNotFoundError()


class RestoreSemiAutoAttemptExistsView(BaseView):

    session_required = False
    token_required = True

    basic_form = RestoreSemiAutoAttemptForm

    required_grants = (
        'allow_search',
    )

    def process_request(self):
        self.process_basic_form()
        uid = self.form_values['restore_id'].uid
        events = get_restore_events(uid)

        for event in events:
            if RestoreId.from_string(event['restore_id']) == self.form_values['restore_id']:
                self.response_values['restore_attempt_exists'] = True
                return
        self.response_values['restore_attempt_exists'] = False


class RestoreSemiAutoAttemptsView(BaseView):

    basic_form = RestoreSemiAutoAttemptsForm

    required_grants = (
        'allow_search',
        'show_history',
    )

    def process_request(self):
        self.process_basic_form()
        uid = self.form_values['uid']
        if uid is None:
            uid = self.form_values['restore_id'].uid

        events_info = get_historydb_events_info(uid=uid, restore_semi_auto_attempts=True)
        self.response_values['restore_attempts'] = [
            _format_attempt_info_timestamps(attempt) for attempt in events_info.restore_semi_auto_attempts
        ]


class RestoreSemiAutoSupportDecisionView(BaseView):

    basic_form = RestoreSemiAutoSupportDecisionForm

    required_grants = (
        'allow_restoration_link_create',
    )

    @property
    def restore_id(self):
        return self.form_values['restore_id']

    @cached_property
    def statbox(self):
        return AdmStatboxLogger(
            ip=self.client_ip,
            host=self.host,
            user_agent=self.user_agent,
            restore_id=self.restore_id.to_string(),
            support_uid=self.support_account.uid,
            support_login=self.support_account.login,
            track_id=self.restore_id.track_id,
            uid=self.restore_id.uid,
        )

    def _assert_decision_applicable(self, attempt, is_for_learning):
        """
        Проверить, что данный restore_id актуален и его статус подходит для принятия решения:
        - статус автоматической обработки анкеты с заданным restore_id - pending, для анкет на обучение статус не важен
        - для restore_id еще не принято решение, либо меняется изначальное решение с отрицательного на положительное,
        либо выполняется повторная генерация ссылки в случае положительного решения (для анкет на обучение не важно)
        Возвращает признак того, меняется ли решение по анкете.
        """
        if attempt['initial_status'] != RESTORE_STATUS_PENDING and not is_for_learning:
            raise ActionNotAllowedError()
        support_decisions = attempt['support_decisions']

        if not support_decisions or is_for_learning:
            # ранее не было сделано решения, либо анкета для обучения
            if self.form_values['regenerate_link']:
                raise ActionNotAllowedError()
            return False

        # обрабатываем случай существования хотя бы одного решения
        last_decision_status = support_decisions[-1]['status']
        if (last_decision_status == RESTORE_STATUS_PASSED and
                self.form_values['passed'] and
                self.form_values['regenerate_link']):
            # перегенерация ссылки: последнее решение положительное и переданы соответствующие параметры
            # условия такие строгие, т.к. перегенерация ссылки влечет за собой блокировку аккаунта
            return False
        if self.form_values['regenerate_link']:
            # любые другие параметры в сочетании с запросом перегенерации ссылки недопустимы
            raise ActionNotAllowedError()
        if (len(support_decisions) > 1 or
                not (last_decision_status == RESTORE_STATUS_REJECTED and self.form_values['passed'])):
            # - решение можно изменить не более одного раза
            # - решение можно изменить только с отрицательного на положительное
            raise ActionNotAllowedError()
        # остается случай изменения решения по анкете
        return True

    def _create_restoration_link(self):
        passport = get_passport()
        response = passport.create_restoration_link(
            uid=self.restore_id.uid,
            link_type=SUPPORT_LINK_TYPE_CHANGE_PASSWORD_SET_METHOD,
            ip=self.client_ip,
            host=self.host,
            admin=self.support_account.login,
        )

        if response['status'] == 'error':
            if 'exception.unhandled' in response['errors']:
                raise PassportPermanentError()
            else:
                raise ProxyError(response['errors'])
        elif 'state' in response:
            raise ProxyError([response['state']])

        self.response_values.update(
            restoration_link=response['secret_link'],
            sensitive_fields=['restoration_link'],
        )
        self.statbox.log(action=u'restoration_link_generated')

    def _write_decision_to_event_log(self):
        if self.form_values['passed']:
            restore_status = RESTORE_STATUS_PASSED
        else:
            restore_status = RESTORE_STATUS_REJECTED
        runner = HistorydbActionRunner(
            {
                'admin': self.support_account.login,
                'comment': '',  # TODO: в перспективе - номер тикета OTRS
            },
            uid=self.restore_id.uid,
            user_ip=self.client_ip,
            user_agent=self.user_agent,
            yandexuid=self.cookies.get('yandexuid'),
            host_id=get_current_host().get_id(),
        )
        runner.execute((EVENT_ACTION, ACTION_RESTORE_SEMI_AUTO_DECISION))
        runner.execute((EVENT_INFO_RESTORE_ID, self.restore_id.to_string()))
        runner.execute((EVENT_INFO_RESTORE_STATUS, restore_status))

    def _write_decision_to_statbox(self, is_decision_changed, is_for_learning):
        self.statbox.log(
            action=u'restore_decision_taken',
            decision=RESTORE_STATUS_PASSED if self.form_values['passed'] else RESTORE_STATUS_REJECTED,
            is_decision_changed=is_decision_changed,
            is_for_learning=is_for_learning,
        )

    def _assert_account_exists(self):
        response = get_blackbox().userinfo(self.restore_id.uid, need_public_name=False)
        try:
            user_account = Account().parse(response)
            self.statbox.bind_context(login=user_account.login)
        except UnknownUid:
            raise AccountNotFoundError()

    def process_request(self):
        self.process_basic_form()
        self._assert_account_exists()

        attempt_info = _get_restore_attempt_info(self.restore_id)
        if not attempt_info:
            raise RestoreIdNotFoundError()

        restore_events = get_restore_events(self.restore_id.uid)
        for event in restore_events:
            if RestoreId.from_string(event['restore_id']) == self.restore_id:
                raw_data = json.loads(event.pop('data_json'))
                extra_info = get_extra_info_from_attempt(raw_data)
                if not extra_info:
                    raise RestoreIdUnknownVersionError()
                break
        else:
            raise RestoreIdNotFoundError()

        is_for_learning = extra_info['is_for_learning']
        is_decision_changed = self._assert_decision_applicable(attempt_info, is_for_learning)

        # Для анкет на обучение, пишем только в статбокс
        # Как побочный эффект, мы не сможем показать это решение в интерфейсе, но пока так
        if not is_for_learning:
            self.response_values['restoration_link'] = None
            if self.form_values['passed']:
                self._create_restoration_link()

            if self.form_values['regenerate_link']:
                return

            self._write_decision_to_event_log()

        self._write_decision_to_statbox(is_decision_changed, is_for_learning)
