# -*- coding: utf-8 -*-
from datetime import datetime
import time

from passport.backend.core.builders.historydb_api import get_historydb_api
from passport.backend.core.compare.compare import (
    FACTOR_BOOL_MATCH,
    simple_string_factor_to_mnemonic,
    STRING_FACTOR_INEXACT_MATCH,
)
from passport.backend.core.compare.dates import (
    BIRTHDAYS_FACTOR_FULL_MATCH,
    BIRTHDAYS_FACTOR_INEXACT_MATCH,
    BIRTHDAYS_FACTOR_NO_MATCH,
    LOOSE_DATE_THRESHOLD_FACTOR,
)
from passport.backend.core.compare.equality.comparator import (
    ALLOWED_DISTANCE_THRESHOLD,
    FACTOR_NOT_SET,
)
from passport.backend.core.historydb.analyzer.event_handlers.phone import flatten_phones_mapping
from passport.backend.core.historydb.analyzer.event_handlers.question_answer import flatten_question_answer_mapping
from passport.backend.core.models.person import MOSCOW_TIMEZONE
from passport.backend.utils.time import (
    datetime_to_string,
    DEFAULT_FORMAT,
    parse_datetime,
)
from pytz import utc


DATETIME_WITH_TZ_FORMAT = '%Y-%m-%d %H:%M:%S %Z%z'
DATE_FORMAT = '%Y-%m-%d'

RESULT_MATCH = 'match'
RESULT_NO_MATCH = 'no_match'
RESULT_INEXACT_MATCH = 'inexact_match'
RESULT_NOT_CALCULATED = 'not_calculated'

FLOAT_FACTOR_ROUND_PLACES = 3


def format_timestamp(unixtime, format=DATETIME_WITH_TZ_FORMAT):
    if unixtime is None:
        return
    utc_datetime = unixtime_to_utc_datetime(unixtime)
    return utc_datetime.astimezone(MOSCOW_TIMEZONE).strftime(format)


def unixtime_to_utc_datetime(unixtime):
    """
    Преобразовать Unix-timestamp в datetime-объект в часовом поясе UTC.
    """
    utc_datetime_naive = datetime.utcfromtimestamp(unixtime)
    return utc.localize(utc_datetime_naive)


def _format_position(position):
    return {
        0: 'First',
        1: 'Second',
        2: 'Third',
        3: 'Fourth',
        4: 'Fifth',
        5: 'Sixth',
        6: 'Seventh',
        7: 'Eighth',
        8: 'Ninth',
        9: 'Tenth',
        -3: 'Third-to-last',
        -2: 'Next-to-last',
        -1: 'Last',
    }.get(position, position)


MASKED_ANSWER = '*****'


class AttemptFormatterBase(object):

    IS_CURRENT_VERSION = False

    DATA_FIELDS = tuple()

    GROUPS = tuple()

    def __init__(self, can_show_answers):
        self.can_show_answers = can_show_answers

    def format(self, data):
        formatted_data = {}

        for field in self.DATA_FIELDS:
            raw_value = data[field]
            formatted_data[field] = self._handle_raw_value(field, raw_value, data)

        extra_info = self._get_extra_info(data)
        ip_info = self._extract_ip_info(data)

        return dict(
            groups=self.GROUPS,
            data=formatted_data,
            extra_info=extra_info,
            ip_info=ip_info,
        )

    def _handle_raw_value(self, field, value, whole_data):
        raise NotImplementedError()  # pragma: no cover

    def _get_extra_info(self, data):
        raise NotImplementedError()  # pragma: no cover

    def _extract_ip_info(self, data):
        raise NotImplementedError()  # pragma: no cover

    def _mask_answer(self, value):
        if self.can_show_answers:
            return value
        return MASKED_ANSWER


class AttemptVersion2Formatter(AttemptFormatterBase):
    GROUPS = (
        {
            u'basic_fields': (
                u'names',
                u'birthday',
            ),
        },
        {
            u'other_fields': (
                u'registration_date',
                u'registration_country',
                u'registration_city',

                u'phone_numbers',
                u'emails',
                u'password',
                u'answer',

                u'email_blacklist',
                u'email_collectors',
                u'email_whitelist',
                u'outbound_emails',
                u'email_folders',

                u'delivery_addresses',
                u'social_accounts',
            ),
        },
    )

    DATA_FIELDS = (
        u'names',
        u'birthday',

        u'registration_date',
        u'registration_country',
        u'registration_city',

        u'phone_numbers',
        u'emails',
        u'password',
        u'answer',

        u'email_blacklist',
        u'email_collectors',
        u'email_whitelist',
        u'outbound_emails',
        u'email_folders',

        u'delivery_addresses',
        u'social_accounts',

        # поля, связанные с IP, обрабатываются отдельно: u'user_ip', u'user_ip_subnet', u'user_env', u'ip_stats'
    )

    def _get_user_ip_auths_percent(self, data):
        stats = data.get('ip_stats', {}).get('statistics', {})
        user_ip_auths_percent = None
        if stats.get('total_auths_count') is not None and stats.get('auths_by_user_ip') is not None:
            user_ip_auths_percent = float(stats['auths_by_user_ip']) * 100 / stats['total_auths_count']
            user_ip_auths_percent = '%.2f' % user_ip_auths_percent
        return user_ip_auths_percent or '-'

    def _extract_ip_info(self, data):
        stats = data.get('ip_stats', {}).get('statistics', {})
        max_auths_stats = stats.get('max_auths_source', {})

        return {
            'user_ip': data['user_ip']['actual'],
            'user_ip_auths_count': stats.get('auths_by_user_ip', '-'),
            'user_ip_auths_percent': self._get_user_ip_auths_percent(data),
            'user_ip_first_auth_datetime': format_timestamp(stats.get('first_user_ip_auth_timestamp')),
            'user_ip_last_auth_datetime': format_timestamp(stats.get('last_user_ip_auth_timestamp')),
            'max_auths_ip': max_auths_stats.get('user_ip', '-'),
            'max_auths_ip_auths_count': max_auths_stats.get('count', '-'),
            'total_auths_count': stats.get('total_auths_count', '-'),
            'different_ips_count': stats.get('different_ips_count', '-'),
            'registration_ip': data['registration_ip'],
            'factors': {
                'user_ip_in_history': self._format_bool_factor(data['user_ip']['factor']),
                'user_subnet_in_history': self._format_bool_factor(data['user_ip_subnet']['factor']),
                'user_ua_in_history': self._format_bool_factor(data['user_env']['factor']),
                'user_ip_eq_registration_ip': RESULT_MATCH if (
                    data['user_ip']['actual'] == data['registration_ip']
                ) else RESULT_NO_MATCH,
            },
            'errors': {
                'auths_successful_envs_api_status': data['auths_successful_envs_api_status'],
                'auths_contain_ip_api_status': data['user_ip']['auths_contain_ip_api_status'],
                'auths_ip_statistics_api_status': data.get('ip_stats', {}).get('auths_ip_statistics_api_status'),
            },
        }

    def _names_factor_to_dict(self, factor):
        return dict(
            firstname=dict(factor['firstname']),
            lastname=dict(factor['lastname']),
        )

    def _format_names_factor(self, factor):
        factor = self._names_factor_to_dict(factor)
        strict_match = any(
            factor[name]['initial_equal'] == 1 or factor[name]['distance'] == 1.0 for name in ['lastname', 'firstname']
        )
        if strict_match:
            return RESULT_MATCH
        inexact_match = any(
            (
                factor[name]['distance'] >= ALLOWED_DISTANCE_THRESHOLD or factor[name]['aggressive_equal'] == 1
            ) for name in ['lastname', 'firstname']
        )
        if inexact_match:
            return RESULT_INEXACT_MATCH
        return RESULT_NO_MATCH

    def _format_string_factor(self, factor):
        factor = dict(factor)
        strict_match = int(factor['initial_equal']) == 1 or float(factor['distance']) == 1.0
        if strict_match:
            return RESULT_MATCH
        inexact_match = float(factor['distance']) >= ALLOWED_DISTANCE_THRESHOLD
        if inexact_match:
            return RESULT_INEXACT_MATCH
        return RESULT_NO_MATCH

    def _format_bool_factor(self, factor):
        if factor == 1:
            return RESULT_MATCH
        if factor == FACTOR_NOT_SET:
            return RESULT_NOT_CALCULATED
        return RESULT_NO_MATCH

    def _format_multiple_selection_factor(self, factor):
        factor = dict(factor) if not isinstance(factor, dict) else factor
        matches_count = factor['matches_count']
        return RESULT_MATCH if matches_count >= 1 else RESULT_NO_MATCH

    def _format_loose_date_factor(self, factor):
        factor = float(factor)
        if factor == 1.0:
            return RESULT_MATCH
        if factor >= LOOSE_DATE_THRESHOLD_FACTOR:
            return RESULT_INEXACT_MATCH
        if factor == FACTOR_NOT_SET:
            return RESULT_NOT_CALCULATED
        return RESULT_NO_MATCH

    def _format_password_factor(self, factor):
        factor = dict(factor)
        date_factor = self._format_loose_date_factor(factor['auth_date'])
        found_factor = self._format_bool_factor(factor['auth_found'])
        return dict(auth_date=date_factor, auth_found=found_factor)

    def _format_delivery_addresses(self, addresses):
        return [
            dict([(item, address.get(item, '')) for item in (
                'country',
                'city',
                'street',
                'building',
                'suite',
            )]) for address in addresses
        ]

    def _format_birthday_factor(self, factor):
        return self._format_bool_factor(factor)

    def _summarize_factors(self, factors):
        if not factors:
            return ''
        if isinstance(factors, str):
            return factors
        if any(factor == RESULT_MATCH for factor in factors.values()):
            return RESULT_MATCH
        if any(factor == RESULT_INEXACT_MATCH for factor in factors.values()):
            return RESULT_INEXACT_MATCH
        return RESULT_NO_MATCH

    def _handle_raw_value_common(self, result, field, value, whole_data):
        matched = False
        if field == 'registration_date':
            matched = True
            result['incoming'] = [value['entered']]
            registration_datetime = MOSCOW_TIMEZONE.localize(parse_datetime(value['account']))
            result['trusted'] = [datetime_to_string(registration_datetime, DATETIME_WITH_TZ_FORMAT)]
            result['factors'] = self._format_loose_date_factor(value['factor'])
        elif field in ('registration_country', 'registration_city'):
            matched = True
            result['incoming'] = [u'%s (%s)' % (value['entered'] or '-', value['entered_id'] or '-')]
            result['trusted'] = [u'%s (%s)' % (value['history'] or '-', value['history_id'] or '-')]
            result['factors'] = self._summarize_factors({
                'id': self._format_bool_factor(value['factor_id']),
                'text': self._format_string_factor(value['factor']),
            })
        elif field in (
            'phone_numbers',
            'emails',
            'email_blacklist',
            'email_collectors',
            'email_whitelist',
            'outbound_emails',
            'email_folders',
            'services',
        ):
            matched = True
            result['incoming'] = value['entered'] or []
            result['trusted'] = (
                value['history'] if 'history' in value else value['actual'] if 'actual' in value else value['account']
            )
            result['factors'] = self._format_multiple_selection_factor(value['factor'])
            if 'api_status' in value:
                result['errors'] = {'api_status': value['api_status']}
            if field in ('email_blacklist', 'email_collectors', 'email_whitelist', 'outbound_emails', 'email_folders'):
                if 'oauth_api_status' in whole_data:
                    result['errors']['oauth_api_status'] = whole_data['oauth_api_status']
        elif field == 'delivery_addresses':
            matched = True
            result['incoming'] = value['entered'] or []
            result['trusted'] = self._format_delivery_addresses(value['account'])
            result['factors'] = self._format_multiple_selection_factor(value['factor'])
        return matched

    def _handle_raw_value(self, field, value, whole_data):
        result = {
            'incoming': [],
            'trusted': {},
            'factors': {},
            'errors': {},
        }
        if field == 'names':
            result['incoming'] = [' '.join(value['entered'])]
            result['trusted'] = {
                'account': [' '.join(value['account'] or [])],
                'history': [' '.join(value['history'] or [])],
            }
            result['factors'] = {
                'history': self._format_names_factor(value['history_factor']),
                'account': self._format_names_factor(value['account_factor']),
            }
            result['errors'] = {'historydb_api_events_status': whole_data['historydb_api_events_status']}
        elif field == 'birthday':
            result['incoming'] = [value['entered']]
            result['trusted'] = {
                'account': [value['account']],
                'history': [value['history']],
            }
            result['factors'] = {
                'history': self._format_birthday_factor(value['history_factor']),
                'account': self._format_birthday_factor(value['account_factor']),
            }
            result['errors'] = {'historydb_api_events_status': whole_data['historydb_api_events_status']}
        elif field == 'password':
            result['incoming'] = [value['auth_date_entered']] if value['auth_date_entered'] is not None else None
            result['trusted'] = [
                {
                    'used_since': format_timestamp(value['used_since']),
                    'used_until': format_timestamp(value['used_until']),
                },
            ] if value['used_since'] is not None else None
            result['factors'] = self._format_password_factor(value['factor'])
            result['errors'] = {'api_status': value['api_status']}
        elif field == 'answer':
            if value['entered']:
                question_id, question_text = value['question'].split(':')
                if int(question_id) < 0:
                    # В случае ввода своего вопроса, имеем отрицательный ID - для удобства заменим на 99
                    value['question'] = '99:%s' % question_text
            result['incoming'] = [
                {'question': value['question'], 'answer': self._mask_answer(value['entered'])}
            ] if value['entered'] else None
            # анкета этой версии не поддерживается, просто убеждаемся, что чувствительные данные не попадут в вывод
            result['trusted'] = [self._mask_answer(item) for item in value['history'] or []]
            result['factors'] = self._format_string_factor(value['factor'])
        elif field == 'social_accounts':
            result['incoming'] = value['entered'] or []
            result['trusted'] = [
                {
                    'addresses': profile['addresses'],
                    'firstname': profile['firstname'],
                    'lastname': profile['lastname'],
                    'birthday': profile['birthday'],
                } for profile in value['profiles']
            ]
            result['errors'] = {'api_status': value['api_status']}
        else:
            self._handle_raw_value_common(result, field, value, whole_data)
        result['factors_summary'] = self._summarize_factors(result['factors'])
        return result

    def _get_extra_info(self, data):
        extra_info = {
            'is_for_learning': False,
            'is_current_version': self.IS_CURRENT_VERSION,
        }
        for field in (
                u'request_source',
                u'restore_status',
                u'version',
        ):
            extra_info[field] = data[field]
        return extra_info


class AttemptVersionMultiStep3Formatter(AttemptVersion2Formatter):
    GROUPS = (
        {
            u'basic_fields': (
                u'names',
                u'birthday',
            ),
        },
        {
            u'other_fields': (
                u'passwords',
                u'phone_numbers',
                u'emails',
                u'answer',
                u'registration_date',
                u'registration_country',
                u'registration_city',
                u'social_accounts',
                u'services',
                u'delivery_addresses',
                u'email_folders',
                u'email_blacklist',
                u'email_whitelist',
                u'email_collectors',
                u'outbound_emails',
            ),
        },
    )

    DATA_FIELDS = (
        u'names',
        u'birthday',
        u'passwords',
        u'phone_numbers',
        u'emails',
        u'answer',
        u'registration_date',
        u'registration_country',
        u'registration_city',
        u'social_accounts',
        u'services',
        u'delivery_addresses',
        u'email_folders',
        u'email_blacklist',
        u'email_whitelist',
        u'email_collectors',
        u'outbound_emails',

        # поля, связанные с IP, обрабатываются отдельно: u'user_ip', u'user_ip_subnet', u'user_env', u'ip_stats'
    )

    def _format_password_factor(self, factor):
        result = {}
        for index, auth_found in enumerate(factor['auth_found']):
            auth_date = factor['auth_date'][index]
            result.update({
                'auth_found_%d' % index: self._format_bool_factor(auth_found),
                'auth_date_%d' % index: self._format_loose_date_factor(auth_date),
            })
        return result

    def _transform_entered_names(self, value):
        """
        Пользователь вводит отдельно варианты имени и фамилии, при этом обязательно задание хотя бы одного варианта
        имени и фамилии. Данная функция группирует варианты имени и фамилии в пары, с учетом того, что пользователь
        может ввести не более двух вариантов имени и фамилии.
        """
        firstnames = list(value['firstnames'])
        lastnames = list(value['lastnames'])
        entered_names = []
        while firstnames or lastnames:
            # в списках всегда есть хотя бы одно имя и одна фамилия
            if firstnames:
                firstname = firstnames.pop(0)
            if lastnames:
                lastname = lastnames.pop(0)
            entered_names.append(dict(firstname=firstname, lastname=lastname))
        return entered_names

    def _format_names(self, names):
        return [
            '%s %s' % (names_info['firstname'], names_info['lastname']) for names_info in names
        ]

    def _format_intervals(self, intervals):
        formatted_intervals = []

        for interval in intervals:
            formatted_interval = {}
            formatted_intervals.append(formatted_interval)
            for point in ('start', 'end'):
                if interval[point] is None:
                    formatted_interval[point] = None
                    continue
                formatted_interval[point] = {
                    'user_ip': interval[point]['user_ip'],
                    'datetime': format_timestamp(interval[point]['timestamp']),
                }
        return formatted_intervals

    def _format_incoming_social_profiles(self, entered_social_accounts):
        result = []
        for account in entered_social_accounts:
            # Сырые данные, полученные по task_id
            result.append({
                'addresses': account['links'],
                'firstname': account.get('firstname'),
                'lastname': account.get('lastname'),
                'birthday': account.get('birthday'),
            })
        return result

    def _format_social_profile(self, profile, **kwargs):
        result = {
            'addresses': profile['addresses'],
            'firstname': profile.get('person', {}).get('firstname'),
            'lastname': profile.get('person', {}).get('lastname'),
            'birthday': profile.get('person', {}).get('birthday'),
        }
        result.update(kwargs)
        return result

    def _format_trusted_social_profiles(self, entered_profiles, account_profiles):
        """
        Схлопываем данные соц. профилей по соц. аккаунту для того, чтобы избежать дублирования в выводе.
        Дополняем данные информацией о совпадении с пользовательским вводом, принадлежности аккаунту,
        списком UID, к которым также привязан соц. аккаунт.
        """
        def _make_social_account_key(profile):
            user_id = profile['userid']
            provider = profile['provider']
            addresses_key = ''.join(sorted(profile['addresses']))
            return (user_id, provider, addresses_key)
        result = []
        social_accounts_info = {}
        account_profile_ids = {profile['profile_id'] for profile in account_profiles}
        for profile in entered_profiles:
            # Построим отображение введенных соц. аккаунтов в аккаунты на Яндексе
            info = social_accounts_info.setdefault(_make_social_account_key(profile), {})
            info.setdefault('uids', set()).add(profile['uid'])
            info.setdefault('profile_data', profile)
            info['is_matched'] = info.get('is_matched') or profile['profile_id'] in account_profile_ids

        for profile in account_profiles:
            info = social_accounts_info.pop(_make_social_account_key(profile), {})
            formatted_profile = self._format_social_profile(
                profile,
                is_matched=info.get('is_matched', False),
                belongs_to_account=True,
            )
            uids = info.get('uids')
            if uids and len(uids) > 1:
                uids.remove(profile['uid'])
                formatted_profile['other_uids'] = ', '.join([str(uid) for uid in uids])
            result.append(formatted_profile)

        result.sort(key=lambda formatted_profile: not formatted_profile['is_matched'])

        for info in social_accounts_info.values():
            result.append(
                self._format_social_profile(
                    info['profile_data'],
                    belongs_to_account=False,
                    other_uids=', '.join([str(uid) for uid in info['uids']]),
                ),
            )
        return result

    def _format_phone_numbers_changes_info(self, value):
        return {}

    def _format_answer_changes_info(self, value):
        return {}

    def _handle_raw_value(self, field, value, whole_data):
        """
        Привести данные многошаговой анкеты в формат, поддерживаемый фронтендом.
        Включает в себя:
        - incoming - список данных, пришедших от пользователя;
        - trusted - список полученных нами данных о пользователе;
        - factors - словарь с факторами сравнения;
        - errors - словарь с признаками ошибок.
        Опционально включает:
        - trusted_intervals - список списков интервалов актуальности данных о пользователе. Элементы соответствуют
        элементам списка trusted;
        - indices - список пар индексов для найденных лучших совпадений. Каждая пара содержит индекс среди введенных
        данных i, индекс среди данных сервиса j. Это означает, что найдено совпадение данных incoming[i] и trusted[j].
        """
        result = {
            'incoming': [],
            'trusted': [],
            'factors': {},
            'errors': {},
        }
        if field == 'names':
            # введенные данные представляют собой независимые списки вариантов имен и фамилий, их нужно
            # предварительно сгруппировать
            names_entered = self._transform_entered_names(value['entered'])
            result['incoming'] = self._format_names(names_entered)
            result['trusted'] = self._format_names(value['account'])
            result['trusted_intervals'] = [self._format_intervals([info['interval']]) for info in value['account']]
            result['indices'] = value['indices']
            result['errors'] = {'historydb_api_events_status': whole_data['historydb_api_events_status']}
            result['factors'] = dict()
            for name in ('current', 'registration', 'intermediate'):
                result['factors'][name] = simple_string_factor_to_mnemonic(value['factor'][name])
        elif field == 'birthday':
            result['incoming'] = [value['entered']]
            result['trusted'] = [info['value'] for info in value['account']]
            result['trusted_intervals'] = [self._format_intervals([info['interval']]) for info in value['account']]
            result['indices'] = value['indices']
            result['errors'] = {'historydb_api_events_status': whole_data['historydb_api_events_status']}
            result['factors'] = dict()
            for name in ('current', 'registration', 'intermediate'):
                result['factors'][name] = self._format_birthday_factor(value['factor'][name])
        elif field == 'passwords':
            result['incoming'] = [{
                'auth_date': value['auth_date_entered'],
                'passwords_count': value['factor']['entered_count'],
            }]
            result['trusted'] = []
            result['trusted_intervals'] = [self._format_intervals(intervals) for intervals in value['intervals']]
            result['factors'] = self._format_password_factor(value['factor'])
            result['errors'] = {'api_statuses': all(value['api_statuses'])}
        elif field in (
            'phone_numbers',
            'emails',
        ):
            result['incoming'] = value['entered'] or []
            result['trusted'] = [info['value'] for info in value['history']]
            result['trusted_intervals'] = [self._format_intervals(info['intervals']) for info in value['history']]
            result['factors'] = self._format_multiple_selection_factor(value['factor'])
            result['indices'] = value['match_indices']
            if field == 'phone_numbers':
                result['changes_info'] = self._format_phone_numbers_changes_info(value)
        elif field == 'answer':
            question = value['entered']['question']
            result['incoming'] = [
                {
                    'question': question,
                    'answer': self._mask_answer(value['entered']['answer']),
                },
            ] if value['entered']['answer'] else None
            result['trusted'] = None
            if value['factor']['best'] != FACTOR_NOT_SET:
                index = value['indices']['best']
                question_index, answer_index = index
                question_info = value['history'][question_index]
                result['trusted'] = [self._mask_answer(info['value']) for info in question_info['answers']]
                result['trusted_intervals'] = [
                    self._format_intervals(info['intervals']) for info in question_info['answers']
                ]
                result['indices'] = {'best': answer_index}

            result['changes_info'] = self._format_answer_changes_info(value)
            result['factors'] = {
                'best': simple_string_factor_to_mnemonic(value['factor']['best']),
                'current': simple_string_factor_to_mnemonic(value['factor']['current']),
            }
        elif field == 'delivery_addresses':
            result['incoming'] = value['entered'] or []
            result['trusted'] = self._format_delivery_addresses(value['account'])
            result['factors'] = self._format_multiple_selection_factor(value['factor'])
        elif field == 'social_accounts':
            result['incoming'] = self._format_incoming_social_profiles(value.get('entered_accounts', []))
            result['trusted'] = self._format_trusted_social_profiles(value['entered_profiles'], value['account_profiles'])
            result['factors'] = self._format_multiple_selection_factor(value['factor'])
            result['errors'] = {'api_status': value['api_status']}
        else:
            self._handle_raw_value_common(result, field, value, whole_data)
        result['factors_summary'] = self._summarize_factors(result['factors'])
        return result

    def _get_extra_info(self, data):
        request_info = data['request_info']
        extra_info = {
            'request_source': request_info['request_source'],
            'is_current_version': self.IS_CURRENT_VERSION,
        }
        if 'contact_email' in request_info:
            extra_info['contact_email'] = request_info['contact_email']
        for field in (
            u'is_for_learning',
            u'restore_status',
            u'version',
        ):
            extra_info[field] = data[field]
        return extra_info


# Разбивка факторов смены на упорядоченные блоки для фронтенда
CHANGE_FACTOR_BLOCKS_CONFIG = [
    (
        'change_count',
        'change_depth',
        'last_change_depth',
    ),
    (
        'ip_match',
        'subnet_match',
        'ua_match',
    ),
    [
        template % prefix
        for template in
        (
            '%sip_eq_user',
            '%ssubnet_eq_user',
            '%sua_eq_user',
        )
        for prefix in ('', 'change_', 'last_change_', 'match_')
    ],
    [
        template % prefix
        for template in
        (
            '%sip_eq_reg',
            '%ssubnet_eq_reg',
            '%sua_eq_reg',
        )
        for prefix in ('', 'change_', 'match_')
    ],
    [
        template % prefix
        for template in
        (
            '%sip_first_auth_depth',
            '%ssubnet_first_auth_depth',
            '%sua_first_auth_depth',
        )
        for prefix in ('', 'change_', 'last_change_', 'match_')
    ],
    (
        'last_change_is_forced_change',
        'forced_change_pending',
    ),
]


# Разбивка данных по IP/subnet/UA на упорядоченные блоки для фронтенда
IP_INFO_BLOCKS_CONFIG = [
    (
        'registration_date',
        'auths_limit_not_reached',
    ),
] + [
    (
        '%s_auth_interval' % field,
        '%s_eq_reg' % field,
        '%s_first_auth_depth' % field,
    ) for field in ('ip', 'subnet', 'ua')
]


def _format_factors_to_blocks(factors, blocks_config):
    """
    Отформатировать словарь факторов в упорядоченные блоки факторов в соответствии
    с конфигурацией blocks_config.
    """
    blocks = []
    for block_fields in blocks_config:
        inner_block = []
        for factor_name in block_fields:
            if factor_name in factors:
                factor_value = factors.pop(factor_name)
                inner_block.append(dict(name=factor_name, value=factor_value))
        if inner_block:
            blocks.append(inner_block)
    if factors:
        blocks.append([dict(name=factor_name, value=value) for factor_name, value in factors.items()])
    return blocks


class AttemptVersionMultiStep4Formatter(AttemptVersionMultiStep3Formatter):

    def format(self, data):
        result = super(AttemptVersionMultiStep4Formatter, self).format(data)
        result['events_info'] = self._extract_events_info(data)
        return result

    def _format_personal_changes_info(self, value):
        result = {
            'change_envs': self._extract_changes_origin_infos(value['account'], self.PERSONAL_DATA_CHANGE_INDICES),
            'factor': {},
        }
        for factor_name in ('change_count', 'change_depth', 'change_ip_eq_user', 'change_subnet_eq_user',
                            'change_ua_eq_user', 'change_ip_eq_reg', 'change_subnet_eq_reg',
                            'change_ua_eq_reg', 'intermediate_depth'):
            if factor_name in value['factor']:
                result['factor'][factor_name] = value['factor'][factor_name]
        result['factor'] = self._format_factor_dict(result['factor'])
        result['change_indices'] = self.PERSONAL_DATA_CHANGE_INDICES
        return result

    def _handle_raw_value(self, field, value, whole_data):
        result = {}
        if field in ('registration_country', 'registration_city'):
            result['incoming'] = [u'%s (%s)' % (value['entered'] or '-', value['entered_id'] or '-')]
            result['trusted'] = [u'%s (%s)' % (value['history'] or '-', value['history_id'] or '-')]
            result['factors'] = self._summarize_factors({
                'id': self._format_bool_factor(value['factor']['id']),
                'text': simple_string_factor_to_mnemonic(value['factor']['text']),
            })
        elif field == 'names':
            # введенные данные представляют собой независимые списки вариантов имен и фамилий, их нужно
            # предварительно сгруппировать
            names_entered = self._transform_entered_names(value['entered'])
            result['incoming'] = self._format_names(names_entered)
            result['trusted'] = self._format_names(value['account'])
            result['trusted_intervals'] = [self._format_intervals([info['interval']]) for info in value['account']]
            result['changes_info'] = self._format_personal_changes_info(value)
            result['indices'] = value['indices']
            result['errors'] = {'historydb_api_events_status': whole_data['historydb_api_events_status']}
            result['factors'] = dict()
            for name in ('current', 'registration', 'intermediate'):
                result['factors'][name] = simple_string_factor_to_mnemonic(value['factor'][name])
        elif field == 'birthday':
            result['incoming'] = [value['entered']]
            result['trusted'] = [info['value'] for info in value['account']]
            result['trusted_intervals'] = [self._format_intervals([info['interval']]) for info in value['account']]
            result['changes_info'] = self._format_personal_changes_info(value)
            result['indices'] = value['indices']
            result['errors'] = {'historydb_api_events_status': whole_data['historydb_api_events_status']}
            result['factors'] = dict()
            for name in ('current', 'registration', 'intermediate'):
                result['factors'][name] = self._format_birthday_factor(value['factor'][name])
        elif field == 'passwords':
            result['incoming'] = [{
                'auth_date': value['auth_date_entered'],
                'passwords_count': value['factor']['entered_count'],
            }]
            result['trusted'] = []
            result['trusted_intervals'] = [self._format_intervals(intervals) for intervals in value['intervals']]
            result['factors'] = self._format_password_factor(value['factor'])
            result['errors'] = {'api_statuses': all(value['api_statuses'])}

        if result:
            result['factors_summary'] = self._summarize_factors(result['factors'])
            return result

        return super(AttemptVersionMultiStep4Formatter, self)._handle_raw_value(field, value, whole_data)

    def _format_password_factor(self, factor):
        result = {}
        for index, auth_found in enumerate(factor['auth_found']):
            auth_date = factor['auth_date'][index]
            auth_depth = factor['first_auth_depth'][index]
            result.update({
                'auth_found_%d' % index: self._format_bool_factor(auth_found),
                'auth_date_%d' % index: self._format_loose_date_factor(auth_date),
                'first_auth_depth_%d' % index: auth_depth,
            })
        return result

    def _format_intervals(self, intervals):
        formatted_intervals = []
        for interval in intervals:
            formatted_interval = {}
            formatted_intervals.append(formatted_interval)
            for point in ('start', 'end'):
                if interval[point] is None:
                    formatted_interval[point] = None
                    continue
                formatted_interval[point] = dict(interval[point])
                formatted_interval[point]['datetime'] = format_timestamp(interval[point]['timestamp'])
        return formatted_intervals

    def _format_ua(self, ua):
        if not ua:
            return
        return ', '.join(
            ['%s: %s' % (
                field,
                ua[field],
            ) for field in ('os.name', 'browser.name', 'yandexuid') if ua[field] is not None],
        )

    def _format_auth_info(self, item):
        if not item:
            return
        return format_timestamp(item['timestamp'], format=DATE_FORMAT)

    def _extract_ip_info(self, data):
        user_env_info = data['user_env_auths']

        factors = user_env_info['factor']
        auths_limit_reached = factors['auths_limit_reached']
        auths_limit_not_reached = FACTOR_BOOL_MATCH - auths_limit_reached if auths_limit_reached != FACTOR_NOT_SET else FACTOR_NOT_SET
        factors['auths_limit_not_reached'] = auths_limit_not_reached
        del factors['auths_limit_reached']
        factors['registration_date'] = data['registration_date']['account']

        result = {
            'gathered_auths_count': user_env_info['actual']['gathered_auths_count'],
            'factors': self._format_factor_dict(factors, blocks_config=IP_INFO_BLOCKS_CONFIG),
            'errors': {
                'auths_aggregated_runtime_api_status': data['auths_aggregated_runtime_api_status'],
            },
        }

        for field in ('ip', 'subnet', 'ua'):
            value = user_env_info['actual'][field]
            if field == 'ua':
                value = self._format_ua(value)
            result[field] = value
            first_auth_field, last_auth_field = '%s_first_auth' % field, '%s_last_auth' % field
            result[first_auth_field] = self._format_auth_info(user_env_info['actual'][first_auth_field])
            result[last_auth_field] = self._format_auth_info(user_env_info['actual'][last_auth_field])

            registration_field = '%s_registration' % field
            result[registration_field] = user_env_info['registration'][field]
            if field == 'ua':
                result[registration_field] = self._format_ua(result[registration_field])

        return result

    PERSONAL_DATA_CHANGE_INDICES = (0, -2, -1)  # Для личных данных смотрим на первую и две последних смены
    RESTORE_METHODS_CHANGE_INDICES = (-3, -2, -1)  # Для средств восстановления смотрим на последние три смены

    ONE_DAY_CHANGES_PAIRS = (
        'password_and_personal_change_one_day',
        'password_and_recovery_change_one_day',
        'personal_and_recovery_change_one_day',
    )

    def _format_origin_info(self, origin_info):
        result = {
            'user_ip': origin_info.get('user_ip'),
            'datetime': format_timestamp(origin_info['timestamp']),
        }
        if 'user_agent' in origin_info:
            result['user_agent'] = origin_info['user_agent']
        if 'yandexuid' in origin_info:
            yandexuid = 'yandexuid/%s' % origin_info['yandexuid']
            result['user_agent'] = '%s %s' % (result['user_agent'], yandexuid) if 'user_agent' in result else yandexuid
        if origin_info.get('position') is not None:
            result['position'] = _format_position(origin_info['position'])
        return result

    def _extract_changes_origin_infos(self, items, change_indices):
        origin_infos = [item['interval']['start'] for item in items]
        changes_count = len(origin_infos) - 1 if origin_infos else 0
        result = [None] * len(change_indices)
        for i, index in enumerate(change_indices):
            if index >= 0 and index < changes_count or index < 0 and abs(index) <= changes_count:
                result[i] = self._format_origin_info(origin_infos[index + 1 if index >= 0 else index])
        return result

    def _format_factor_dict(self, factor, blocks_config=CHANGE_FACTOR_BLOCKS_CONFIG):
        result = {}
        for name, value in factor.items():
            result[name] = value
            if isinstance(value, list):
                result[name] = ', '.join(
                    [str(round(item, FLOAT_FACTOR_ROUND_PLACES)) if isinstance(item, float) else str(item) for item in value],
                )
            elif isinstance(value, float):
                result[name] = round(value, FLOAT_FACTOR_ROUND_PLACES)
        return _format_factors_to_blocks(result, blocks_config)

    def _format_aggregated(self, data):
        aggregated_result = {}
        aggregated_data = data['aggregated']
        for pair in self.ONE_DAY_CHANGES_PAIRS:
            factor = aggregated_data['factor'][pair]
            if any([factor[field] for field in ('ip_match', 'subnet_match', 'ua_match')]):
                aggregated_result[pair] = {
                    'factor': self._format_factor_dict(factor),
                }
                for field in ('ip', 'subnet', 'ua'):
                    first_auth_field = '%s_first_auth' % field
                    values = [self._format_auth_info(info) for info in aggregated_data['actual'][pair][first_auth_field]]
                    aggregated_result[pair][first_auth_field] = values
        return aggregated_result

    def _extract_password_events_info(self, data):
        passwords_info = {'factor': {}}
        for item_name in ('last_change_ip_first_auth', 'last_change_subnet_first_auth', 'last_change_ua_first_auth'):
            passwords_info[item_name] = self._format_auth_info(data['passwords']['actual'][item_name])
        last_change_info = data['passwords']['actual']['last_change']
        passwords_info['change_envs'] = [
            self._format_origin_info(last_change_info['origin_info']) if last_change_info else None,
        ]
        for factor_name, value in data['passwords']['factor'].items():
            if 'last_change' in factor_name or factor_name in ('forced_change_pending', 'change_count'):
                passwords_info['factor'][factor_name] = value
        passwords_info['factor'] = self._format_factor_dict(passwords_info['factor'])
        return passwords_info

    def _extract_events_info(self, data):
        """
        Вытаскиваем необходимые факторы в блок events_info
        """
        result = {'passwords': self._extract_password_events_info(data)}

        recovery_changes = {}
        for recovery_item_name in ('phone_numbers', 'answer'):
            recovery_info = {'factor': {}}
            recovery_data = data[recovery_item_name]
            for factor_name in ('change_count', 'change_depth', 'change_ip_eq_user', 'change_subnet_eq_user',
                                'change_ua_eq_user'):
                recovery_info['factor'][factor_name] = recovery_data['factor'][factor_name]
            recovery_info['factor'] = self._format_factor_dict(recovery_info['factor'])
            flatten_func = flatten_phones_mapping if recovery_item_name == 'phone_numbers' else flatten_question_answer_mapping
            flat_items = flatten_func(recovery_data['history'])
            recovery_info['change_envs'] = self._extract_changes_origin_infos(
                flat_items,
                self.RESTORE_METHODS_CHANGE_INDICES,
            )
            recovery_info['change_indices'] = self.RESTORE_METHODS_CHANGE_INDICES
            recovery_changes[recovery_item_name] = recovery_info
        result['recovery_changes'] = recovery_changes
        result['aggregated'] = self._format_aggregated(data)
        return result


class AttemptVersionMultiStep41Formatter(AttemptVersionMultiStep4Formatter):

    def _format_aggregated(self, data):
        aggregated_result = {}
        aggregated_data = data['aggregated']
        for pair in self.ONE_DAY_CHANGES_PAIRS:
            factor = aggregated_data['factor'][pair]
            if any([factor[field] for field in ('ip_match', 'subnet_match', 'ua_match')]):
                aggregated_result[pair] = {
                    'factor': self._format_factor_dict(factor),
                }
                matches = aggregated_data['matches'][pair]
                for match in matches:
                    for env in match['envs']:
                        for field in ('ip', 'subnet', 'ua'):
                            auth_info_field = '%s_first_auth_info' % field
                            env[auth_info_field] = self._format_auth_info(env[auth_info_field])
                        env['ua'] = self._format_ua(env['ua'])
                        env['datetime'] = format_timestamp(env.pop('timestamp'), format=DEFAULT_FORMAT)
                        env['entity'] = env['entity'].rstrip('s')
                aggregated_result[pair]['matches'] = matches
        return aggregated_result


def _extract_values_by_indices(values, indices):
    """
    Из списка values вытаскивает доступные значения по индексам в списке indices,
    каждый элемент списка вытаскивается не более одного раза (индексы могут быть отрицательными).
    Для индексов, которым не соответствует значение, в результат записывается None
    """
    accessed_values = [False] * len(values)
    extracted_values = []
    for index in indices:
        try:
            if accessed_values[index]:
                extracted_values.append(None)
                continue
            extracted_values.append(values[index])
            accessed_values[index] = True
        except IndexError:
            extracted_values.append(None)
            pass
    return extracted_values


def _get_intervals_for_matches_from_answer_data(answer_data):
    """
    Получить информацию об интервалах актуальности КВ/КО в истории для найденного совпадения
    с КО, который ввел пользователь.
    """
    best_factor = answer_data['factor']['best']
    intervals = []
    if best_factor >= STRING_FACTOR_INEXACT_MATCH:
        question_index, answer_index = answer_data['indices']['best']
        question_info = answer_data['history'][question_index]
        answer_info = question_info['answers'][answer_index]
        intervals = [dict(interval=interval, position=answer_index) for interval in answer_info['intervals']]
    return intervals


def _flatten_phones_mapping_with_indices(phones, indices):
    """
    Построить список телефонов, отсортированный по времени подтверждения
    Присоединить информацию об индексах совпадений
    """
    return sorted(
        [
            dict(
                value=phone_info['value'],
                interval=interval,
                position=indices[index],
            )
            for (index, phone_info) in enumerate(phones)
            for interval in phone_info['intervals']
        ],
        key=lambda item: item['interval']['start']['timestamp'],
    )


def _get_intervals_for_matches_from_phone_numbers_data(phone_numbers_data):
    """
    Получить информацию об интервалах актуальности телефонов в истории для найденных совпадений
    с телефонами, которые ввел пользователь.
    """
    history_phones_info = phone_numbers_data['history']
    match_indices = phone_numbers_data['match_indices']
    history_match_indices = [history_index for (_, history_index) in match_indices]
    matched_phones_info = [history_phones_info[index] for index in history_match_indices]
    # работаем с телефонами в плоском виде. совпадение значений может быть одно, но интервалов
    # актуальности - несколько, в этом случае посмотрим на два различных окружения
    flat_phones = _flatten_phones_mapping_with_indices(matched_phones_info, history_match_indices)
    return flat_phones


class AttemptVersionMultiStep42Formatter(AttemptVersionMultiStep41Formatter):
    def _format_birthday_factor(self, factor):
        """
        Начиная с версии 4.2 сравниваем ДР неточно
        """
        return {
            BIRTHDAYS_FACTOR_FULL_MATCH: RESULT_MATCH,
            BIRTHDAYS_FACTOR_INEXACT_MATCH: RESULT_INEXACT_MATCH,
            BIRTHDAYS_FACTOR_NO_MATCH: RESULT_NO_MATCH,
        }.get(factor, RESULT_NOT_CALCULATED)

    def _handle_raw_value(self, field, value, whole_data):
        result = {}
        if field == 'names':
            # введенные данные представляют собой независимые списки вариантов имен и фамилий, их нужно
            # предварительно сгруппировать
            # начиная с данной версии сравниваем также имя, факторы раздвоились
            names_entered = self._transform_entered_names(value['entered'])
            result['incoming'] = self._format_names(names_entered)
            result['trusted'] = self._format_names(value['account'])
            result['trusted_intervals'] = [self._format_intervals([info['interval']]) for info in value['account']]
            result['factors'] = {}
            for factor_name in ('current', 'registration', 'intermediate'):
                for factor_value, name in zip(value['factor'][factor_name], ('firstname', 'lastname')):
                    result['factors']['%s_%s' % (factor_name, name)] = simple_string_factor_to_mnemonic(factor_value)
            result['changes_info'] = self._format_personal_changes_info(value)
            result['indices'] = value['indices']
            result['errors'] = {'historydb_api_events_status': whole_data['historydb_api_events_status']}

        if result:
            result['factors_summary'] = self._summarize_factors(result['factors'])
            return result

        return super(AttemptVersionMultiStep42Formatter, self)._handle_raw_value(field, value, whole_data)

    def _extract_changes_origin_infos(self, items, change_indices):
        origin_infos = [dict(item['interval']['start'], position=item.get('position')) for item in items]
        origin_infos = _extract_values_by_indices(origin_infos, change_indices)
        result = []
        for index, origin_info in enumerate(origin_infos):
            if origin_info is not None:
                if origin_info.get('position') is None:
                    origin_info['position'] = change_indices[index]
                formatted_info = self._format_origin_info(origin_info)
                result.append(formatted_info)
        return result

    def _extract_password_events_info(self, data):
        passwords_info = {'factor': {}}
        last_change_info = data['passwords']['actual']['last_change']
        formatted_change_env = None
        if last_change_info:
            formatted_change_env = self._format_origin_info(dict(last_change_info['origin_info'], position=-1))
            for item_name in ('ip_first_auth', 'subnet_first_auth', 'ua_first_auth'):
                auth_info = data['passwords']['actual']['change_%s' % item_name][0]
                formatted_change_env['%s_date' % item_name] = self._format_auth_info(auth_info)
        passwords_info['change_envs'] = [formatted_change_env]
        for factor_name, factor_value in data['passwords']['factor'].items():
            if factor_name.startswith('change_') or factor_name in ('forced_change_pending', 'last_change_is_forced_change'):
                passwords_info['factor'][factor_name] = factor_value
        passwords_info['factor'] = self._format_factor_dict(passwords_info['factor'])
        return passwords_info

    def _format_recovery_match_changes_info(self, value, entity):
        result = {'factor': {}}
        for factor_name in value['factor']:
            if factor_name.startswith('match_'):
                result['factor'][factor_name] = value['factor'][factor_name]
        result['factor'] = self._format_factor_dict(result['factor'])
        if entity == 'answer':
            flat_items = _get_intervals_for_matches_from_answer_data(value)
        else:
            flat_items = _get_intervals_for_matches_from_phone_numbers_data(value)
        result['change_envs'] = self._extract_changes_origin_infos(flat_items, [0, -1])

        return result

    def _format_phone_numbers_changes_info(self, value):
        return self._format_recovery_match_changes_info(value, 'phone_numbers')

    def _format_answer_changes_info(self, value):
        return self._format_recovery_match_changes_info(value, 'answer')

    def _format_personal_changes_info(self, value):
        result = {
            'change_envs': self._extract_changes_origin_infos(value['account'][1:], self.PERSONAL_DATA_CHANGE_INDICES),
            'factor': {},
        }
        for factor_name in ('change_count', 'change_depth', 'change_ip_eq_user', 'change_subnet_eq_user',
                            'change_ua_eq_user', 'change_ip_eq_reg', 'change_subnet_eq_reg',
                            'change_ua_eq_reg', 'intermediate_depth'):
            result['factor'][factor_name] = value['factor'][factor_name]
        result['factor'] = self._format_factor_dict(result['factor'])
        result['change_indices'] = self.PERSONAL_DATA_CHANGE_INDICES
        return result


class AttemptVersionMultiStep43Formatter(AttemptVersionMultiStep42Formatter):
    IS_CURRENT_VERSION = True

    def _format_password_factor(self, factor):
        result = {}
        for index, auth_found in enumerate(factor['auth_found']):
            auth_date = factor['auth_date'][index]
            auth_depth = factor['first_auth_depth'][index]
            equals_current = factor['equals_current'][index]
            result.update({
                'auth_found_%d' % index: self._format_bool_factor(auth_found),
                'auth_date_%d' % index: self._format_loose_date_factor(auth_date),
                'first_auth_depth_%d' % index: auth_depth,
                'equals_current_%d' % index: self._format_bool_factor(equals_current),
            })
        return result


VERSION_TO_FORMATTER_MAPPING = {
    2: AttemptVersion2Formatter,
    'multistep.3': AttemptVersionMultiStep3Formatter,
    'multistep.4': AttemptVersionMultiStep4Formatter,
    'multistep.4.1': AttemptVersionMultiStep41Formatter,
    'multistep.4.2': AttemptVersionMultiStep42Formatter,
    'multistep.4.3': AttemptVersionMultiStep43Formatter,
}


def get_restore_events(uid):
    return get_historydb_api().events_restore(uid, 0, int(time.time()))


def format_restore_attempt(data, can_show_answers):
    """
    Преобразование данных попытки восстановления для отображения, с учетом права просмотра КО саппорта
    """
    version = data.get('version')
    formatter_cls = VERSION_TO_FORMATTER_MAPPING.get(version)
    if formatter_cls:
        return formatter_cls(can_show_answers).format(data)
    return


def get_extra_info_from_attempt(data):
    version = data.get('version')
    formatter_cls = VERSION_TO_FORMATTER_MAPPING.get(version)
    if formatter_cls:
        return formatter_cls(can_show_answers=False)._get_extra_info(data)
    return
