# -*- coding: utf-8 -*-
import logging
from logging.config import dictConfig
from os.path import (
    abspath,
    dirname,
    join,
)

from flask import (
    Blueprint,
    Flask,
)
from jinja2 import StrictUndefined as JinjaStrictUndefined
from passport.backend.api.common import escape_query_characters_middleware
from passport.backend.api.common.logbroker import get_api_logbroker_writer
from passport.backend.api.settings import settings as p_settings
from passport.backend.core.conf import (
    DynamicSettings,
    settings,
)
from passport.backend.core.cookies.cookie_l import CookieLPackError
from passport.backend.core.cookies.lrandoms import get_lrandoms_manager
from passport.backend.core.cookies.utils.utils import fix_simple_coookie_samesite_issue29613
from passport.backend.core.crypto.signing import (
    get_signing_registry,
    get_yandex_and_yandex_team_signing_registry,
)
from passport.backend.core.dbmanager.manager import (
    get_db_errors,
    get_dbm,
)
from passport.backend.core.dbmanager.sharder import (
    build_range_shard_function,
    get_sharder,
)
from passport.backend.core.geobase import (
    get_as_lookup,
    get_geobase,
)
from passport.backend.core.grants import get_grants_config
from passport.backend.core.host.host import get_current_host
from passport.backend.core.language_detect import get_language_detect
from passport.backend.core.lazy_loader import LazyLoader
from passport.backend.core.logbroker.exceptions import BaseLogbrokerError
from passport.backend.core.mailer import get_mailer
from passport.backend.core.password import get_password_qualifier
from passport.backend.core.portallib import (
    get_ipreg,
    get_uatraits,
)
from passport.backend.core.redis_manager.redis_manager import get_redis_mgr
from passport.backend.core.suggest.first_names import get_first_names
from passport.backend.core.suggest.transliterations import get_transliteration_rules
from passport.backend.core.turboapp_partners import get_ydb_turboapp_partners
from passport.backend.core.tvm import get_tvm_credentials_manager
from passport.backend.core.types.phone_number.phone_number import initialize as initialize_phone_number
from passport.backend.core.ydb import (
    get_ydb_drive_session,
    get_ydb_family_invite,
    get_ydb_profile,
    get_ydb_support_code,
)
from passport.backend.core.ydb.exceptions import YdbInstanceNotAvailable
from passport.backend.utils import template_loader
from passport.backend.utils.string import smart_text
from sqlalchemy.exc import ResourceClosedError
from werkzeug.datastructures import ImmutableDict
import yenv


log = logging.getLogger('passport.initialize')


def create_app():
    from . import common
    from . import views
    from . import templatefilters
    from . import templatetags
    from .legacy import views as views_legacy
    from .legacy import common as common_legacy
    import passport.backend.api.email_validator.views as email_validator
    import passport.backend.api.mobile_proxy.controllers as mobile_proxy

    import passport.backend.api.views.bundle.account as account_bundle
    import passport.backend.api.views.bundle.account.avatars.controllers as avatars_set_default_bundle
    import passport.backend.api.views.bundle.account.external_data.controllers as external_data_bundle
    import passport.backend.api.views.bundle.auth.social.controllers as auth_social_bundle
    import passport.backend.api.views.bundle.auth.password as password_bundle
    import passport.backend.api.views.bundle.auth.oauth as oauth_bundle
    import passport.backend.api.views.bundle.auth.otp as auth_otp_bundle
    import passport.backend.api.views.bundle.auth.sso as auth_sso_bundle
    import passport.backend.api.views.bundle.auth.token as token_bundle
    import passport.backend.api.views.bundle.auth.key_link as auth_by_key_link_bundle
    import passport.backend.api.views.bundle.auth.drive.controllers as auth_drive_bundle
    import passport.backend.api.views.bundle.auth.forwarding as auth_forwarding_bundle
    import passport.backend.api.views.bundle.auth as auth_bundle
    import passport.backend.api.views.bundle.autofill as autofill_bundle
    import passport.backend.api.views.bundle.auth.suggest.controllers as auth_suggest_bundle
    import passport.backend.api.views.bundle.auth.suggest_by_phone.controllers as auth_suggest_by_phone_bundle
    import passport.backend.api.views.bundle.billing as billing_bundle
    import passport.backend.api.views.bundle.billing.trust_3ds as trust_3ds
    import passport.backend.api.views.bundle.captcha as captcha_bundle
    import passport.backend.api.views.bundle.challenge.controllers as challenge_bundle
    import passport.backend.api.views.bundle.change_avatar.controllers as avatars_change_bundle
    import passport.backend.api.views.bundle.change_password as change_password_bundle
    import passport.backend.api.views.bundle.cookies.controllers as cookies_bundle
    import passport.backend.api.views.bundle.complete as account_complete_bundle
    import passport.backend.api.views.bundle.device_public_key.controllers as device_public_key_bundle
    import passport.backend.api.views.bundle.email as email_bundle
    import passport.backend.api.views.bundle.family as family_bundle
    import passport.backend.api.views.bundle.idm as idm_bundle
    import passport.backend.api.views.bundle.internal as test_bundle
    import passport.backend.api.views.bundle.mail_subscriptions.controllers as mail_subscriptions_bundle
    import passport.backend.api.views.bundle.mda2 as mda2_bundle
    import passport.backend.api.views.bundle.mdapi as mdapi_bundle
    import passport.backend.api.views.bundle.mobile.controllers as mobile_bundle
    import passport.backend.api.views.bundle.oauth.controllers as oauth_custom_bundle
    import passport.backend.api.views.bundle.otp as otp_bundle
    import passport.backend.api.views.bundle.otp.disable as otp_disable_bundle
    import passport.backend.api.views.bundle.otp.enable as otp_enable_bundle
    import passport.backend.api.views.bundle.otp.migrate as otp_migrate_bundle
    import passport.backend.api.views.bundle.otp.rfc.controllers as rfc_otp_bundle
    import passport.backend.api.views.bundle.phone.controllers as phone_bundle
    import passport.backend.api.views.bundle.phone.controllers_v2 as phone_bundle_2
    import passport.backend.api.views.bundle.phone.manage.controllers as phone_manage_bundle
    import passport.backend.api.views.bundle.push.controllers as push_bundle
    import passport.backend.api.views.bundle.push.push_2fa.controllers as push_2fa_bundle
    import passport.backend.api.views.bundle.register.controllers as register_bundle
    import passport.backend.api.views.bundle.restore.controllers as restore_bundle
    import passport.backend.api.views.bundle.restore.login.controllers as login_restore_bundle
    import passport.backend.api.views.bundle.restore.method_controllers as restore_methods_bundle
    import passport.backend.api.views.bundle.restore.other_controllers as restore_other_bundle
    import passport.backend.api.views.bundle.restore.semi_auto.controllers as restore_semi_auto_bundle
    import passport.backend.api.views.bundle.restore.semi_auto.step_controllers as restore_semi_auto_step_bundle
    import passport.backend.api.views.bundle.register.get_or_create as get_or_create_bundle
    import passport.backend.api.views.bundle.security.controllers as security_bundle
    import passport.backend.api.views.bundle.session as session_bundle
    import passport.backend.api.views.bundle.social.controllers as social_change_bundle
    import passport.backend.api.views.bundle.suggest as suggest_bundle
    import passport.backend.api.views.bundle.support_code.controllers as support_code_bundle
    import passport.backend.api.views.bundle.takeout.controllers as takeout_bundle
    import passport.backend.api.views.bundle.track.controllers as track_bundle
    import passport.backend.api.views.bundle.validate.controllers as validate_bundle
    import passport.backend.api.views.bundle.webauthn.controllers as webauthn_bundle
    import passport.backend.api.views.bundle.yakey_backups as yakey_backup_bundle

    import passport.backend.api.yasms.controllers as yasms

    # новое api
    api = Blueprint('api', __name__)
    # API версии 2
    api_2 = Blueprint('api_2', __name__)

    # создание аккаунта
    api.add_url_rule('/account/register/alternative/',
                     view_func=views.account_register_alternative, methods=['POST'])
    api.add_url_rule('/account/register/email/',
                     view_func=views.account_register_require_confirmed_email, methods=['POST'])
    api.add_url_rule('/account/register/phone/',
                     view_func=views.account_register_require_confirmed_phone, methods=['POST'])
    api.add_url_rule('/account/register/phonish/',
                     view_func=views.account_register_phonish, methods=['POST'])
    api.add_url_rule('/account/register/uncompleted/',
                     view_func=views.account_register_uncompleted, methods=['POST'])
    api.add_url_rule('/account/register/uncompleted/setpassword/',
                     view_func=views.account_uncompleted_set_password, methods=['POST'])

    # подписка на сервис
    api.add_url_rule('/account/<int:uid>/subscription/<path:service>/',
                     view_func=views.subscription, methods=['POST', 'PUT'])
    api.add_url_rule('/account/<int:uid>/subscription/<path:service>/',
                     view_func=views.delete_subscription, methods=['DELETE'])

    # пока только блокирование аккаунта
    api.add_url_rule('/account/<int:uid>/',
                     view_func=views.account, methods=['POST'])

    # выставление кармы
    api.add_url_rule('/account/<int:uid>/karma/',
                     view_func=views.karma, methods=['POST'])

    # выставление персональных данных
    api.add_url_rule('/account/<int:uid>/person/',
                     view_func=views.person, methods=['POST'])

    # версия 2 ручки работы с опциями пароля
    api_2.add_url_rule('/account/<int:uid>/password_options/',
                       view_func=account_bundle.PasswordOptionsView.as_view(),
                       methods=['POST'])

    # ручки для работы с опциями аккаунта
    api.add_url_rule('/account/<int:uid>/options/',
                     view_func=account_bundle.AccountOptionsViewV1.as_view(),
                     methods=['POST'])
    api_2.add_url_rule('/account/<int:uid>/options/',
                       view_func=account_bundle.AccountOptionsView.as_view(),
                       methods=['POST'])
    api_2.add_url_rule('/account/options/',
                       view_func=account_bundle.AccountOptionsView.as_view(),
                       methods=['POST'],
                       endpoint='AccountOptionsViewAlias')

    # ручки для работы с дополнительными логинами ПДД-алиаса (8 алиас)
    api.add_url_rule('/account/<int:uid>/alias/pddalias/<string:login>/',
                     view_func=account_bundle.AccountPddAliasLoginCreateView.as_view(),
                     methods=['POST'])
    api.add_url_rule('/account/<int:uid>/alias/pddalias/<string:login>/',
                     view_func=account_bundle.AccountPddAliasLoginDeleteView.as_view(),
                     methods=['DELETE'])

    # ручки для работы с алиасами pdd (7й)
    api.add_url_rule('/account/<int:uid>/alias/pdddomain/',
                     view_func=account_bundle.AccountPddCreateView.as_view(),
                     methods=['POST'])

    # ручки для работы с алиасами на альтернативных доменах Яндекса
    api.add_url_rule('/account/<int:uid>/alias/altdomain/',
                     view_func=account_bundle.AccountAliasCreateView.as_view(),
                     methods=['POST'])

    # ручки для работы с банковским номером телефона (25 алиас)
    api.add_url_rule('/account/<int:uid>/alias/bank_phonenumber/',
                     view_func=account_bundle.AccountBankPhoneNumberAliasCreateView.as_view(),
                     methods=['POST'])
    api.add_url_rule('/account/<int:uid>/alias/bank_phonenumber/',
                     view_func=account_bundle.AccountBankPhoneNumberAliasDeleteView.as_view(),
                     methods=['DELETE'])

    # удаление алиасов, общая ручка на все типы
    api.add_url_rule('/account/<int:uid>/alias/<string:alias_type>/',
                     view_func=account_bundle.AccountAliasDeleteView.as_view(),
                     methods=['DELETE'])

    # активация/деактивация паролей приложений
    api.add_url_rule('/account/app_passwords/activate/',
                     view_func=account_bundle.AppPasswordsActivator.as_view(),
                     methods=['POST'])
    api.add_url_rule('/account/app_passwords/deactivate/',
                     view_func=account_bundle.AppPasswordsDeactivator.as_view(),
                     methods=['POST'])

    # сброс ПДД до "заводского" состояния
    api.add_url_rule('/account/<int:uid>/flush_pdd/submit/',
                     view_func=account_bundle.FlushPddSubmit.as_view(),
                     methods=['POST'])
    api.add_url_rule('/account/<int:uid>/flush_pdd/commit/',
                     view_func=account_bundle.FlushPddCommit.as_view(),
                     methods=['POST'])

    # капча
    api.add_url_rule('/captcha/generate/',
                     view_func=views.captcha_generate, methods=['GET', 'POST'])
    api.add_url_rule('/captcha/check/',
                     view_func=views.captcha_check, methods=['GET', 'POST'])

    # валидация
    api.add_url_rule('/validation/login/',
                     view_func=views.login_validation, methods=['GET', 'POST'])
    api.add_url_rule('/validation/password/',
                     view_func=views.password_validation, methods=['GET', 'POST'])
    api.add_url_rule('/validation/phone_number/',
                     view_func=views.phone_number_validation, methods=['GET', 'POST'])
    api.add_url_rule('/validation/hint/',
                     view_func=views.hint_validation, methods=['GET', 'POST'])
    api.add_url_rule('/validation/retpath/',
                     view_func=views.retpath_validation, methods=['GET', 'POST'])

    # различные саджесты
    api.add_url_rule('/suggest/login/',
                     view_func=views.suggest_login, methods=['GET', 'POST'])
    api.add_url_rule('/suggest/name/',
                     view_func=views.suggest_name, methods=['GET', 'POST'])
    api.add_url_rule('/suggest/gender/',
                     view_func=views.suggest_gender, methods=['GET', 'POST'])
    api.add_url_rule('/suggest/country/',
                     view_func=views.suggest_country, methods=['GET', 'POST'])
    api.add_url_rule('/suggest/timezone/',
                     view_func=views.suggest_timezone, methods=['GET', 'POST'])
    api.add_url_rule('/suggest/language/',
                     view_func=views.suggest_language, methods=['GET', 'POST'])

    # секретные вопросы
    api.add_url_rule('/questions/',
                     view_func=views.control_questions, methods=['GET', 'POST'])

    # сессии
    api.add_url_rule('/session/',
                     view_func=views.create_session, methods=['POST'])

    api.add_url_rule('/session/check/',
                     view_func=views.check_session, methods=['POST'])

    # oauth-токены
    api.add_url_rule('/oauth/token/',
                     view_func=views.create_oauth_token, methods=['POST'])

    # запись логов для статистики
    api.add_url_rule('/statbox/',
                     view_func=views.statbox_create, methods=['GET', 'POST'])

    # подтверждение телефона через смс
    api.add_url_rule('/phonenumber/send_confirmation_code/',
                     view_func=views.phone_confirmation_send_code, methods=['GET', 'POST'])
    api.add_url_rule('/phonenumber/can_resend/',
                     view_func=views.phone_confirmation_can_resend, methods=['GET', 'POST'])
    api.add_url_rule('/phonenumber/confirm/',
                     view_func=views.phone_confirmation_confirm, methods=['GET', 'POST'])
    api.add_url_rule('/phonenumber/copy/',
                     view_func=views.phone_copy, methods=['GET', 'POST'])

    # треки
    api.add_url_rule('/track/',
                     view_func=views.track_create, methods=['POST'])
    api.add_url_rule('/track/<string:track_id>/',
                     view_func=views.track_get, methods=['GET'])
    api.add_url_rule('/track/<string:track_id>/',
                     view_func=views.track_save, methods=['POST'])
    api.add_url_rule('/track/<string:track_id>/',
                     view_func=views.track_delete, methods=['DELETE'])

    # Аватарки (Внутренние ручки для записи ключа в бд паспорта)
    api.add_url_rule('/account/avatars/default/',
                     view_func=avatars_set_default_bundle.AvatarYapicInternalSetDefault.as_view(), methods=['POST'])

    # Аватарки (Ручки для страницы смены аватара)
    api.add_url_rule('/change_avatar/init/',
                     view_func=avatars_change_bundle.AvatarGetListStartTrack.as_view(), methods=['GET'])

    api.add_url_rule('/change_avatar/',
                     view_func=avatars_change_bundle.AvatarGetList.as_view(), methods=['GET'])

    api_2.add_url_rule('/change_avatar/',
                       view_func=avatars_change_bundle.AvatarUploadV2.as_view(), methods=['POST'])

    api.add_url_rule('/change_avatar/',
                     view_func=avatars_change_bundle.AvatarUpload.as_view(), methods=['POST'])

    api.add_url_rule('/change_avatar/default/',
                     view_func=avatars_change_bundle.AvatarDeleteDefault.as_view(), methods=['DELETE'])

    # social.yandex.ru (Ручки для страницы социальных аккаунтов)
    api.add_url_rule('/change_social/init/',
                     view_func=social_change_bundle.SocialGetListStartTrack.as_view(), methods=['GET'])

    api.add_url_rule('/change_social/',
                     view_func=social_change_bundle.SocialGetList.as_view(), methods=['GET'])

    api.add_url_rule('/change_social/profile/',
                     view_func=social_change_bundle.SocialDelete.as_view(), methods=['DELETE'])

    api.add_url_rule('/change_social/profile/',
                     view_func=social_change_bundle.SocialSetAuth.as_view(), methods=['POST'])

    api.add_url_rule('/change_social/subscription/',
                     view_func=social_change_bundle.SocialSetSubscription.as_view(), methods=['POST'])

    api.add_url_rule('/change_social/subscription/',
                     view_func=social_change_bundle.SocialDeleteSubscription.as_view(), methods=['DELETE'])

    # API email валидатора
    api.add_url_rule('/email/send_confirmation_email/',
                     view_func=email_bundle.SendConfirmationEmailViewV1.as_view(), methods=['POST'])

    api.add_url_rule('/email/setup_confirmed/',
                     view_func=email_bundle.SetupConfirmedEmailViewV1.as_view(), methods=['POST'])

    api.errorhandler(Exception)(common.error_handler)

    # бандлы
    bundle_api = Blueprint('bundle_api', __name__)
    bundle_api_2 = Blueprint('bundle_api_2', __name__)
    bundle_api_3 = Blueprint('bundle_api_3', __name__)
    bundle_api_4 = Blueprint('bundle_api_4', __name__)

    # АПИ для мультидоменной аутентификации
    bundle_api.add_url_rule('/mda2/container/build/',
                            view_func=mda2_bundle.BuildContainerView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/mda2/container/use/',
                            view_func=mda2_bundle.UseContainerView.as_view(),
                            methods=['POST'])

    # управление доменами ПДД
    bundle_api.add_url_rule('/pdd/domain/',
                            view_func=mdapi_bundle.PddAddDomainView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/pdd/domain/<domain_id>/',
                            view_func=mdapi_bundle.PddEditDomainView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/pdd/domain/<domain_id>/',
                            view_func=mdapi_bundle.PddDeleteDomainView.as_view(),
                            methods=['DELETE'])

    bundle_api.add_url_rule('/pdd/domain/<domain_id>/alias/',
                            view_func=mdapi_bundle.PddAddDomainAliasView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/pdd/domain/<domain_id>/alias/<alias_id>/',
                            view_func=mdapi_bundle.PddDeleteDomainAliasView.as_view(),
                            methods=['DELETE'])
    bundle_api.add_url_rule('/pdd/domain/<domain_id>/alias/<alias_id>/make_master/',
                            view_func=mdapi_bundle.PddAliasToMasterView.as_view(),
                            methods=['POST'])

    # работа с подпиской Плюс
    bundle_api.add_url_rule('/plus/<int:uid>/',
                            view_func=account_bundle.PlusChangeAttributesView.as_view(),
                            methods=['POST'])

    # социальная авторизация
    bundle_api.add_url_rule('/auth/social/start/',
                            view_func=auth_social_bundle.StartView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/social/native_start/',
                            view_func=auth_social_bundle.NativeStartView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/social/callback/',
                            view_func=auth_social_bundle.CallbackView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/social/secure_callback/',
                            view_func=auth_social_bundle.SecureCallbackView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/social/choose/',
                            view_func=auth_social_bundle.ChooseView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/social/register/',
                            view_func=auth_social_bundle.RegisterView.as_view(),
                            methods=['POST'])
    auth_social_bundle_register_info_view = auth_social_bundle.RegisterInfoView.as_view()
    bundle_api.add_url_rule('/auth/social/register/',
                            view_func=auth_social_bundle_register_info_view,
                            methods=['GET'])
    bundle_api.add_url_rule('/auth/social/issue_credentials/',
                            view_func=auth_social_bundle.IssueCredentialsView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/social/register_by_task/',
                            view_func=auth_social_bundle.RegisterByTaskView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/social/register_by_token/',
                            view_func=auth_social_bundle.RegisterByTokenView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/social/third_party_native_start/',
                            view_func=auth_social_bundle.ThirdPartyNativeStartView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/social/third_party_choose/',
                            view_func=auth_social_bundle.ThirdPartyChooseView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/social/third_party_register/',
                            view_func=auth_social_bundle.ThirdPartyRegisterView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/social/third_party_register/',
                            view_func=auth_social_bundle_register_info_view,
                            methods=['GET'])

    # авторизация
    bundle_api.add_url_rule('/auth/password/submit/',
                            view_func=password_bundle.SubmitAuthorizationView.as_view(),
                            methods=['POST'])

    # начало процесса авторизации
    bundle_api_2.add_url_rule('/auth/password/submit/',
                              view_func=password_bundle.StartSubmitView.as_view(),
                              methods=['POST'])
    # авторизация по паролю
    bundle_api_2.add_url_rule('/auth/password/commit_password/',
                              view_func=password_bundle.StartCommitPasswordView.as_view(),
                              methods=['POST'])
    # авторизация по магии
    bundle_api_2.add_url_rule('/auth/password/commit_magic/',
                              view_func=password_bundle.StartCommitMagicView.as_view(),
                              methods=['POST'])

    # начало многошагового процесса авторизации
    bundle_api.add_url_rule('/auth/password/multi_step/start/',
                            view_func=password_bundle.MultiStepStartView.as_view(),
                            methods=['POST'])
    # авторизация по паролю
    bundle_api.add_url_rule('/auth/password/multi_step/commit_password/',
                            view_func=password_bundle.MultiStepCommitPasswordView.as_view(),
                            methods=['POST'])
    # авторизация по магии
    bundle_api.add_url_rule('/auth/password/multi_step/commit_magic/',
                            view_func=password_bundle.MultiStepCommitMagicView.as_view(),
                            methods=['POST'])

    # авторизация по магии
    bundle_api.add_url_rule('/auth/password/multi_step/commit_sms_code/',
                            view_func=password_bundle.MultiStepCommitSmsCodeView.as_view(),
                            methods=['POST'])

    # авторизация после восстановления логина
    bundle_api.add_url_rule('/auth/password/multi_step/after_login_restore/',
                            view_func=password_bundle.MultiStepAuthAfterLoginRestoreView.as_view(),
                            methods=['POST'])

    # авторизация после восстановления логина - версия с антифродом
    bundle_api.add_url_rule('/auth/password/multi_step/after_suggest_by_phone/',
                            view_func=password_bundle.MultiStepAuthAfterSuggestByPhone.as_view(),
                            methods=['POST'])

    # второй фактор - ввод кода из email
    bundle_api.add_url_rule('/auth/password/multi_step/email_code/submit/',
                            view_func=password_bundle.MultiStepEmailCodeSubmitView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/password/multi_step/email_code/commit/',
                            view_func=password_bundle.MultiStepEmailCodeCommitView.as_view(),
                            methods=['POST'])

    # авторизация по магической ссылке
    bundle_api.add_url_rule('/auth/password/multi_step/magic_link/submit/',
                            view_func=password_bundle.MultiStepMagicLinkSendView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/password/multi_step/magic_link/commit/',
                            view_func=password_bundle.MultiStepMagicLinkConfirmView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/password/multi_step/magic_link/commit_registration/',
                            view_func=password_bundle.MultiStepMagicLinkConfirmRegistrationView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/password/multi_step/magic_link/invalidate/',
                            view_func=password_bundle.MultiStepMagicLinkInvalidateView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/password/multi_step/magic_link/status/',
                            view_func=password_bundle.MultiStepMagicLinkStatusView.as_view(),
                            methods=['GET'])
    bundle_api.add_url_rule('/auth/password/multi_step/magic_link/info/',
                            view_func=password_bundle.MultiStepMagicLinkInfoView.as_view(),
                            methods=['GET'])

    bundle_api.add_url_rule('/auth/password/rfc_otp/check/',
                            view_func=password_bundle.RfcOtpView.as_view(),
                            methods=['POST'])

    bundle_api.add_url_rule('/auth/password/challenge/submit/',
                            view_func=challenge_bundle.ChallengeOnAuthSubmitView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/password/challenge/commit/',
                            view_func=challenge_bundle.ChallengeOnAuthCommitView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/password/challenge/send_email/',
                            view_func=challenge_bundle.SendAuthEmailView.as_view(),
                            methods=['POST'])

    bundle_api.add_url_rule('/challenge/standalone/create_track/',
                            view_func=challenge_bundle.ChallengeStandaloneCreateTrackView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/challenge/standalone/submit/',
                            view_func=challenge_bundle.ChallengeStandaloneSubmitView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/challenge/standalone/commit/',
                            view_func=challenge_bundle.ChallengeStandaloneCommitView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/challenge/standalone/save/',
                            view_func=challenge_bundle.ChallengeStandaloneSaveView.as_view(),
                            methods=['POST'])

    bundle_api.add_url_rule('/auth/password/confirm/submit/',
                            view_func=password_bundle.ConfirmSubmit.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/password/confirm/commit_password/',
                            view_func=password_bundle.ConfirmCommitPassword.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/password/confirm/commit_magic/',
                            view_func=password_bundle.ConfirmCommitMagic.as_view(),
                            methods=['POST'])
    bundle_api_2.add_url_rule('/auth/password/complete_pdd/',
                              view_func=password_bundle.CompletePddView.as_view(),
                              methods=['POST'])
    bundle_api_2.add_url_rule('/auth/password/complete_autoregistered/',
                              view_func=password_bundle.CompleteAutoregisteredView.as_view(),
                              methods=['POST'])
    bundle_api_2.add_url_rule('/auth/password/change_password/',
                              view_func=password_bundle.ForcedChangePasswordView.as_view(),
                              methods=['POST'])
    bundle_api.add_url_rule('/auth/password/get_state/',
                            view_func=password_bundle.GetStateView.as_view(),
                            methods=['POST'])

    bundle_api.add_url_rule('/auth/suggest/',
                            view_func=auth_suggest_bundle.GetSuggestView.as_view(),
                            methods=['GET'])
    bundle_api.add_url_rule('/auth/suggest/forget/',
                            view_func=auth_suggest_bundle.DeleteAccountFromSuggestView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/suggest/get_input_login/',
                            view_func=auth_suggest_bundle.GetSuggestedInputLogin.as_view(),
                            methods=['GET'])
    bundle_api.add_url_rule('/auth/suggest/check/',
                            view_func=auth_suggest_bundle.CheckSuggestAvailability.as_view(),
                            methods=['GET'])

    bundle_api.add_url_rule('/auth/suggest/by_phone/check_availability/',
                            view_func=auth_suggest_by_phone_bundle.SuggestAccountsByPhoneCheckAvailabilityView.as_view(),
                            methods=['GET'])
    bundle_api.add_url_rule('/auth/suggest/by_phone/list/',
                            view_func=auth_suggest_by_phone_bundle.SuggestAccountsByPhoneListView.as_view(),
                            methods=['GET'])

    # авторизация через сторонний ADFS
    bundle_api.add_url_rule('/auth/sso/submit/',
                            view_func=auth_sso_bundle.SubmitToSSOLoginView.as_view(),
                            methods=['GET'])
    bundle_api.add_url_rule('/auth/sso/commit/',
                            view_func=auth_sso_bundle.CommitToSSOLoginView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/sso/complete_federal/',
                            view_func=auth_sso_bundle.CompleteFederalView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/sso/logout/',
                            view_func=auth_sso_bundle.LogoutSSOView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/sso/metadata/get/',
                            view_func=auth_sso_bundle.MetadataGeneratorView.as_view(),
                            methods=['GET'])

    bundle_api.add_url_rule('/auth/logout/',
                            view_func=auth_bundle.LogoutView.as_view(),
                            methods=['POST'])

    bundle_api.add_url_rule('/auth/additional_data/ask/',
                            view_func=auth_bundle.AskAdditionalDataView.as_view(),
                            methods=['GET'])
    bundle_api.add_url_rule('/auth/additional_data/freeze/',
                            view_func=auth_bundle.FreezeAdditionalDataAskingView.as_view(),
                            methods=['POST'])

    bundle_api.add_url_rule('/auth/change_default/',
                            view_func=auth_bundle.ChangeDefaultView.as_view(),
                            methods=['POST'])

    bundle_api.add_url_rule('/auth/oauth/',
                            view_func=oauth_bundle.OpenAuthorizationView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/oauth/code_for_am/',
                            view_func=oauth_bundle.IssueCodeForAMView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/oauth/recreate_yambot_token/',
                            view_func=oauth_bundle.RecreateYambotTokenView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/oauth/recreate_kolonkish_token/',
                            view_func=oauth_bundle.RecreateKolonkishTokenView.as_view(),
                            methods=['POST'])

    # два алиаса для одной и той же ручки обмена токена на сессию (второй - для обратной совместимости)
    bundle_api.add_url_rule('/auth/token/',
                            view_func=token_bundle.SubmitView.as_view('auth_by_token'),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/x_token/',
                            view_func=token_bundle.SubmitView.as_view('auth_by_xtoken'),
                            methods=['POST'])

    # Аутентификация по ключу
    bundle_api.add_url_rule('/auth/key_link/submit/',
                            view_func=auth_by_key_link_bundle.SubmitKeyLinkView.as_view(),
                            methods=['POST'])

    # Проброс авторизации с десктопа на мобильный по смс
    bundle_api.add_url_rule('/auth/forward_by_sms/submit/',
                            view_func=auth_forwarding_bundle.ForwardAuthBySmsSubmitView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/forward_by_sms/commit/',
                            view_func=auth_forwarding_bundle.ForwardAuthBySmsCommitView.as_view(),
                            methods=['POST'])

    bundle_api.add_url_rule('/auth/forward_drive/start/',
                            view_func=auth_drive_bundle.AuthDriveStartView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/forward_drive/build_nonce/',
                            view_func=auth_drive_bundle.AuthDriveBuildNonceView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/forward_drive/issue_authorization_code/',
                            view_func=auth_drive_bundle.AuthDriveIssueAuthorizationCodeView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/forward_drive/stop/',
                            view_func=auth_drive_bundle.AuthDriveStopView.as_view(),
                            methods=['POST'])

    # Проброс авторизации с десктопа на мобильный по track_id
    bundle_api.add_url_rule('/auth/forward_by_track/create_track/',
                            view_func=auth_forwarding_bundle.CreateTrackByTrackForwardingView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/forward_by_track/get_status/',
                            view_func=auth_forwarding_bundle.GetStatusByTrackAuthForwardingView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/forward_by_track/exchange/',
                            view_func=auth_forwarding_bundle.ExchangeByTrackAuthForwardingView.as_view(),
                            methods=['POST'])

    # Смена пароля
    bundle_api.add_url_rule('/change_password/submit/',
                            view_func=change_password_bundle.ChangePasswordSubmitView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/change_password/commit/',
                            view_func=change_password_bundle.ChangePasswordCommitView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/change_password/optional_logout/',
                            view_func=change_password_bundle.ChangePasswordOptionalLogoutView.as_view(),
                            methods=['POST'])

    # Смена пароля Кинопоиска
    bundle_api.add_url_rule('/change_password/kinopoisk/submit/',
                            view_func=change_password_bundle.ChangeKinopoiskPasswordSubmitView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/change_password/kinopoisk/commit/',
                            view_func=change_password_bundle.ChangeKinopoiskPasswordCommitView.as_view(),
                            methods=['POST'])
    # Выход из аккаунта
    bundle_api.add_url_rule('/account/logout/',
                            view_func=account_bundle.AccountLogoutView.as_view(),
                            methods=['POST'])
    # Удаление аккаунта
    bundle_api.add_url_rule('/account/<uid>/',
                            view_func=account_bundle.AccountDeleteView.as_view(),
                            methods=['DELETE'])
    bundle_api_2.add_url_rule('/delete_account/submit/',
                              view_func=account_bundle.AccountDeleteSubmitViewV2.as_view(),
                              methods=['POST'])
    bundle_api_2.add_url_rule('/delete_account/commit/',
                              view_func=account_bundle.AccountDeleteCommitViewV2.as_view(),
                              methods=['POST'])
    bundle_api_2.add_url_rule('/delete_account/send_sms/',
                              view_func=account_bundle.AccountDeleteSendPhoneCodeViewV2.as_view(),
                              methods=['POST'])
    bundle_api_2.add_url_rule('/delete_account/confirm_phone/',
                              view_func=account_bundle.AccountDeleteConfirmPhoneViewV2.as_view(),
                              methods=['POST'])
    bundle_api_2.add_url_rule('/delete_account/check_answer/',
                              view_func=account_bundle.AccountDeleteCheckAnswerViewV2.as_view(),
                              methods=['POST'])
    bundle_api_2.add_url_rule('/delete_account/send_email/',
                              view_func=account_bundle.AccountDeleteSendEmailCodeViewV2.as_view(),
                              methods=['POST'])
    bundle_api_2.add_url_rule('/delete_account/confirm_email/',
                              view_func=account_bundle.AccountDeleteConfirmEmailViewV2.as_view(),
                              methods=['POST'])
    bundle_api.add_url_rule('/delete_account/by_credentials/',
                             view_func=account_bundle.AccountDeleteByCredentialsView.as_view(),
                             methods=['POST'])

    # Установка пароля аккаунта
    bundle_api.add_url_rule('/account/<uid>/password/',
                            view_func=account_bundle.AccountSetPasswordView.as_view(),
                            methods=['POST'])

    # отправка письма и пуша при создании пароля приложений
    bundle_api.add_url_rule('/account/app_passwords/create/send_notifications/',
                            view_func=account_bundle.AppPasswordCreatedNotification.as_view(),
                            methods=['POST'])

    # Отправка оповещений о смене настроек аккаунта
    bundle_api.add_url_rule('/account/modification/send_notifications/',
                            view_func=account_bundle.AccountModificationNotify.as_view(),
                            methods=['POST'])

    # отправка письма при редактировании OAuth-приложения
    bundle_api.add_url_rule('/oauth/client/edit/send_notifications/',
                            view_func=oauth_custom_bundle.ClientEditedNotificationView.as_view(),
                            methods=['POST'])

    # проксирование ручек oauth наружу
    bundle_api.add_url_rule('/oauth/device_authorize/qr_code/submit/',
                            view_func=oauth_custom_bundle.DeviceAuthorizeSubmit.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/oauth/device_authorize/qr_code/info/',
                            view_func=oauth_custom_bundle.DeviceAuthorizeInfo.as_view(),
                            methods=['GET'])
    bundle_api.add_url_rule('/oauth/device_authorize/qr_code/commit/',
                            view_func=oauth_custom_bundle.DeviceAuthorizeCommit.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/oauth/token_by_sessionid/',
                            view_func=oauth_custom_bundle.TokenBySessionid.as_view(),
                            methods=['POST'])

    # интеграция с IDM (постфиксы урлов фиксированы требованиями IDM)
    bundle_api.add_url_rule('/idm/info/',
                            view_func=idm_bundle.IDMInfoView.as_view(),
                            methods=['GET'])
    bundle_api.add_url_rule('/idm/add-role/',
                            view_func=idm_bundle.IDMAddRoleView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/idm/remove-role/',
                            view_func=idm_bundle.IDMRemoveRoleView.as_view(),
                            methods=['POST'])

    # подписка и отписка на пуши приложений
    bundle_api.add_url_rule('/push/register/',
                            view_func=push_bundle.PushApiRegisterBundleView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/push/subscribe/',
                            view_func=push_bundle.PushApiSubscribeBundleView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/push/unsubscribe/',
                            view_func=push_bundle.PushApiUnsubscribeBundleView.as_view(),
                            methods=['POST'])

    # отправка пушей
    bundle_api.add_url_rule('/push/send/am/',
                            view_func=push_bundle.PushApiSendAMPushBundleView.as_view(),
                            methods=['POST'])

    # 2fa пуши
    bundle_api.add_url_rule('/push/2fa/send/',
                            view_func=push_2fa_bundle.SendAuthPush2faView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/push/2fa/get_code/',
                            view_func=push_2fa_bundle.GetPush2faAuthCodeView.as_view(),
                            methods=['GET'])

    # Работа с КВ/КО
    api.add_url_rule('/account/questions/',
                     view_func=account_bundle.QuestionsSetQuestionView.as_view(), methods=['POST'])
    api.add_url_rule('/account/questions/optional/',
                     view_func=account_bundle.QuestionsAddOptional.as_view(), methods=['POST'])
    api.add_url_rule('/account/questions/secure/',
                     view_func=account_bundle.QuestionsGetQuestionView.as_view(),
                     methods=['GET'])
    api.add_url_rule('/account/questions/secure/check/',
                     view_func=account_bundle.QuestionsCheckAnswerView.as_view(),
                     methods=['POST'])
    api.add_url_rule('/account/questions/history/get/',
                     view_func=account_bundle.GetQuestionsHistoryView.as_view(),
                     methods=['GET'])
    api.add_url_rule('/account/questions/history/check/',
                     view_func=account_bundle.CheckQuestionAnswerHistoryView.as_view(),
                     methods=['POST'])
    api.add_url_rule('/account/questions/change/',
                     view_func=account_bundle.QuestionsChangeView.as_view(),
                     methods=['POST'])
    bundle_api.add_url_rule('/account/questions/secure/check/',
                            view_func=account_bundle.QuestionsCheckAnswerViewV2.as_view(),
                            methods=['POST'])

    # Секретные данные аккаунта
    bundle_api.add_url_rule('/account/secrets/browser_key/',
                            view_func=account_bundle.SecretsReadBrowserKeyView.as_view(), methods=['GET'])
    bundle_api.add_url_rule('/account/secrets/browser_key/',
                            view_func=account_bundle.SecretsWriteBrowserKeyView.as_view(), methods=['POST'])
    bundle_api.add_url_rule('/account/secrets/browser_key/',
                            view_func=account_bundle.SecretsDeleteBrowserKeyView.as_view(), methods=['DELETE'])

    bundle_api.add_url_rule('/account/secrets/passman_recovery_key/',
                            view_func=account_bundle.SecretsReadPassmanRecoveryKeyView.as_view(), methods=['GET'])
    bundle_api.add_url_rule('/account/secrets/passman_recovery_key/',
                            view_func=account_bundle.SecretsWritePassmanRecoveryKeyView.as_view(), methods=['POST'])

    # Получение информации об аккаунте, находящемся в треке
    bundle_api.add_url_rule('/account/from_track/',
                            view_func=account_bundle.GetAccountInfoFromTrackView.as_view(),
                            methods=['GET'])

    # Получение данных об аккаунте
    bundle_api.add_url_rule('/account/',
                            view_func=account_bundle.GetAccountFullInfoView.as_view(),
                            methods=['GET'])
    bundle_api.add_url_rule('/account/short_info/',
                            view_func=account_bundle.GetAccountShortInfoView.as_view(),
                            methods=['GET'])

    # Расширенный lastauth
    bundle_api.add_url_rule('/account/lastauth/',
                            view_func=account_bundle.LastauthView.as_view(), methods=['GET'])
    # Получение журнала пользователя
    bundle_api.add_url_rule('/account/history/',
                            view_func=account_bundle.HistoryViewOld.as_view(), methods=['GET'])
    bundle_api_2.add_url_rule('/account/history/',
                              view_func=account_bundle.HistoryView.as_view(), methods=['GET'])
    bundle_api.add_url_rule('/account/history/challenge/',
                            view_func=account_bundle.ChallengeView.as_view(), methods=['GET'])
    bundle_api.add_url_rule('/account/events/',
                            view_func=account_bundle.EventsView.as_view(), methods=['GET'])

    # Проставление персональных данных
    bundle_api.add_url_rule('/account/person/',
                            view_func=account_bundle.UpdateAccountPersonalInfoView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/account/person/language/',
                            view_func=account_bundle.ChangeLanguageView.as_view(),
                            methods=['POST'])

    bundle_api.add_url_rule('/account/federal/change/',
                            view_func=account_bundle.FederalChangeView.as_view(),
                            methods=['POST'])

    # Проверка существования аккаунта с заданным email-адресом
    bundle_api.add_url_rule('/account/check_exists/',
                            view_func=account_bundle.AccountCheckExistsView.as_view(),
                            methods=['GET', 'POST'])

    # Возможность залогиниться заданным фониш-аккаунтом
    bundle_api.add_url_rule('/account/phonish/can_login/',
                            view_func=account_bundle.AccountPhonishCanLoginView.as_view(),
                            methods=['GET'])

    # Получение UID фониш-аккаунта по номеру телефона
    bundle_api.add_url_rule('/account/phonish/uid_by_phone/',
                            view_func=account_bundle.AccountPhonishUidByPhoneView.as_view(),
                            methods=['GET', 'POST'])

    # Запрет авторизаций для фониша (aka "удаление фониша")
    bundle_api.add_url_rule('/account/phonish/disable_auth/',
                            view_func=account_bundle.AccountPhonishDisableAuth.as_view(),
                            methods=['POST'])

    # Запрет авторизаций для фониша AM'ом по xtoken-у, доступно в mobileproxy
    # (aka "удаление фониша")
    bundle_api.add_url_rule(
        '/account/phonish/disable_auth_by_xtoken/',
        view_func=account_bundle.AccountPhonishDisableAuthByXToken.as_view(),
        methods=['POST'],
    )

    # Управление подписками на рассылки
    bundle_api.add_url_rule('/account/mail_subscriptions/',
                            view_func=mail_subscriptions_bundle.GetCurrentSubscriptions.as_view(),
                            methods=['GET'])
    bundle_api.add_url_rule('/account/mail_subscriptions/',
                            view_func=mail_subscriptions_bundle.SetCurrentSubscriptions.as_view(),
                            methods=['POST'])

    # Миграция мейлишей (переход на новый формат хранения)
    # TODO: выпилить после закрытия PASSP-19190
    bundle_api.add_url_rule('/account/mailish/migrate/',
                            view_func=account_bundle.MigrateMailish.as_view(),
                            methods=['POST'])

    # Получение данных от других сервисов
    bundle_api.add_url_rule('/account/external_data/disk/info/',
                            view_func=external_data_bundle.DiskInfoView.as_view(),
                            methods=['GET'])
    bundle_api.add_url_rule('/account/external_data/disk/subscriptions/',
                            view_func=external_data_bundle.DiskSubscriptionsView.as_view(),
                            methods=['GET'])
    bundle_api.add_url_rule('/account/external_data/afisha/orders/',
                            view_func=external_data_bundle.BiletApiOrdersView.as_view(),
                            methods=['GET'])
    bundle_api.add_url_rule('/account/external_data/afisha/order_info/',
                            view_func=external_data_bundle.BiletApiOrderInfoView.as_view(),
                            methods=['GET'])
    bundle_api.add_url_rule('/account/external_data/afisha/user_points/',
                            view_func=external_data_bundle.BiletApiUserPointsView.as_view(),
                            methods=['GET'])
    bundle_api.add_url_rule('/account/external_data/afisha/favourites/',
                            view_func=external_data_bundle.AfishaFavouritesView.as_view(),
                            methods=['GET'])
    bundle_api.add_url_rule('/account/external_data/market/orders/',
                            view_func=external_data_bundle.MarketOrdersView.as_view(),
                            methods=['GET'])
    bundle_api.add_url_rule('/account/external_data/market/favourites/',
                            view_func=external_data_bundle.MarketFavouritesView.as_view(),
                            methods=['GET'])
    bundle_api.add_url_rule('/account/external_data/video/favourites/',
                            view_func=external_data_bundle.VideoFavouritesView.as_view(),
                            methods=['GET'])
    bundle_api.add_url_rule('/account/external_data/maps/bookmarks/',
                            view_func=external_data_bundle.MapsBookmarksView.as_view(),
                            methods=['GET'])
    bundle_api.add_url_rule('/account/external_data/maps/bookmark_info/',
                            view_func=external_data_bundle.MapsBookmarkInfo.as_view(),
                            methods=['GET'])
    bundle_api.add_url_rule('/account/external_data/music/account_status/',
                            view_func=external_data_bundle.MusicAccountStatusView.as_view(),
                            methods=['GET'])
    bundle_api.add_url_rule('/account/external_data/pictures/collections/',
                            view_func=external_data_bundle.PicturesCollectionsView.as_view(),
                            methods=['GET'])
    bundle_api.add_url_rule('/account/external_data/pictures/favourites/',
                            view_func=external_data_bundle.PicturesFavouritesView.as_view(),
                            methods=['GET'])

    # Работа с роботловилкой и счетчиками yandexuid/IP/подсетей
    bundle_api.add_url_rule('/security/session/',
                            view_func=security_bundle.CreateSecuritySession.as_view(),
                            methods=['POST'])

    # телефонное апи
    bundle_api.add_url_rule('/phone/confirm/submit/',
                            view_func=phone_bundle.ConfirmSubmitter.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/confirm/commit/',
                            view_func=phone_bundle.ConfirmCommitter.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/confirm/check_status/',
                            view_func=phone_bundle.ConfirmCheckStatus.as_view(),
                            methods=['GET'])
    bundle_api.add_url_rule('/phone/confirm_bound/submit/',
                            view_func=phone_bundle.ConfirmBoundSubmitter.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/confirm_bound/commit/',
                            view_func=phone_bundle.ConfirmBoundCommitter.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/confirm_and_bind_secure/submit/',
                            view_func=phone_bundle.ConfirmAndBindSecureSubmitter.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/confirm_and_bind_secure/commit/',
                            view_func=phone_bundle.ConfirmAndBindSecureCommitter.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/confirm_and_bind_secure_and_aliasify/submit/',
                            view_func=phone_bundle.ConfirmAndBindSecureAndAliasifySubmitter.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/confirm_and_bind_secure_and_aliasify/commit/',
                            view_func=phone_bundle.ConfirmAndBindSecureAndAliasifyCommitter.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/confirm_secure_bound_and_aliasify/submit/',
                            view_func=phone_bundle.ConfirmSecureBoundAndAliasifySubmitter.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/confirm_secure_bound_and_aliasify/commit/',
                            view_func=phone_bundle.ConfirmSecureBoundAndAliasifyCommitter.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/delete_alias/submit/',
                            view_func=phone_bundle.DeleteAliasSubmitter.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/delete_alias/commit/',
                            view_func=phone_bundle.DeleteAliasCommitter.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/confirm_tracked_secure/submit/',
                            view_func=phone_bundle.ConfirmTrackedSecureSubmitter.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/confirm_tracked_secure/commit/',
                            view_func=phone_bundle.ConfirmTrackedSecureCommitter.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/confirm_and_update_tracked_secure/submit/',
                            view_func=phone_bundle.ConfirmAndUpdateTrackedSecureSubmitter.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/confirm_and_update_tracked_secure/commit/',
                            view_func=phone_bundle.ConfirmAndUpdateTrackedSecureCommitter.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/tracked_prolong_valid_secure/',
                            view_func=phone_bundle.TrackedProlongValidSecureView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/confirm_and_bind/submit/',
                            view_func=phone_bundle.ConfirmAndBindSimpleSubmitter.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/confirm_and_bind/commit/',
                            view_func=phone_bundle.ConfirmAndBindSimpleCommitter.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/is_secure_confirmed/',
                            view_func=phone_bundle.PhoneConfirmedView.as_view(),
                            methods=['POST'])
    bundle_api_2.add_url_rule('/phone/confirm_and_bind_secure/submit/',
                              view_func=phone_bundle_2.ConfirmAndBindSecureSubmitterV2.as_view(),
                              methods=['POST'])
    bundle_api_2.add_url_rule('/phone/confirm_and_bind_secure/commit/',
                              view_func=phone_bundle_2.ConfirmAndBindSecureCommitterV2.as_view(),
                              methods=['POST'])
    bundle_api_2.add_url_rule('/phone/confirm_and_bind_secure_and_aliasify/submit/',
                              view_func=phone_bundle_2.ConfirmAndBindSecureAndAliasifySubmitterV2.as_view(),
                              methods=['POST'])
    bundle_api_2.add_url_rule('/phone/confirm_and_bind_secure_and_aliasify/commit/',
                              view_func=phone_bundle_2.ConfirmAndBindSecureAndAliasifyCommitterV2.as_view(),
                              methods=['POST'])
    bundle_api_2.add_url_rule('/phone/confirm_secure_bound_and_aliasify/submit/',
                              view_func=phone_bundle_2.ConfirmSecureBoundAndAliasifySubmitterV2.as_view(),
                              methods=['POST'])
    bundle_api_2.add_url_rule('/phone/confirm_secure_bound_and_aliasify/commit/',
                              view_func=phone_bundle_2.ConfirmSecureBoundAndAliasifyCommitterV2.as_view(),
                              methods=['POST'])
    bundle_api_2.add_url_rule('/phone/bind_simple_or_confirm_bound/submit/',
                              view_func=phone_bundle_2.BindOrConfirmBoundSubmitter.as_view(),
                              methods=['POST'])
    bundle_api_2.add_url_rule('/phone/bind_simple_or_confirm_bound/commit/',
                              view_func=phone_bundle_2.BindOrConfirmBoundCommitter.as_view(),
                              methods=['POST'])
    bundle_api_2.add_url_rule('/phone/delete_alias/submit/',
                              view_func=phone_bundle_2.DeleteAliasSubmitterV2.as_view(),
                              methods=['POST'])
    bundle_api_2.add_url_rule('/phone/delete_alias/commit/',
                              view_func=phone_bundle_2.DeleteAliasCommitterV2.as_view(),
                              methods=['POST'])
    bundle_api_2.add_url_rule('/phone/bind_phone_from_phonish_to_portal/',
                              view_func=phone_bundle_2.BindPhoneFromPhonishToPortal.as_view(),
                              methods=['POST'])

    # Работа с емейл-адресами
    bundle_api.add_url_rule('/email/list_all/',
                            view_func=email_bundle.ListEmailsView.as_view(), methods=['GET', 'POST'])
    bundle_api.add_url_rule('/email/delete/',
                            view_func=email_bundle.DeleteEmailView.as_view(), methods=['POST'])
    bundle_api.add_url_rule('/email/send_confirmation_email/',
                            view_func=email_bundle.SendConfirmationEmailView.as_view(), methods=['POST'])
    bundle_api.add_url_rule('/email/confirm/by_link/',
                            view_func=email_bundle.ConfirmEmailByLinkView.as_view(), methods=['POST'])
    bundle_api.add_url_rule('/email/confirm/by_code/',
                            view_func=email_bundle.ConfirmEmailByCodeView.as_view(), methods=['POST'])
    bundle_api.add_url_rule('/email/setup_confirmed/',
                            view_func=email_bundle.SetupConfirmedEmailView.as_view(), methods=['POST'])
    bundle_api.add_url_rule('/email/delete_by_admin/',
                            view_func=email_bundle.DeleteEmailByAdminView.as_view(), methods=['POST'])

    # Ручки для подтверждения email кодом без привязки
    bundle_api.add_url_rule('/email/check_ownership/send_code/',
                            view_func=email_bundle.CheckEmailOwnershipSendCodeView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/email/check_ownership/confirm/',
                            view_func=email_bundle.CheckEmailOwnershipConfirmView.as_view(),
                            methods=['POST'])

    # Вспомогательные ручки, связанные с социализмом
    bundle_api.add_url_rule('/social/thumbnail/',
                            view_func=social_change_bundle.SocialThumbnailView.as_view(),
                            methods=['GET', 'POST'])
    bundle_api.add_url_rule('/social/fill_personal_data/submit/',
                            view_func=social_change_bundle.SocialFillPersonalDataSubmit.as_view(),
                            methods=['GET'])
    bundle_api.add_url_rule('/social/fill_personal_data/commit/',
                            view_func=social_change_bundle.SocialFillPersonalDataCommit.as_view(),
                            methods=['POST'])

    # Новые ручки для Я.СМС
    bundle_api.add_url_rule('/phone/manage/bind_simple/submit/',
                            view_func=phone_manage_bundle.BindPhoneSubmit.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/manage/bind_simple/commit/',
                            view_func=phone_manage_bundle.BindPhoneCommit.as_view(),
                            methods=['POST'])

    bundle_api.add_url_rule('/phone/manage/bind_secure/submit/',
                            view_func=phone_manage_bundle.BindSecurePhoneSubmit.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/manage/bind_secure/commit/',
                            view_func=phone_manage_bundle.BindSecurePhoneCommit.as_view(),
                            methods=['POST'])

    bundle_api.add_url_rule('/phone/manage/securify/submit/',
                            view_func=phone_manage_bundle.SecurifySubmitPhone.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/manage/securify/commit/',
                            view_func=phone_manage_bundle.SecurifyCommitPhone.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/manage/securify/sudo/',
                            view_func=phone_manage_bundle.SecurifySudoPhone.as_view(),
                            methods=['POST'])

    bundle_api.add_url_rule('/phone/manage/remove_simple/',
                            view_func=phone_manage_bundle.RemoveSimplePhone.as_view(),
                            methods=['POST'])

    bundle_api.add_url_rule('/phone/manage/remove_secure/submit/',
                            view_func=phone_manage_bundle.RemoveSecurePhoneSubmit.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/manage/remove_secure/commit/',
                            view_func=phone_manage_bundle.RemoveSecurePhoneCommit.as_view(),
                            methods=['POST'])

    bundle_api.add_url_rule('/phone/manage/replace/submit/',
                            view_func=phone_manage_bundle.ReplaceSecurePhoneSubmit.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/manage/replace/commit/',
                            view_func=phone_manage_bundle.ReplaceSecurePhoneCommit.as_view(),
                            methods=['POST'])

    bundle_api.add_url_rule('/phone/manage/aliasify/submit/',
                            view_func=phone_manage_bundle.AliasifySecurePhoneSubmit.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/manage/aliasify/commit/',
                            view_func=phone_manage_bundle.AliasifySecurePhoneCommit.as_view(),
                            methods=['POST'])

    bundle_api.add_url_rule('/phone/manage/dealiasify/submit/',
                            view_func=phone_manage_bundle.DealiasifySecurePhoneSubmit.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/manage/dealiasify/commit/',
                            view_func=phone_manage_bundle.DealiasifySecurePhoneCommit.as_view(),
                            methods=['POST'])

    bundle_api.add_url_rule('/phone/manage/send_code/',
                            view_func=phone_manage_bundle.SendSmsCode.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/manage/check_code/',
                            view_func=phone_manage_bundle.CheckSmsCode.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/manage/check_password/',
                            view_func=phone_manage_bundle.CheckPassword.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/manage/cancel_operation/',
                            view_func=phone_manage_bundle.CancelOperation.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/manage/get_state/',
                            view_func=phone_manage_bundle.GetState.as_view(),
                            methods=['GET', 'POST'])
    bundle_api.add_url_rule('/phone/manage/prolong_valid/',
                            view_func=phone_manage_bundle.ProlongValidPhone.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/manage/set_default/',
                            view_func=phone_manage_bundle.SetDefaultPhone.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/manage/enable_alias_as_email/',
                            view_func=phone_manage_bundle.EnablePhonenumberAliasAsEmail.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/phone/manage/disable_alias_as_email/',
                            view_func=phone_manage_bundle.DisablePhonenumberAliasAsEmail.as_view(),
                            methods=['POST'])

    # Ручки, проксирующие Биллинг
    bundle_api.add_url_rule(
        '/billing/list_payment_methods/',
        view_func=billing_bundle.ListPaymentsMethodView.as_view(),
        methods=['POST'],
    )
    bundle_api.add_url_rule(
        '/billing/create_binding/',
        view_func=billing_bundle.CreateBindingView.as_view(),
        methods=['POST'],
    )
    bundle_api.add_url_rule(
        '/billing/do_binding/',
        view_func=billing_bundle.DoBindingView.as_view(),
        methods=['POST'],
    )
    bundle_api.add_url_rule(
        '/billing/check_binding/',
        view_func=billing_bundle.CheckBindingView.as_view(),
        methods=['POST'],
    )
    bundle_api.add_url_rule(
        '/billing/unbind_card/',
        view_func=billing_bundle.UnbindCardView.as_view(),
        methods=['POST'],
    )

    # Ручки для webauthn
    bundle_api.add_url_rule('/webauthn/credentials/list/',
                            view_func=webauthn_bundle.WebauthnListCredentialsView.as_view(),
                            methods=['GET', 'POST'])
    bundle_api.add_url_rule('/webauthn/credentials/add/submit/',
                            view_func=webauthn_bundle.WebauthnAddCredentialSubmitView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/webauthn/credentials/add/commit/',
                            view_func=webauthn_bundle.WebauthnAddCredentialCommitView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/webauthn/credentials/remove/',
                            view_func=webauthn_bundle.WebauthnRemoveCredentialView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/webauthn/verify/submit/',
                            view_func=webauthn_bundle.WebauthnVerifySubmitView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/webauthn/verify/commit/',
                            view_func=webauthn_bundle.WebauthnVerifyCommitView.as_view(),
                            methods=['POST'])

    # Ручки для 3ds
    bundle_api.add_url_rule('/billing/3ds/verify/submit/',
                            view_func=trust_3ds.Trust3DSSubmitView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/billing/3ds/verify/commit/',
                            view_func=trust_3ds.Trust3DSCommitView.as_view(),
                            methods=['POST'])

    # Ручка для реализации legacy API email-валидатора
    email_validator_api = Blueprint('email_validator', __name__)
    email_validator_api.add_url_rule(
        '/validate.xml',
        view_func=email_validator.dispatch_validator_request,
        methods=['GET', 'POST'],
    )

    # Ручки для реализации Я.Смс на Питоне
    yasms_api = Blueprint('yasms_api', __name__)
    yasms_api.add_url_rule(
        '/checkuser',
        view_func=yasms.CheckUserView.as_view(),
        methods=['GET', 'POST'],
    )
    yasms_api.add_url_rule(
        '/haveuseroncevalidatedphone',
        view_func=yasms.HaveUserOnceValidatedPhoneView.as_view(),
        methods=['GET', 'POST'],
    )
    yasms_api.add_url_rule(
        '/validationsnumberofuserphones',
        view_func=yasms.ValidationsNumberOfUserPhonesView.as_view(),
        methods=['GET', 'POST'],
    )
    yasms_api.add_url_rule(
        '/register',
        view_func=yasms.RegisterView.as_view(),
        methods=['GET', 'POST'],
    )
    yasms_api.add_url_rule(
        '/confirm',
        view_func=yasms.ConfirmView.as_view(),
        methods=['GET', 'POST'],
    )
    yasms_api.add_url_rule(
        '/userphones',
        view_func=yasms.UserPhonesView.as_view(),
        methods=['GET', 'POST'],
    )
    yasms_api.add_url_rule(
        '/api/checkphone',
        view_func=yasms.CheckPhoneView.as_view(),
        methods=['GET', 'POST'],
    )
    yasms_api.add_url_rule(
        '/api/deletephone',
        view_func=yasms.DeletePhoneView.as_view(),
        methods=['GET', 'POST'],
    )
    yasms_api.add_url_rule(
        '/api/dropphone',
        view_func=yasms.DropPhoneView.as_view(),
        methods=['GET', 'POST'],
    )
    yasms_api.add_url_rule(
        '/api/prolongvalid',
        view_func=yasms.ProlongValidView.as_view(),
        methods=['GET', 'POST'],
    )
    yasms_api.add_url_rule(
        '/api/removeuserphones',
        view_func=yasms.RemoveUserPhonesView.as_view(),
        methods=['GET', 'POST'],
    )

    # Ручки мобильного прокси
    mobile_proxy_simple_passport_view = mobile_proxy.MobileProxySimplePassportView.as_view()
    mobileproxy_api = Blueprint('mobileproxy', __name__)
    for validate_type in ('hint', 'login', 'password', 'phone_number', 'retpath'):
        mobileproxy_api.add_url_rule(
            '/1/validation/%s/' % validate_type,
            view_func=mobile_proxy_simple_passport_view,
            methods=['GET', 'POST'],
        )
    for suggest_type in ('country', 'gender', 'language', 'login'):
        mobileproxy_api.add_url_rule(
            '/1/suggest/%s/' % suggest_type,
            view_func=mobile_proxy_simple_passport_view,
            methods=['GET', 'POST'],
        )
    mobileproxy_api.add_url_rule(
        '/1/bundle/suggest/mobile_language/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['GET'],
    )
    for register_type in ('phone', 'phonish', 'uncompleted', 'uncompleted/setpassword'):
        mobileproxy_api.add_url_rule(
            '/1/account/register/%s/' % register_type,
            view_func=mobile_proxy_simple_passport_view,
            methods=['POST'],
        )
    mobileproxy_api.add_url_rule(
        '/1/statbox/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['GET', 'POST'],
    )
    mobileproxy_api.add_url_rule(
        '/1/track/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['POST'],
    )
    mobileproxy_api.add_url_rule(
        '/1/track/<string:track_id>/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['POST'],
    )
    mobileproxy_api.add_url_rule(
        '/1/questions/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['GET', 'POST'],
    )
    mobileproxy_api.add_url_rule(
        '/1/oauth/token/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['POST'],
    )
    for action in ('generate', 'check'):
        mobileproxy_api.add_url_rule(
            '/1/captcha/%s/' % action,
            view_func=mobile_proxy_simple_passport_view,
            methods=['GET', 'POST'],
        )
    for action in ('send_confirmation_code', 'confirm'):
        mobileproxy_api.add_url_rule(
            '/1/phonenumber/%s/' % action,
            view_func=mobile_proxy_simple_passport_view,
            methods=['GET', 'POST'],
        )
    mobileproxy_api.add_url_rule(
        '/1/account/subscription/<service>/',
        view_func=mobile_proxy.MobileProxyUidPassportView.as_view(),
        methods=['POST', 'PUT', 'DELETE'],
    )
    mobileproxy_api.add_url_rule(
        '/2/change_avatar/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['POST'],
    )
    mobileproxy_bundle_phone_view_with_masked_number = mobile_proxy.MobileProxyPhoneBundleViewWithMaskedNumber.as_view()
    for action in ('submit', 'commit'):
        mobileproxy_api.add_url_rule(
            '/1/bundle/phone/confirm/%s/' % action,
            view_func=mobile_proxy_simple_passport_view,
            methods=['POST'],
        )
        mobileproxy_api.add_url_rule(
            '/1/bundle/phone/confirm_and_bind/%s/' % action,
            view_func=mobile_proxy_simple_passport_view,
            methods=['POST'],
        )
        mobileproxy_api.add_url_rule(
            '/2/bundle/phone/bind_simple_or_confirm_bound/%s/' % action,
            view_func=mobile_proxy_simple_passport_view,
            methods=['POST'],
        )
        mobileproxy_api.add_url_rule(
            '/1/bundle/phone/confirm_tracked_secure/%s/' % action,
            view_func=mobileproxy_bundle_phone_view_with_masked_number,
            methods=['POST'],
        )
    for validate_type in ('login', 'password', 'phone_number'):
        mobileproxy_api.add_url_rule(
            '/1/bundle/validate/%s/' % validate_type,
            view_func=mobile_proxy_simple_passport_view,
            methods=['GET', 'POST'],
        )
    mobileproxy_api.add_url_rule(
        '/1/bundle/auth/x_token/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['POST'],
    )
    mobileproxy_api.add_url_rule(
        '/1/bundle/auth/x_token/prepare/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['POST'],
    )
    mobileproxy_api.add_url_rule(
        '/2/bundle/auth/password/submit/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['POST'],
    )
    mobileproxy_api.add_url_rule(
        '/2/bundle/auth/password/commit_magic/',
        view_func=mobile_proxy.MobileProxyStartCommitMagicView().as_view(),
        methods=['POST'],
    )
    mobileproxy_api.add_url_rule(
        '/1/bundle/auth/oauth/code_for_am/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['POST'],
    )
    mobileproxy_api.add_url_rule(
        '/1/bundle/account/person/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['POST'],
    )
    mobileproxy_api.add_url_rule(
        '/1/bundle/account/phonish/disable_auth_by_xtoken/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['POST'],
    )
    mobileproxy_api.add_url_rule(
        '/1/bundle/account/check_exists/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['POST'],
    )
    mobileproxy_api.add_url_rule(
        '/1/bundle/account/short_info/',
        view_func=mobile_proxy.MobileProxyCacheablePassportView.as_view(),
        methods=['GET'],
    )
    mobileproxy_api.add_url_rule(
        '/1/bundle/account/',
        view_func=mobile_proxy.MobileProxyGetAccountFullInfoView.as_view(),
        methods=['GET'],
    )
    mobileproxy_api.add_url_rule(
        '/1/bundle/complete/status/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['GET'],
    )
    mobileproxy_api.add_url_rule(
        '/1/bundle/complete/submit/',
        view_func=mobile_proxy.MobileProxyCompleteSubmitView.as_view(),
        methods=['POST'],
    )
    for action in ('commit_social_with_login', 'commit_social', 'commit_lite', 'commit_neophonish'):
        mobileproxy_api.add_url_rule(
            '/1/bundle/complete/%s/' % action,
            view_func=mobile_proxy_simple_passport_view,
            methods=['POST'],
        )
    mobileproxy_api.add_url_rule(
        '/1/bundle/track/init/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['POST'],
    )
    mobileproxy_api.add_url_rule(
        '/1/bundle/mobile/start/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['POST'],
    )
    mobileproxy_api.add_url_rule(
        '/2/bundle/mobile/start/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['POST'],
    )
    for auth_type in ('password', 'rfc_otp', 'magic_link', 'sms_code', 'after_login_restore'):
        mobileproxy_api.add_url_rule(
            '/1/bundle/mobile/auth/%s/' % auth_type,
            view_func=mobile_proxy_simple_passport_view,
            methods=['POST'],
        )
    mobileproxy_api.add_url_rule(
        '/1/bundle/mobile/magic_link/send/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['POST'],
    )
    mobileproxy_api.add_url_rule(
        '/1/bundle/mobile/magic_link/status/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['GET', 'POST'],
    )
    mobileproxy_api.add_url_rule(
        '/1/bundle/mobile/register/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['POST'],
    )
    for account_type in ('lite', 'phonish', 'neophonish'):
        mobileproxy_api.add_url_rule(
            '/1/bundle/mobile/register/%s/' % account_type,
            view_func=mobile_proxy_simple_passport_view,
            methods=['POST'],
        )
    mobileproxy_api.add_url_rule(
        '/1/bundle/mobile/restore_login/',
        view_func=mobile_proxy.MobileProxyMobileRestoreLoginView.as_view(),
        methods=['GET', 'POST'],
    )
    mobileproxy_api.add_url_rule(
        '/1/bundle/mobile/validate/phone_number/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['GET', 'POST'],
    )
    mobileproxy_api.add_url_rule(
        '/1/bundle/push/subscribe/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['POST'],
    )
    mobileproxy_api.add_url_rule(
        '/1/bundle/push/unsubscribe/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['POST'],
    )
    mobileproxy_api.add_url_rule(
        '/1/bundle/auth/forward_by_track/exchange/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['POST'],
    )
    mobileproxy_api.add_url_rule(
        '/1/bundle/auth/password/multi_step/magic_link/commit/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['POST'],
    )
    mobileproxy_api.add_url_rule(
        '/1/bundle/auth/password/multi_step/magic_link/commit_registration/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['POST'],
    )
    mobileproxy_api.add_url_rule(
        '/1/bundle/auth/password/multi_step/magic_link/invalidate/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['POST'],
    )
    mobileproxy_api.add_url_rule(
        '/1/bundle/auth/forward_drive/build_nonce/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['POST'],
    )
    mobileproxy_api.add_url_rule(
        '/1/bundle/auth/forward_drive/issue_authorization_code/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['POST'],
    )
    mobileproxy_api.add_url_rule(
        '/1/bundle/auth/otp/prepare/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['POST'],
    )
    mobileproxy_api.add_url_rule(
        '/1/bundle/auth/otp/not_me/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['POST'],
    )
    mobileproxy_api.add_url_rule(
        '/1/bundle/drive_device_public_key/create/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['POST'],
    )
    mobileproxy_api.add_url_rule(
        '/1/bundle/auth/social/register_by_token/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['POST'],
    )
    mobileproxy_api.add_url_rule(
        '/1/bundle/oauth/token_by_sessionid/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['POST'],
    )
    for yakey_backup_action in ('send_code', 'check_code', 'upload', 'download', 'info'):
        mobileproxy_api.add_url_rule(
            '/1/bundle/yakey_backup/%s/' % yakey_backup_action,
            view_func=mobile_proxy_simple_passport_view,
            methods=['POST'],
        )
    mobileproxy_api.add_url_rule(
        '/1/bundle/delete_account/by_credentials/',
        view_func=mobile_proxy_simple_passport_view,
        methods=['POST'],
    )
    mobile_proxy_get_login_view = mobile_proxy.MobileProxyGetLoginView.as_view()
    mobile_proxy_user_info_view = mobile_proxy.MobileProxyUserInfoView.as_view()
    mobile_proxy_social_token_view = mobile_proxy.MobileProxySocialTokenView.as_view()
    for url_suffix in ('', '/'):
        # Данные урлы имеют опциональный слеш на конце.
        # Правило без слеша должно указываться первым.
        mobileproxy_api.add_url_rule(
            '/1/login%s' % url_suffix,
            view_func=mobile_proxy_get_login_view,
            methods=['GET', 'POST'],
        )
        mobileproxy_api.add_url_rule(
            '/1/user_info%s' % url_suffix,
            view_func=mobile_proxy_user_info_view,
            methods=['GET', 'POST'],
        )
        mobileproxy_api.add_url_rule(
            '/1/social_token%s' % url_suffix,
            view_func=mobile_proxy_social_token_view,
            methods=['POST'],
        )

    # валидация телефонного номера
    bundle_api.add_url_rule('/validate/phone_number/',
                            view_func=validate_bundle.PhoneNumber.as_view(),
                            methods=['GET', 'POST'])
    # Проверка телефонного номера через PhoneSquatter
    bundle_api.add_url_rule('/validate/phone_number/by_squatter/',
                            view_func=validate_bundle.PhoneNumberBySquatter.as_view(),
                            methods=['GET', 'POST'])
    # валидация телефонного номера по phone_id
    bundle_api.add_url_rule('/validate/phone_id/',
                            view_func=validate_bundle.PhoneID.as_view(),
                            methods=['GET', 'POST'])
    # валидация пина для процесса включения 2fa
    bundle_api.add_url_rule('/validate/otp_pin/',
                            view_func=validate_bundle.TotpPin.as_view(),
                            methods=['GET', 'POST'])
    bundle_api_2.add_url_rule('/validate/otp_pin/',
                              view_func=validate_bundle.TotpPinV2.as_view(),
                              methods=['GET', 'POST'])
    # валидация ПДД-домена
    bundle_api.add_url_rule('/validate/domain/',
                            view_func=validate_bundle.PddDomain.as_view(),
                            methods=['GET', 'POST'])

    # валидация домена для Директории
    bundle_api.add_url_rule('/validate/directory_domain/',
                            view_func=validate_bundle.DirectoryDomain.as_view(),
                            methods=['GET', 'POST'])

    # валидация логина
    bundle_api.add_url_rule('/validate/login/',
                            view_func=validate_bundle.Login.as_view(),
                            methods=['GET', 'POST'])

    # валидация пароля
    bundle_api.add_url_rule('/validate/password/',
                            view_func=validate_bundle.Password.as_view(),
                            methods=['POST'])

    # валидация display_name
    bundle_api.add_url_rule('/validate/display_name/',
                            view_func=validate_bundle.DisplayName.as_view(),
                            methods=['POST'])

    # валидация firstname
    bundle_api.add_url_rule('/validate/firstname/',
                            view_func=validate_bundle.FirstName.as_view(),
                            methods=['POST'])

    # валидация lastname
    bundle_api.add_url_rule('/validate/lastname/',
                            view_func=validate_bundle.LastName.as_view(),
                            methods=['POST'])

    # валидация firstname и lastname
    bundle_api.add_url_rule('/validate/fullname/',
                            view_func=validate_bundle.FullName.as_view(),
                            methods=['POST'])

    # валидация public_id
    bundle_api.add_url_rule('/validate/public_id/',
                            view_func=validate_bundle.PublicId.as_view(),
                            methods=['POST'])

    # ручка для сброса аватарки
    bundle_api.add_url_rule('/account/reset/avatar/',
                            view_func=account_bundle.ResetAvatarView.as_view(),
                            methods=['POST'])

    # ручка для сброса диспленейма
    bundle_api.add_url_rule('/account/reset/display_name/',
                            view_func=account_bundle.ResetDisplayNameView.as_view(),
                            methods=['POST'])

    # ручка для сброса контрольного вопроса
    bundle_api.add_url_rule('/account/reset/question/',
                            view_func=account_bundle.ResetQuestionView.as_view(),
                            methods=['POST'])

    # ручка для сброса привязанного емейла
    bundle_api.add_url_rule('/account/reset/email/',
                            view_func=account_bundle.ResetEmailView.as_view(),
                            methods=['POST'])

    # ручка для сброса привязанного телефона
    bundle_api.add_url_rule('/account/reset/phone/',
                            view_func=account_bundle.ResetPhoneView.as_view(),
                            methods=['POST'])

    # регистрация с телефонным алиасом
    bundle_api.add_url_rule('/account/register/phone_and_aliasify/',
                            view_func=register_bundle.AccountRegisterPhoneAndAlisify.as_view(),
                            methods=['POST'])

    # авторегистрация для Директа
    bundle_api.add_url_rule('/account/register/by_middleman/',
                            view_func=register_bundle.AccountRegisterByMiddleman.as_view(),
                            methods=['POST'])

    # регистрация для Денег с возможностью создания цифрового алиаса
    bundle_api_2.add_url_rule('/account/register/alternative/easy/',
                              view_func=register_bundle.AccountRegisterAlternativeEasyViewV2.as_view(),
                              methods=['POST'])

    bundle_api.add_url_rule('/account/register/send_email/',
                            view_func=register_bundle.SendRegistrationConfirmationEmailView.as_view(), methods=['POST'])
    bundle_api.add_url_rule('/account/register/confirm_email/by_code/',
                            view_func=register_bundle.ConfirmRegistrationView.as_view(), methods=['POST'])

    bundle_api.add_url_rule('/account/register/lite/submit/',
                            view_func=register_bundle.AccountRegisterLiteSubmit.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/account/register/lite/commit/',
                            view_func=register_bundle.AccountRegisterLiteCommit.as_view(),
                            methods=['POST'])

    bundle_api.add_url_rule('/account/register/neophonish/',
                            view_func=register_bundle.AccountRegisterNeophonish.as_view(),
                            methods=['POST'])

    # проверка ограничений на отправку смс при регистрации аккаунта
    bundle_api.add_url_rule('/account/register/limits/',
                            view_func=register_bundle.ShowLimits.as_view(),
                            methods=['GET', 'POST'])

    # регистрация пользователей Директории
    bundle_api.add_url_rule('/account/register/directory/',
                            view_func=register_bundle.AccountRegisterDirectory.as_view(),
                            methods=['POST'])

    # регистрация пользователей ПДД
    bundle_api.add_url_rule('/account/register/pdd/',
                            view_func=register_bundle.AccountRegisterPdd.as_view(),
                            methods=['POST'])

    # ятимная регистрация - замена для admsimplereg - доступна в интранете и в деве/тестинге
    if yenv.name == 'intranet' or yenv.type in ('development', 'testing'):
        bundle_api.add_url_rule('/account/register/intranet/',
                                view_func=register_bundle.AccountRegisterIntranet.as_view(),
                                methods=['POST'])

    # регистрация федеральных пользователей
    bundle_api.add_url_rule('/account/register/federal/',
                            view_func=register_bundle.AccountRegisterFederal.as_view(),
                            methods=['POST'])

    # регистрация mailish-пользователей
    bundle_api.add_url_rule('/account/get_or_create/mailish/',
                            view_func=get_or_create_bundle.GetOrCreateMailish.as_view(),
                            methods=['POST'])

    # регистрация uber-пользователей
    bundle_api.add_url_rule('/account/get_or_create/uber/',
                            view_func=get_or_create_bundle.GetOrCreateUberUserView.as_view(),
                            methods=['POST'])

    # Создаем фониш-аккаунт или возвращаем uid существующего фониша по номеру телефона
    bundle_api.add_url_rule('/account/get_or_create/phonish/',
                            view_func=get_or_create_bundle.GetOrCreatePhonishView.as_view(),
                            methods=['POST'])

    # регистрация на замену старому admreg
    bundle_api.add_url_rule('/account/register/',
                            view_func=register_bundle.AccountCreate.as_view(),
                            methods=['POST'])

    # регистрация бота для Ямба
    bundle_api.add_url_rule('/account/register/yambot/',
                            view_func=register_bundle.AccountRegisterYambot.as_view(),
                            methods=['POST'])

    # регистрация колонкиша
    bundle_api.add_url_rule('/account/register/kolonkish/',
                            view_func=register_bundle.AccountRegisterKolonkish.as_view(),
                            methods=['POST'])

    # ручки для автофилла
    bundle_api.add_url_rule('/autofill/validate_partner/',
                            view_func=autofill_bundle.ValidatePartnerView.as_view(),
                            methods=['GET', 'POST'])

    # дорегистрация аккаунтов
    bundle_api.add_url_rule('/complete/status/',
                            view_func=account_complete_bundle.CompleteStatusView.as_view(),
                            methods=['GET'])
    bundle_api.add_url_rule('/complete/submit/',
                            view_func=account_complete_bundle.CompleteSubmitView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/complete/commit_social/',
                            view_func=account_complete_bundle.CompleteCommitSocialView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/complete/commit_social_with_login/',
                            view_func=account_complete_bundle.CompleteCommitSocialWithLoginView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/complete/commit_lite/',
                            view_func=account_complete_bundle.CompleteCommitLiteView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/complete/force_commit_lite/',
                            view_func=account_complete_bundle.ForceCompleteCommitLiteView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/complete/commit_neophonish/',
                            view_func=account_complete_bundle.CompleteCommitNeophonishView.as_view(),
                            methods=['POST'])

    # куки
    bundle_api.add_url_rule('/cookies/l/parse/',
                            view_func=cookies_bundle.ParseCookieL.as_view(),
                            methods=['POST'])

    # Восстановление доступа - начало процедуры
    bundle_api.add_url_rule('/restore/submit/',
                            view_func=restore_bundle.RestoreSubmitView.as_view(),
                            methods=['POST'])
    # Начало (продолжение) процедуры с использованием кода восстановления
    bundle_api.add_url_rule('/restore/key/submit/',
                            view_func=restore_bundle.RestoreKeySubmitView.as_view(),
                            methods=['POST'])
    # Саджест логина при восстановлении
    bundle_api.add_url_rule('/restore/login_suggest/',
                            view_func=restore_bundle.RestoreLoginSuggestViewV1.as_view(),
                            methods=['GET', 'POST'])
    bundle_api_2.add_url_rule('/restore/login_suggest/',
                              view_func=restore_bundle.RestoreLoginSuggestView.as_view(),
                              methods=['GET', 'POST'])
    # Получение состояния восстановления
    bundle_api.add_url_rule('/restore/get_state/',
                            view_func=restore_bundle.RestoreGetStateView.as_view(),
                            methods=['GET', 'POST'])
    # Выбор способа восстановления
    bundle_api.add_url_rule('/restore/select_method/',
                            view_func=restore_bundle.RestoreSelectMethodView.as_view(),
                            methods=['POST'])
    # Проверка номера защищенного телефона и отправка СМС
    bundle_api.add_url_rule('/restore/phone/check/',
                            view_func=restore_methods_bundle.RestoreCheckPhoneView.as_view(),
                            methods=['POST'])
    # Проверка кода подтверждения защищенного телефона
    bundle_api.add_url_rule('/restore/phone/confirm/',
                            view_func=restore_methods_bundle.RestoreConfirmPhoneView.as_view(),
                            methods=['POST'])
    # Проверка пин-кода
    bundle_api.add_url_rule('/restore/pin/check/',
                            view_func=restore_methods_bundle.RestoreCheckPinView.as_view(),
                            methods=['POST'])
    # Проверка короткой 2ФА-анкеты
    bundle_api.add_url_rule('/restore/2fa_form/check/',
                            view_func=restore_methods_bundle.RestoreCheck2FAFormView.as_view(),
                            methods=['POST'])
    # Проверка email-адреса и отправка письма с кодом восстановления
    bundle_api.add_url_rule('/restore/email/check/',
                            view_func=restore_methods_bundle.RestoreCheckEmailView.as_view(),
                            methods=['POST'])
    # Проверка контрольного ответа
    bundle_api.add_url_rule('/restore/hint/check/',
                            view_func=restore_methods_bundle.RestoreCheckAnswerView.as_view(),
                            methods=['POST'])
    # Отправка кода подтверждения для привязываемого телефона
    bundle_api.add_url_rule('/restore/new_phone/check/',
                            view_func=restore_other_bundle.RestoreCheckNewPhoneView.as_view(),
                            methods=['POST'])
    # Подтверждение привязываемого телефона
    bundle_api.add_url_rule('/restore/new_phone/confirm/',
                            view_func=restore_other_bundle.RestoreConfirmNewPhoneView.as_view(),
                            methods=['POST'])
    # Установка нового пароля и, опционально, других данных
    bundle_api.add_url_rule('/restore/commit/',
                            view_func=restore_bundle.RestoreCommitView.as_view(),
                            methods=['POST'])
    # Создание ссылки для восстановления саппортом
    bundle_api.add_url_rule('/restore/create_link/',
                            view_func=restore_bundle.RestoreCreateLinkView.as_view(),
                            methods=['POST'])

    # Анкета восстановления доступа
    bundle_api.add_url_rule('/restore/semi_auto/submit/',
                            view_func=restore_semi_auto_bundle.RestoreSemiAutoSubmitView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/restore/semi_auto/submit_with_captcha/',
                            view_func=restore_semi_auto_bundle.RestoreSemiAutoSubmitWithCaptchaView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/restore/semi_auto/form_data/',
                            view_func=restore_semi_auto_bundle.RestoreSemiAutoFormDataView.as_view(),
                            methods=['GET', 'POST'])
    bundle_api.add_url_rule('/restore/semi_auto/validate/',
                            view_func=restore_semi_auto_bundle.RestoreSemiAutoValidateView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/restore/semi_auto/short_form/',
                            view_func=restore_semi_auto_bundle.RestoreShortFormCommitView.as_view(),
                            methods=['POST'])

    # Многошаговые ручки анкеты восстановления доступа
    bundle_api_3.add_url_rule('/restore/semi_auto/get_state/',
                              view_func=restore_semi_auto_step_bundle.RestoreSemiAutoMultiStepGetStateView.as_view(),
                              methods=['GET', 'POST'])
    bundle_api_3.add_url_rule('/restore/semi_auto/validate/',
                              view_func=restore_semi_auto_step_bundle.RestoreSemiAutoMultiStepValidateView.as_view(),
                              methods=['POST'])
    bundle_api_4.add_url_rule('/restore/semi_auto/commit/',
                              view_func=restore_semi_auto_step_bundle.RestoreSemiAutoMultiStepCommitView.as_view(),
                              methods=['POST'])

    # Восстановление логина
    bundle_api.add_url_rule('/restore/login/submit/',
                            view_func=login_restore_bundle.LoginRestoreSubmitView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/restore/login/get_state/',
                            view_func=login_restore_bundle.LoginRestoreGetStateView.as_view(),
                            methods=['GET', 'POST'])
    bundle_api.add_url_rule('/restore/login/check_phone/',
                            view_func=login_restore_bundle.LoginRestoreCheckPhoneView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/restore/login/confirm_phone/',
                            view_func=login_restore_bundle.LoginRestoreConfirmPhoneView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/restore/login/check_names/',
                            view_func=login_restore_bundle.LoginRestoreCheckNamesView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/restore/login/check_names/simple/',
                            view_func=login_restore_bundle.LoginRestoreCheckNamesSimpleView.as_view(),
                            methods=['GET', 'POST'])

    # Новая ручка создания сессии после регистрации
    bundle_api.add_url_rule(
        '/session/',
        view_func=session_bundle.SessionCreateView.as_view(),
        methods=['POST'],
    )

    bundle_api.add_url_rule(
        '/session/update/',
        view_func=session_bundle.SessionUpdateView.as_view(),
        methods=['POST'],
    )

    bundle_api.add_url_rule(
        '/session/sessguard_container/use/',
        view_func=session_bundle.UseSessGuardContainerView.as_view(),
        methods=['POST'],
    )

    # ручка для создания трека с привязанным уидом
    bundle_api.add_url_rule('/track/init/',
                            view_func=track_bundle.InitializeTrackView.as_view(),
                            methods=['POST'])

    # ручка для создания трека для процесса инфицирования (PASSP-13889)
    bundle_api.add_url_rule('/track/init_infected/',
                            view_func=track_bundle.InitializeInfectedTrackView.as_view(),
                            methods=['POST'])

    # Ручка для определения дальнейшего действия по треку (для коротких ссылок)
    bundle_api.add_url_rule('/track/dispatch/',
                            view_func=track_bundle.DispatchTrackView.as_view(),
                            methods=['POST'])

    # Бандлы для тестировщиков
    bundle_api.add_url_rule('/test/confirm_and_bind_phone/',
                            view_func=test_bundle.ConfirmAndBindPhone.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/test/register_lite/',
                            view_func=test_bundle.LiteAccountRegister.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/test/track/',
                            view_func=test_bundle.GetTrackView.as_view(),
                            methods=['GET'])
    bundle_api.add_url_rule('/test/track/',
                            view_func=test_bundle.SaveTrackView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/test/totp_edit/',
                            view_func=test_bundle.TotpEdit.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/test/totp_delete/',
                            view_func=test_bundle.TotpDelete.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/test/totp_generate/',
                            view_func=test_bundle.TotpGenerate.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/test/get_phones/',
                            view_func=test_bundle.GetPhones.as_view(),
                            methods=['GET'])
    bundle_api.add_url_rule('/test/update_phone/',
                            view_func=test_bundle.UpdatePhone.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/test/set_password/',
                            view_func=test_bundle.SetPassword.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/test/persistent_tracks/',
                            view_func=test_bundle.GetPersistentTracks.as_view(),
                            methods=['GET'])
    bundle_api.add_url_rule('/test/yakey_backup/',
                            view_func=test_bundle.DeleteYakeyBackup.as_view(),
                            methods=['DELETE'])
    bundle_api.add_url_rule('/test/finish_account_deletion/',
                            view_func=test_bundle.FinishAccountDeletion.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/test/update_account_deletion_operation/',
                            view_func=test_bundle.UpdateAccountDeletionOperation.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/test/update_account_registration_datetime/',
                            view_func=test_bundle.UpdateAccountRegistrationDatetime.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/test/update_email/',
                            view_func=test_bundle.UpdateEmail.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/test/logbroker_test/',
                            view_func=test_bundle.LogbrokerTest.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/test/am_push_message/',
                            view_func=test_bundle.AMPushMessage.as_view(),
                            methods=['POST'])

    # капча
    bundle_api.add_url_rule('/captcha/status/',
                            view_func=captcha_bundle.CaptchaStatusView.as_view(),
                            methods=['GET', 'POST'])

    # 2FA - "магическая" авторизация
    bundle_api_2.add_url_rule('/auth/otp/submit/',
                              view_func=auth_otp_bundle.SubmitView.as_view(),
                              methods=['GET', 'POST'])

    bundle_api_2.add_url_rule('/auth/otp/commit/',
                              view_func=auth_otp_bundle.CommitView.as_view(),
                              methods=['POST'])

    bundle_api.add_url_rule('/auth/x_token/prepare/',
                            view_func=auth_otp_bundle.PrepareAuthWithCredView.as_view('prepare_with_xtoken'),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/prepare_with_cred/',
                            view_func=auth_otp_bundle.PrepareAuthWithCredView.as_view('prepare_with_cred'),
                            methods=['POST'])

    # ручки включения 2fa на аккаунте
    bundle_api_2.add_url_rule('/otp/enable/submit/',
                              view_func=otp_enable_bundle.OtpEnableSubmitView.as_view(),
                              methods=['POST'])
    bundle_api.add_url_rule('/otp/enable/get_secret/',
                            view_func=otp_enable_bundle.OtpEnableGetSecretView.as_view(),
                            methods=['GET', 'POST'])
    bundle_api.add_url_rule('/otp/enable/set_pin/',
                            view_func=otp_enable_bundle.OtpEnableSetPinView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/otp/enable/get_state/',
                            view_func=otp_enable_bundle.OtpEnableGetStateView.as_view(),
                            methods=['GET', 'POST'])
    bundle_api.add_url_rule('/otp/enable/check_otp/',
                            view_func=otp_enable_bundle.OtpEnableCheckOtpView.as_view(),
                            methods=['GET', 'POST'])
    bundle_api.add_url_rule('/otp/enable/commit/',
                            view_func=otp_enable_bundle.OtpEnableCommitView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/otp/enable/save_device/',
                            view_func=otp_enable_bundle.OtpEnableSaveDeviceView.as_view(),
                            methods=['POST'])

    # ручки выключения 2fa на аккаунте
    bundle_api.add_url_rule('/otp/disable/submit/',
                            view_func=otp_disable_bundle.OtpDisableSubmitView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/otp/disable/check_otp/',
                            view_func=otp_disable_bundle.OtpDisableCheckOtpView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/otp/disable/commit/',
                            view_func=otp_disable_bundle.OtpDisableCommitView.as_view(),
                            methods=['POST'])

    # ручки переноса Ключа на другое устройство
    bundle_api.add_url_rule('/otp/migrate/submit/',
                            view_func=otp_migrate_bundle.OtpMigrateSubmitView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/otp/migrate/get_secret/',
                            view_func=otp_migrate_bundle.OtpMigrateGetSecretView.as_view(),
                            methods=['GET', 'POST'])
    bundle_api.add_url_rule('/otp/migrate/check_otp/',
                            view_func=otp_migrate_bundle.OtpMigrateCheckOtpView.as_view(),
                            methods=['GET', 'POST'])
    bundle_api.add_url_rule('/otp/migrate/commit/',
                            view_func=otp_migrate_bundle.OtpMigrateCommitView.as_view(),
                            methods=['POST'])

    # временная ручка для ЧЯ
    bundle_api.add_url_rule('/otp/set_check_time/',
                            view_func=otp_bundle.OtpSetCheckTimeView.as_view(),
                            methods=['GET', 'POST'])

    # ручка включения и перегенерации RFC 2FA на я-тимном аккаунте для фронта
    bundle_api.add_url_rule('/rfc_otp/recreate_secret/',
                            view_func=rfc_otp_bundle.RecreateRfcOtpSecretView.as_view(),
                            methods=['POST'])

    # Ручки для работы с "полноценной" RFC 2FA для периметра
    bundle_api.add_url_rule('/rfc_otp/enable/',
                            view_func=rfc_otp_bundle.RfcOtpEnableView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/rfc_otp/disable/',
                            view_func=rfc_otp_bundle.RfcOtpDisableView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/rfc_otp/set_check_time/',
                            view_func=rfc_otp_bundle.RfcOtpSetCheckTimeView.as_view(),
                            methods=['POST'])

    # Апи для приложения Яндекс.Ключ.
    bundle_api.add_url_rule('/account/otp/short_info/',
                            view_func=otp_bundle.ShortInfoView.as_view(),
                            methods=['GET', 'POST'])

    bundle_api.add_url_rule('/auth/otp/prepare/',
                            view_func=otp_bundle.OtpPrepareAuthView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/auth/otp/not_me/',
                            view_func=otp_bundle.OtpNotMeView.as_view(),
                            methods=['POST'])

    # АПИ для бэкапов Ключа
    bundle_api.add_url_rule('/yakey_backup/send_code/',
                            view_func=yakey_backup_bundle.BackupSendSmsView.as_view(),
                            methods=['POST'])

    bundle_api.add_url_rule('/yakey_backup/check_code/',
                            view_func=yakey_backup_bundle.BackupCheckSmsView.as_view(),
                            methods=['POST'])

    bundle_api.add_url_rule('/yakey_backup/upload/',
                            view_func=yakey_backup_bundle.BackupUploadView.as_view(),
                            methods=['POST'])

    bundle_api.add_url_rule('/yakey_backup/download/',
                            view_func=yakey_backup_bundle.BackupDownloadView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/yakey_backup/info/',
                            view_func=yakey_backup_bundle.BackupInfoView.as_view(),
                            methods=['POST'])

    # АПИ для Takeout
    bundle_api.add_url_rule('/takeout/user_info/',
                            view_func=takeout_bundle.UserInfoView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/takeout/extract/request/',
                            view_func=takeout_bundle.RequestExtractView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/takeout/extract/start/',
                            view_func=takeout_bundle.StartExtractView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/takeout/extract/finish/',
                            view_func=takeout_bundle.FinishExtractView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/takeout/extract/get_status/',
                            view_func=takeout_bundle.GetExtractStatusView.as_view(),
                            methods=['GET', 'POST'])
    bundle_api.add_url_rule('/takeout/extract/get_archive_url/',
                            view_func=takeout_bundle.GetArchiveUrlView.as_view(),
                            methods=['GET', 'POST'])
    bundle_api.add_url_rule('/takeout/extract/get_archive_password/',
                            view_func=takeout_bundle.GetArchivePasswordView.as_view(),
                            methods=['GET', 'POST'])
    bundle_api.add_url_rule('/takeout/extract/delete_result/',
                            view_func=takeout_bundle.DeleteExtractResultView.as_view(),
                            methods=['POST'])

    # добавление подписки на сервис
    bundle_api.add_url_rule('/account/subscribe/',
                            view_func=account_bundle.SubscribeView.as_view(),
                            methods=['POST'])

    # ручки удаления подписки на сервис
    bundle_api.add_url_rule('/account/unsubscribe/submit/',
                            view_func=account_bundle.UnsubscribeSubmitView.as_view(),
                            methods=['POST'])

    bundle_api.add_url_rule('/account/unsubscribe/commit/',
                            view_func=account_bundle.UnsubscribeCommitView.as_view(),
                            methods=['POST'])

    # ручки самообеления пользователя
    bundle_api.add_url_rule('/account/userapprove/submit/',
                            view_func=account_bundle.UserApproveSubmitView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/account/userapprove/commit/',
                            view_func=account_bundle.UserApproveCommitView.as_view(),
                            methods=['POST'])

    # Саджест логина
    bundle_api.add_url_rule('/suggest/login/',
                            view_func=suggest_bundle.SuggestLoginView.as_view(),
                            methods=['GET'])

    # Саджест языка
    bundle_api.add_url_rule('/suggest/mobile_language/',
                            view_func=suggest_bundle.SuggestMobileLanguage.as_view(),
                            methods=['GET'])

    # ручки для авторизационно-регистрационного флоу АМ
    bundle_api.add_url_rule('/mobile/start/',
                            view_func=mobile_bundle.StartProcessV1View.as_view(),
                            methods=['POST'])
    bundle_api_2.add_url_rule('/mobile/start/',
                              view_func=mobile_bundle.StartProcessView.as_view(),
                              methods=['POST'])
    bundle_api.add_url_rule('/mobile/auth/password/',
                            view_func=mobile_bundle.AuthByPasswordView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/mobile/auth/rfc_otp/',
                            view_func=mobile_bundle.AuthByRfcOtpView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/mobile/auth/sms_code/',
                            view_func=mobile_bundle.AuthBySmsCodeView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/mobile/auth/after_login_restore/',
                            view_func=mobile_bundle.AuthAfterLoginRestoreView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/mobile/auth/magic_link/',
                            view_func=mobile_bundle.AuthByMagicLinkView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/mobile/magic_link/send/',
                            view_func=mobile_bundle.MagicLinkSendView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/mobile/magic_link/status/',
                            view_func=mobile_bundle.MagicLinkStatusView.as_view(),
                            methods=['GET', 'POST'])
    bundle_api.add_url_rule('/mobile/validate/phone_number/',
                            view_func=mobile_bundle.ValidatePhoneNumberView.as_view(),
                            methods=['GET', 'POST'])
    bundle_api.add_url_rule('/mobile/register/lite/',
                            view_func=mobile_bundle.MobileRegisterLiteView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/mobile/register/neophonish/',
                            view_func=mobile_bundle.MobileRegisterNeophonishView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/mobile/register/phonish/',
                            view_func=mobile_bundle.RegisterPhonishView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/mobile/register/',
                            view_func=mobile_bundle.MobileRegisterView.as_view(),
                            methods=['POST'])

    bundle_api.add_url_rule('/device_public_key/update/',
                            view_func=device_public_key_bundle.UpdateView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/device_public_key/delete/',
                            view_func=device_public_key_bundle.DeleteView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/drive_device_public_key/create/',
                            view_func=device_public_key_bundle.DriveCreateView.as_view(),
                            methods=['POST'])

    bundle_api.add_url_rule('/support_code/create/',
                            view_func=support_code_bundle.CreateSupportCodeView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/support_code/check/',
                            view_func=support_code_bundle.CheckSupportCodeView.as_view(),
                            methods=['POST'])

    # ручки семьи
    bundle_api.add_url_rule('/family/create/',
                            view_func=family_bundle.CreateFamilyView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/family/delete/',
                            view_func=family_bundle.DeleteFamilyView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/family/add_member/',
                            view_func=family_bundle.AddMemberView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/family/remove_member/',
                            view_func=family_bundle.RemoveMemberView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/family/leave/',
                            view_func=family_bundle.LeaveView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/family/issue_invite/',
                            view_func=family_bundle.IssueInviteView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/family/revoke_invite/',
                            view_func=family_bundle.RevokeInviteView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/family/accept_invite/',
                            view_func=family_bundle.AcceptInviteView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/family/invite_info/',
                            view_func=family_bundle.InviteInfoView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/family/create_kiddish/',
                            view_func=family_bundle.CreateKiddishView.as_view(),
                            methods=['POST'])
    # регистрация детского аккаунта
    bundle_api.add_url_rule('/family/create_child/',
                            view_func=family_bundle.CreateChildView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/family/delete_kiddish/',
                            view_func=family_bundle.DeleteKiddishView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/family/change_kiddish/',
                            view_func=family_bundle.ChangeKiddishView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/family/delete_child/',
                            view_func=family_bundle.DeleteChildView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/family/change_child_avatar/',
                            view_func=family_bundle.ChangeChildAvatarView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule(
        '/family/invites_info/',
        view_func=family_bundle.InvitesInfoView.as_view(),
        methods=['GET', 'POST'],
    )

    # Школьники
    bundle_api.add_url_rule('/scholar/register/',
                            view_func=register_bundle.AccountRegisterScholar.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/scholar/change_password/',
                            view_func=change_password_bundle.ChangeScholarPasswordView.as_view(),
                            methods=['POST'])
    bundle_api.add_url_rule('/scholar/update/',
                            view_func=account_bundle.UpdateScholarPersonalInfoView.as_view(),
                            methods=['POST'])

    # Дамп профайла на диск
    api.add_url_rule('/profile/dump/', view_func=views.profile_dump, methods=['POST', 'GET'])

    # поддержка старого api
    legacy_api = Blueprint('legacy_api', __name__)

    legacy_api.add_url_rule('/admloginrule', view_func=views_legacy.admloginrule,
                            methods=['GET', 'POST'])
    legacy_api.add_url_rule('/admkarma', view_func=views_legacy.admkarma,
                            methods=['GET', 'POST'])
    legacy_api.add_url_rule('/admblock', view_func=views_legacy.admblock,
                            methods=['GET', 'POST'])
    legacy_api.add_url_rule('/admchangereg', view_func=views_legacy.admchangereg,
                            methods=['GET', 'POST'])
    legacy_api.add_url_rule('/admsimplereg', view_func=views_legacy.admsimplereg,
                            methods=['GET', 'POST'])

    legacy_api.errorhandler(Exception)(common_legacy.error_handler)

    admsubscribe_legacy = Blueprint('admsubscribe_legacy', __name__)
    admsubscribe_legacy.add_url_rule('/admsubscribe',
                                     view_func=views_legacy.admsubscribe,
                                     methods=['GET', 'POST'])

    admsubscribe_legacy.errorhandler(Exception)(common_legacy.admsubscribe_error_handler)

    admreg_legacy = Blueprint('admreg_legacy', __name__)
    admreg_legacy.add_url_rule('/admreg', view_func=views_legacy.admreg,
                               methods=['GET', 'POST'])

    admreg_legacy.errorhandler(Exception)(common_legacy.admreg_error_handler)

    mailhost_legacy = Blueprint('mailhost_legacy', __name__)
    mailhost_legacy.add_url_rule('/mailhost', view_func=views_legacy.mailhost,
                                 methods=['GET', 'POST'])

    mailhost_legacy.errorhandler(Exception)(common_legacy.mailhost_error_handler)

    app = Flask(__name__)
    app.add_url_rule('/ping', view_func=views.ping)
    app.register_blueprint(api, url_prefix='/1')
    app.register_blueprint(api_2, url_prefix='/2')
    app.register_blueprint(bundle_api, url_prefix='/1/bundle')
    app.register_blueprint(bundle_api_2, url_prefix='/2/bundle')
    app.register_blueprint(bundle_api_3, url_prefix='/3/bundle')
    app.register_blueprint(bundle_api_4, url_prefix='/4/bundle')
    app.register_blueprint(legacy_api, url_prefix='/passport')
    app.register_blueprint(admsubscribe_legacy, url_prefix='/passport')
    app.register_blueprint(admreg_legacy, url_prefix='/passport')
    app.register_blueprint(mailhost_legacy)
    app.register_blueprint(yasms_api, url_prefix='/yasms')
    app.register_blueprint(email_validator_api, url_prefix='/email-validator')
    app.register_blueprint(mobileproxy_api, url_prefix='/mobileproxy')

    # FIXME error_handler не должен быть общим на все ошибки и
    # приложение, т.к. у нас часть ручек отдает json, часть xml,
    # может быть для старых ручек стоит выдавать html-страницу с
    # авторизацией, как при отсутствии грантов
    app.register_error_handler(500, common.error_handler)

    app.before_first_request(fix_simple_coookie_samesite_issue29613)
    app.before_request(common.try_reload_dynamic_settings)
    app.before_request(common.prepare_env)
    app.before_request(common.setup_log_prefix)
    app.before_request(common.log_request)

    app.after_request(common.log_response)
    app.after_request(common.log_access)

    # Настраиваем джинджу
    app.jinja_options = ImmutableDict(
        app.jinja_options,
        extensions=app.jinja_options.get('extensions', []) + templatetags.extensions(),
        undefined=JinjaStrictUndefined,
    )
    app.jinja_env = app.create_jinja_environment()
    app.jinja_env.tanker_default_keyset = settings.translations.TANKER_DEFAULT_KEYSET
    app.jinja_env.auto_reload = False
    app.jinja_env.globals.update(templatetags.tags())
    app.jinja_env.filters.update(templatefilters.filters())
    template_dir = join(dirname(abspath(__file__)), 'templates')
    app.jinja_loader = template_loader.CachedTemplateLoader(template_dir)

    LazyLoader.register('TemplateLoader', lambda: app.jinja_env)

    app.wsgi_app = escape_query_characters_middleware(app.wsgi_app)
    return app


def run_with_logging(func, name):
    log.info(u'Start initializing %s', name)
    try:
        func()
    except Exception as e:
        log.info(u'Stop initializing %s, status ERROR', name)
        log.error(u'Failed with error %s', smart_text(e))
        raise
    log.info(u'Stop initializing %s, status OK', name)


def load_lrandoms():
    manager = get_lrandoms_manager()
    if manager.current_lrandom() is None:
        raise CookieLPackError('LRandom not loaded')


def configure_db_connection(logger, db_name, config):
    dbm = get_dbm(db_name)
    dbm.configure(config)
    allowed_exceptions = get_db_errors() + (ResourceClosedError, )
    for engine in set(dbm.get_engine(force_master=force_master) for force_master in (True, False)):
        # Не используется safe_execute_query из core, потому что сложно обернуть
        # select 1 в Query, без привязки к таблице и к сессии.
        for attempt_no in range(settings.PREPARE_CONNECTION_ATTEMPTS):
            try:
                conn = engine.connect()
                # Делаем запрос, чтобы Алхимия реально открыла соединение с СУБД
                # https://st.yandex-team.ru/PASSP-19473
                conn.execute('select 1')
                # close возвращает соединение в пул Алхимии
                conn.close()
                # если дошли досюда, то прекращаем попытки прогреть соединение и выходим из цикла
                break
            except allowed_exceptions as e:
                logger.warning(u'Failed to create db connection: {}, attempt [{}/{}]'.format(
                    smart_text(e),
                    attempt_no + 1,
                    settings.PREPARE_CONNECTION_ATTEMPTS,
                ))


def init_profiler(app):
    if not settings.PROFILE_ENABLED:
        return

    import cProfile
    app.profile = cProfile.Profile()
    app.profile.enable()


def prepare_db_connections():
    # Не создаём соединения с БД в prepare_environment,
    # т.к. prepare_environment выполняется _до_ форка gunicorn,
    # и всё созданное там потом переиспользуется в воркерах _после_ форка.
    # Поэтому явно вызываем эту функцию в хуке гуникорна post_worker_init
    # уже _после_ форка, в каждом воркере, отдельно.

    # создаем подключения к базе и редисам
    for db_name, config in settings.DB_CONFIG.items():
        run_with_logging(
            lambda: configure_db_connection(log, db_name, config),
            'db name %s' % db_name,
        )

    shard_function = build_range_shard_function(settings.DB_SHARDING_RANGES)
    for table_name, config in settings.DB_SHARDING_CONFIG.items():
        run_with_logging(
            lambda: get_sharder(table_name).configure(config, shard_function),
            'table name %s for shard' % table_name,
        )

    # создаём подключения к redis-ам
    for redis_name, config in settings.REDIS_CONFIG.items():
        run_with_logging(
            lambda: get_redis_mgr(redis_name).configure(config),
            'redis on host_id %x' % redis_name,
        )


def prepare_ydb_connection():
    # Создаем объект после форка, так как PASSP-21490
    init_functions = [
        get_ydb_profile,
        get_ydb_drive_session,
        get_ydb_support_code,
        get_ydb_family_invite,
        get_ydb_turboapp_partners,
    ]
    for fn in init_functions:
        try:
            fn()
        except YdbInstanceNotAvailable:
            pass


def prepare_logbroker_connections():
    for name in settings.LOGBROKER_WRITERS.keys():
        log.info('Initializing logbroker connection {}'.format(name))
        try:
            get_api_logbroker_writer(name).warm_up()
        except BaseLogbrokerError:
            log.exception('Exception in logbroker connection {}'.format(name))


def prepare_environment():
    """
    Инициализируем сеттинги, подгружаем разные тяжёлые либы, вычитываем конфиги
    """

    # настраиваем логгинг
    dictConfig(settings.LOGGING)

    # запрашиваем хост при запуске приложения, чтобы быть уверенными
    # что этот хост есть в конфигах, иначе будет RuntimeError
    run_with_logging(get_current_host, 'host_id')

    # подгружаем менеджер TVM-креденшлов
    get_tvm_credentials_manager()

    # подгружаем конфиги для грантов
    get_grants_config()

    # подгружаем lrandoms
    load_lrandoms()

    # убеждаемся что geobase подгружена
    get_geobase()

    # убеждаемся что ASLookup подгружен
    get_as_lookup()

    # убеждаемся что ipreg подгружена
    get_ipreg()

    # убеждаемся что uatraits подгружена
    get_uatraits()

    # убеждаемся что langdetect инициализирован
    get_language_detect()

    # убеждаемся что password_qualifier инициализирован
    get_password_qualifier()

    # убеждаемся, что mailer подгружен
    get_mailer()

    # подгружаем имена для саджестора
    get_first_names()

    # подгружаем правила транслитераций для саджестора
    get_transliteration_rules()

    # Подгружаем metadata библиотеки phone numbers
    #   те делаем import region_UA; import region_RU и тд
    #   в противном случае phonenumbers использует lazy импорты
    initialize_phone_number()

    get_signing_registry()
    get_yandex_and_yandex_team_signing_registry()


def init_connections():
    prepare_db_connections()
    prepare_ydb_connection()
    prepare_logbroker_connections()


def post_worker_init(worker):
    settings.try_reload_dynamic_settings()
    init_connections()


def configure_settings():
    if not settings.configured:
        dynamic_settings = DynamicSettings(
            py_filename='/usr/lib/yandex/passport-settings/overridden_settings.py',
            import_as_name='passport_overridden_settings',
        )
        settings.configure(settings=p_settings, dynamic_settings=dynamic_settings)


def execute_app():
    configure_settings()
    prepare_environment()

    application = create_app()
    application.config['LOGGER_NAME'] = settings.APP_LOGGER_NAME

    init_profiler(application)

    return application
