# -*- coding: utf-8 -*-
from functools import partial
import logging
from time import time

from passport.backend.api.common.suggest import (
    get_countries_suggest,
    get_language_suggest,
    safe_detect_timezone,
)
from passport.backend.core.conf import settings
from passport.backend.core.cookies.cookie_lah import try_parse_cookie_lah
from passport.backend.core.counters import (
    magic_link_per_ip_counter,
    magic_link_per_uid_counter,
)
from passport.backend.core.eav_type_mapping import ALIAS_NAME_TO_TYPE
from passport.backend.core.env_profile.loader import load_ufo_profile
from passport.backend.core.mail_subscriptions.services_manager import get_mail_subscription_services_manager
from passport.backend.core.models.account import Account
from passport.backend.core.models.alias import UberAlias
from passport.backend.core.models.password import (
    PASSWORD_ENCODING_VERSION_MD5_CRYPT,
    PASSWORD_ENCODING_VERSION_MD5_CRYPT_ARGON,
)
from passport.backend.core.models.person import Person
from passport.backend.core.services import get_service
from passport.backend.core.subscription import add_subscription
from passport.backend.core.types.email.email import (
    mask_email_for_challenge,
    unicode_email,
)
from passport.backend.core.types.login.login import (
    is_test_yandex_login,
    login_is_scholar,
    normalize_login,
    raw_login_from_email,
)
from passport.backend.core.types.mobile_device_info import get_app_id_from_track
from passport.backend.core.utils.experiments import (
    is_experiment_enabled_by_login,
    is_experiment_enabled_by_uid,
)


log = logging.getLogger('passport.api.common')


PERSON_ARGS = {
    'firstname', 'lastname', 'language', 'country',
    'gender', 'timezone', 'birthday', 'city', 'display_name',
}


def build_default_person_registration_info(user_ip):
    return dict(
        **build_default_person_info(user_ip)
    )


def build_default_person_info(user_ip):
    return {
        'timezone': lambda: safe_detect_timezone(user_ip),
        'country': lambda: get_countries_suggest()[0],
        'language': lambda: get_language_suggest(),
    }


def build_empty_person_info():
    return dict(
        timezone=None,
        country=None,
        language=None,
    )


def fill_person_from_args(person, args, default_values=None, ignored_args=None):
    args = args.copy()
    args_to_fill = PERSON_ARGS - (ignored_args or set())

    display_name = args.get('display_name')
    if person.display_name and person.display_name.is_social and display_name:
        # Происходит обновление display_name для недорегистрированного социальщика: при этом мы должны обновить имя,
        # но не трогать provider и profile_id. Provider должен указывать на того соцпровайдера,
        # с которым социальщик зарегистрировался.
        person.display_name.name = display_name.name
        args.pop('display_name')

    for field in args_to_fill:
        value = args.get(field)
        default_value = default_values.get(field) if default_values else None
        if value is None and default_value is None:
            continue
        elif value is not None and default_value is None:
            setattr(person, field, value)
        elif value:
            setattr(person, field, value)
        elif default_value is not None:
            value = default_value() if callable(default_value) else default_value
            if value is not None:
                setattr(person, field, value)
    return person


def set_password_with_experiment(account, password, quality, uid=None, login=None):
    get_hash_from_blackbox = False
    version = PASSWORD_ENCODING_VERSION_MD5_CRYPT

    if uid:
        is_experiment_enabled = partial(is_experiment_enabled_by_uid, uid)
    else:
        is_experiment_enabled = partial(is_experiment_enabled_by_login, login)
    if is_experiment_enabled(settings.BLACKBOX_MD5_ARGON_HASH_DENOMINATOR):
        get_hash_from_blackbox = True
        version = PASSWORD_ENCODING_VERSION_MD5_CRYPT_ARGON
    account.password.set(
        get_hash_from_blackbox=get_hash_from_blackbox,
        password=password,
        quality=quality,
        version=version,
    )


def set_impossible_password(account):
    # В ятиме пароли хранятся в ActiveDirectory. Поэтому в паспортную БД выставим недопустимый хэш,
    # чтобы пароль считался заданным, но проверить его было нельзя.
    account.password.set_hash(password_hash='-', version=PASSWORD_ENCODING_VERSION_MD5_CRYPT_ARGON, try_create_hash=False)
    account.password.quality = 0
    account.password.quality_version = 3


def default_account(login, registration_datetime, args, default_person_info, alias_type='portal'):
    account = Account().parse({
        'subscriptions': {},
        'aliases': {
            str(ALIAS_NAME_TO_TYPE[alias_type]): login.lower(),
        },
    })
    if alias_type in ('portal', 'pdd'):
        account.user_defined_login = login.split('@', 1)[0]
    elif alias_type == 'federal':
        account.user_defined_login = login.split('/', 1)[1]

    account.karma.prefix = 0
    account.karma.suffix = 0
    account.is_enabled = args.get('is_enabled', True)

    account.registration_datetime = registration_datetime

    # Устанавливаем пароль
    if args.get('password'):
        set_password_with_experiment(account, args['password'], args['quality'], login=login)

    account.password.setup_password_changing_requirement(is_required=args.get('is_changing_required', False))
    account.password.is_creating_required = args.get('is_creating_required', False)

    if alias_type not in ('pdd', 'federal'):
        # Создание подписки на 8-ой сид
        add_subscription(
            account,
            service=get_service(slug='passport'),
            host_id=12,
            login=login,
        )

    account.person = fill_person_from_args(
        Person(account),
        args,
        default_person_info,
    )
    return account


def uber_account(uber_id, registration_datetime):
    """
    Специальный аккаунт для Uber с минимальным набором данных.
    Используется для выдачи UID по uber_id таксистам.
    """
    account = Account().parse({
        'aliases': {
            str(ALIAS_NAME_TO_TYPE['uber']): str(uber_id),
        },
        'subscriptions': {},
    })

    account.karma.prefix = 0
    account.karma.suffix = 0
    account.is_enabled = True
    account.registration_datetime = registration_datetime

    account.password.is_creating_required = False

    account.person = Person(account)

    account.uber_alias = UberAlias(account, uber_id=str(uber_id))
    return account


def yambot_account(alias, registration_datetime):
    """
    Специальный аккаунт для Ямба с минимальным набором данных.
    Используется для выдачи oauth-токена для бота.
    """
    account = Account().parse({
        'aliases': {
            str(ALIAS_NAME_TO_TYPE['yambot']): alias,
        },
        'subscriptions': {},
    })

    account.karma.prefix = 0
    account.karma.suffix = 0
    account.is_enabled = True
    account.registration_datetime = registration_datetime

    account.password.is_creating_required = False

    account.person = Person(account)

    return account


def kolonkish_account(alias, registration_datetime, creator_uid):
    """
    Специальный аккаунт для Колонки с минимальным набором данных.
    """
    account = Account().parse({
        'aliases': {
            str(ALIAS_NAME_TO_TYPE['kolonkish']): alias,
        },
        'subscriptions': {},
    })
    account.creator_uid = creator_uid

    account.karma.prefix = 0
    account.karma.suffix = 0
    account.is_enabled = True
    account.registration_datetime = registration_datetime

    account.password.is_creating_required = False

    account.person = Person(account)

    return account


def default_revokers(tokens=True, web_sessions=True, app_passwords=True, allow_select=False):
    if settings.IS_INTRANET:
        tokens = False
        web_sessions = False
        app_passwords = False
        allow_select = True
    return {
        'default': {
            'tokens': tokens,
            'web_sessions': web_sessions,
            'app_passwords': app_passwords,
        },
        'allow_select': allow_select,
    }


def any_third_steps_requiring_current_password(account):
    if (
        account.is_pdd and not account.is_complete_pdd or
        account.password.is_changing_required_by_any_reason or
        account.is_incomplete_autoregistered
    ):
        return True
    return False


def magic_link_allowed(account=None, user_ip=None, check_counters=True, check_policies=True, allow_lite=False):
    if not settings.ALLOW_MAGIC_LINK:
        return False

    if check_counters:
        if (
            not user_ip or magic_link_per_ip_counter.get_counter(user_ip).hit_limit(user_ip)
        ):
            return False
        if account is not None and magic_link_per_uid_counter.hit_limit(account.uid):
            return False

    if account is not None:
        # Нужен аккаунт определённого типа, имеющий дефолтный емейл и не обладающий усиленной защитой (2фа или 67 сид)
        if (
            account.totp_secret.is_set or
            account.is_strong_password_required or
            not (account.is_normal or account.is_pdd or (allow_lite and account.is_lite)) or
            not (account.emails.default and (account.is_lite or account.emails.default.is_native))
            or account.magic_link_login_forbidden
        ):
            return False

        # Если включена sms_2fa, то magic_link предлагаем только суперлайтам (для них это единственный способ входа)
        if not (account.is_lite and not account.password.is_set) and account.sms_2fa_on:
            return False

        if check_policies and any_third_steps_requiring_current_password(account):
            return False

    return True


def is_known_device(account, ydb_profile=None, cookies=None, max_lah_age=None,
                    device_id=None, cloud_token=None):
    if cookies:
        auth_history_container = try_parse_cookie_lah(cookies.get('lah'))
        auth_history_item = auth_history_container.by_uid(account.uid)
        if auth_history_item is not None:
            if max_lah_age is None or (time() - auth_history_item.timestamp <= max_lah_age):
                return True

    if device_id or cloud_token:
        if ydb_profile is None:
            _, _, ydb_profile = load_ufo_profile(account.uid, account.global_logout_datetime)
        if device_id and device_id in ydb_profile.trusted_device_ids:
            return True
        if cloud_token and cloud_token in ydb_profile.trusted_cloud_tokens:
            return True

    return False


def is_auth_by_sms_secure_enough_for_account(account):
    """
    Можно ли в принципе позволять входить в этот аккаунт по смс (не вызовет ли это снижения уровня защищённости)
    """
    return not (
        account.totp_secret.is_set or
        (account.sms_2fa_on and settings.FORBID_AUTH_BY_SMS_FOR_SMS_2FA) or
        account.is_strong_password_required or
        account.sms_code_login_forbidden
    )


def is_auth_by_sms_code_allowed(account, is_web, cookies=None, device_id=None, cloud_token=None):
    # отключение фичи настройкой
    if (
        (is_web and not settings.ALLOW_AUTH_BY_SMS_FOR_WEB) or
        (not is_web and not settings.ALLOW_AUTH_BY_SMS_FOR_MOBILE)
    ):
        return False

    # базовые требования
    if not (account.phonenumber_alias and account.phones.secure):
        return False

    # ограничения для безопасности
    if not is_auth_by_sms_secure_enough_for_account(account):
        return False

    if not is_web:
        # в тестинге не всегда корректно срабатывает проверка профиля
        is_test_login = is_test_yandex_login(account.login)
        if settings.ALLOW_AUTH_BY_SMS_FOR_MOBILE_ONLY_FOR_TEST_LOGINS and not is_test_login:
            return False
        if settings.AUTH_BY_SMS_FOR_MOBILE__ALLOW_SKIP_SIB_CHECKS_FOR_TEST_LOGINS and is_test_login:
            return True

    if settings.AUTH_BY_SMS__ALLOW_SKIP_SIB_CHECKS:
        return True

    # СИБ-проверки (эвристики: верим ли мы, что это тот же пользователь, а не новый владелец телефонного номера)
    return is_known_device(
        account=account,
        cookies=cookies,
        max_lah_age=settings.MAX_LAH_AGE_TO_ALLOW_AUTH_BY_SMS,
        device_id=device_id,
        cloud_token=cloud_token,
    )


def is_auth_by_magic_xtoken_allowed(account, allow_passwordless=False):
    return (
        not settings.IS_INTRANET and
        not account.totp_secret.is_set and
        (allow_passwordless or account.password.is_set) and
        not account.qr_code_login_forbidden
    )


def get_allowed_auth_methods(
    account,
    is_web=True,
    social_profiles=None,
    user_ip=None,
    cookies=None,
    device_id=None,
    cloud_token=None,
    user_entered_login=None,
    saml_settings=None,
):
    """Возвращает список доступных методов авторизации, упорядоченных по убыванию приоритета"""
    if account.scholar_password and user_entered_login and login_is_scholar(user_entered_login):
        return [settings.AUTH_METHOD_PASSWORD]

    if account.is_federal:
        if (
            account.domain.is_enabled
            and saml_settings is not None and saml_settings['idp'].get('enabled', True)
        ):
            return [settings.AUTH_METHOD_SAML_SSO]
        else:
            return []

    methods = []

    if account.totp_secret.is_set:
        if is_web:
            methods.append(settings.AUTH_METHOD_MAGIC)
        methods.append(settings.AUTH_METHOD_OTP)
    elif account.password.is_set:
        methods.append(settings.AUTH_METHOD_PASSWORD)

    if is_auth_by_sms_code_allowed(account, is_web=is_web, cookies=cookies, device_id=device_id, cloud_token=cloud_token):
        methods.append(settings.AUTH_METHOD_SMS_CODE)

    if magic_link_allowed(account, user_ip, allow_lite=settings.ALLOW_MAGIC_LINK_FOR_LITE):
        methods.append(settings.AUTH_METHOD_MAGIC_LINK)

    if is_web and is_auth_by_magic_xtoken_allowed(account):
        methods.append(settings.AUTH_METHOD_MAGIC_X_TOKEN)

    if social_profiles:
        social_providers = set([
            profile['provider_code']
            for profile in social_profiles
            if profile['allow_auth']
        ])
        methods += [
            settings.AUTH_METHOD_SOCIAL_TEMPLATE % provider.lower()
            for provider in social_providers
        ]

    return methods


def get_masked_magic_link_email(account, user_entered_login):
    """
    Емейл, который мы покажем пользователю. Если он принципиально отличается от введённого пользователем
    (например, если пользователь входит по цифровому алиасу) - емейл замаскируем.
    """
    if account.is_pdd:
        default_login = unicode_email(account.emails.default.address)
        form_login = unicode_email(user_entered_login)
        email_to_show = default_login
    else:
        default_login = raw_login_from_email(account.emails.default.address)
        form_login = raw_login_from_email(user_entered_login)
        email_to_show = account.emails.default.address
    if normalize_login(form_login) == normalize_login(default_login):
        return email_to_show
    return mask_email_for_challenge(email_to_show, mask_domain=False)


def unsubscribe_from_maillists_if_nessesary(account, form_values, track):
    if not form_values.get('unsubscribe_from_maillists'):
        return

    app_id = form_values.get('app_id') or get_app_id_from_track(track)
    origin = form_values.get('origin') or track.origin
    manager = get_mail_subscription_services_manager()

    if app_id:
        factor = 'app_id `%s`' % app_id
        services = manager.get_services_by_app_id(app_id)
    elif origin:
        factor = 'origin `%s`' % origin
        services = manager.get_services_by_origin(origin)
    else:
        factor = None
        services = None

    if services:
        log.debug(
            'Got %s. Unsubscribing from %s.' % (
                factor,
                ', '.join(map(str, [service.slug for service in services])),
            ),
        )
        account.unsubscribed_from_maillists.set(values=[service.id for service in services])
    else:
        if factor:
            log.debug('Unknown %s. Unsubscribing from all services instead.' % factor)
        else:
            log.debug('No params passed. Unsubscribing from all services.')
        account.unsubscribed_from_maillists.set(all_=True)


__all__ = (
    'any_third_steps_requiring_current_password',
    'fill_person_from_args',
    'default_account',
    'build_default_person_registration_info',
    'build_default_person_info',
    'build_empty_person_info',
    'default_revokers',
    'get_masked_magic_link_email',
    'kolonkish_account',
    'set_password_with_experiment',
    'uber_account',
    'yambot_account',
    'is_known_device',
    'get_allowed_auth_methods',
    'is_auth_by_magic_xtoken_allowed',
    'is_auth_by_sms_code_allowed',
    'is_auth_by_sms_secure_enough_for_account',
    'magic_link_allowed',
    'unsubscribe_from_maillists_if_nessesary',
    'set_impossible_password',
)
