# -*- coding: utf-8 -*-
import base64
import logging
import time

from passport.backend.api.common import account_manager
from passport.backend.api.common.phone import (
    CONFIRM_METHOD_BY_CALL,
    CONFIRM_METHOD_BY_FLASH_CALL,
    CONFIRM_METHOD_BY_SMS,
)
from passport.backend.api.exceptions import (
    AccountGlobalLogoutError,
    InvalidIpError,
    SessionExpiredError,
)
from passport.backend.core import (
    authtypes,
    Undefined,
)
from passport.backend.core.builders import blackbox
from passport.backend.core.builders.blackbox.constants import (
    BLACKBOX_ERROR_IP_NOT_YANDEX,
    BLACKBOX_ERROR_SESSION_LOGGED_OUT,
)
from passport.backend.core.builders.oauth import get_oauth
from passport.backend.core.conf import settings
from passport.backend.core.cookies.consts import (
    MY_COOKIE_NAME,
    YANDEX_GID_COOKIE_NAME,
)
from passport.backend.core.cookies.cookie_l import (
    CookieL,
    CookieLUnpackError,
)
from passport.backend.core.cookies.cookie_lah import CookieLAH
from passport.backend.core.cookies.cookie_lah.wrapper.wrapper import try_parse_cookie_lah
from passport.backend.core.cookies.cookie_y import (
    PermanentCookieYContainer,
    SessionCookieYContainer,
    YpSpSubvalueContainer,
)
from passport.backend.core.cookies.utils import (
    dump_cookie,
    MAX_COOKIE_EXPIRATION_TIMESTAMP,
)
from passport.backend.core.encrypted_container import BaseEncryptedContainer
from passport.backend.core.geobase import Region
from passport.backend.core.historydb.entry import AuthEntry
from passport.backend.core.historydb.statuses import (
    SESSION_CREATE,
    SESSION_UPDATE,
)
from passport.backend.core.host.host import get_current_host
from passport.backend.core.language_detect import get_language_number
from passport.backend.core.logging_utils.helpers import (
    mask_sessguard,
    mask_sessionid,
)
from passport.backend.core.logging_utils.loggers.statbox import (
    AntifraudLogger,
    CredentialsLogger,
    StatboxLogger,
)
from passport.backend.core.models.account import is_logouted_after
from passport.backend.core.portallib import detect_user_agent
from passport.backend.core.types.account.account import ACCOUNT_TYPE_LITE
from passport.backend.core.types.login.login import login_is_scholar
from passport.backend.core.utils.domains import (
    build_passport_domain,
    get_keyspace_by_host,
)
from passport.backend.utils.string import smart_text
from passport.backend.utils.time import (
    get_unixtime,
    string_to_integer_unixtime,
    unixtime_to_datetime,
)
import six
from six import iteritems
from six.moves.urllib.parse import (
    quote,
    urlparse,
)


auth_log = logging.getLogger('historydb.auth')
debug_log = logging.getLogger('passport.authorization')

AUTHORIZATION_SESSION_POLICY_SESSIONAL = 'sessional'
AUTHORIZATION_SESSION_POLICY_SHORT = 'short'
AUTHORIZATION_SESSION_POLICY_PERMANENT = 'long'

PDD_COOKIE_HOST = 'pochta'

COOKIE_YANDEX_GID_MAX_AGE = 3 * 365 * settings.TIMESTAMP_DELTA_ONE_DAY  # 3 года

SESS_GUARD_CONTAINER_TYPE = 'sessguard_container'
SESS_GUARD_CONTAINER_SIGN_SPACE = 'sessguard_container'
SESS_GUARD_CONTAINER_TTL = 60


class SessGuardContainer(BaseEncryptedContainer):
    type_to_ttl = {
        SESS_GUARD_CONTAINER_TYPE: SESS_GUARD_CONTAINER_TTL,
    }
    type_to_sign_space = {
        SESS_GUARD_CONTAINER_TYPE: SESS_GUARD_CONTAINER_SIGN_SPACE,
    }
    default_type = SESS_GUARD_CONTAINER_TYPE


class SessionScope(object):
    """
    Область действия сессии (например, только на сервисах, которые работают со
    школьниками).
    """
    def __init__(self):
        self._scope_set = set()

    def __str__(self):
        return ' '.join(sorted(self._scope_set))

    @classmethod
    def from_string(cls, scope):
        self = SessionScope()
        self._scope_set.update(set(scope.lower().split()))
        return self

    def issubset(self, other):
        if isinstance(other, six.string_types):
            other = SessionScope.from_string(other)
        elif not isinstance(other, SessionScope):
            raise NotImplementedError()
        return self._scope_set.issubset(other._scope_set)


# Сессия действует только на сервисах, которые умеют работать со школьниками
SessionScope.scholar = SessionScope.from_string('scholar')

# Сессия действует на всех сервисах
SessionScope.xsession = SessionScope.from_string('xsession')


def is_session_valid(status):
    return status in [
        blackbox.BLACKBOX_SESSIONID_VALID_STATUS,
        blackbox.BLACKBOX_SESSIONID_NEED_RESET_STATUS,
    ]


def is_user_session_valid(user_session):
    return is_session_valid(user_session.get('status'))


def users_from_multisession(session_info):
    users = dict()
    if session_info.response:
        users = session_info.response.get('users', {})
    return users


def user_from_multisession(session_info, uid):
    users = users_from_multisession(session_info)
    return users.get(uid, {})


def is_user_in_multisession(session_info, uid):
    users = users_from_multisession(session_info)
    return uid in users


def is_user_default_in_multisession(session_info, uid):
    return session_info.response['default_uid'] == uid


def is_user_valid_in_multisession(session_info, uid):
    if is_user_in_multisession(session_info, uid):
        user_session = user_from_multisession(session_info, uid)
        return is_user_session_valid(user_session)


def user_session_scope(session_info, uid):
    user = user_from_multisession(session_info, uid)
    auth = user.get('auth')
    if auth and auth.get('is_scholar_session'):
        return SessionScope.scholar
    return SessionScope.xsession


def to_authlog(events):
    entry = AuthEntry(**events)
    auth_log.debug(str(entry))


def build_auth_log_kwargs(env, status, uid, comment=None, auth_type=None, retpath='',
                          extra_kwargs=None, extra_comment_kwargs=None, time=None):
    comment = comment or {}

    if extra_comment_kwargs is not None:
        comment.update(extra_comment_kwargs)
    kwargs = {
        'status': status,
        'uid': uid,
        'type': auth_type or authtypes.AUTH_TYPE_WEB,
        'comment': comment,
        'host_id': get_current_host().get_id(),
        'user_ip': env.user_ip,
        'client_name': 'passport',
        'useragent': env.user_agent,
        'yandexuid': env.cookies.get('yandexuid'),
        'retpath': retpath,
        'time': time,
    }
    if extra_kwargs is not None:
        kwargs.update(extra_kwargs)
    return kwargs


def get_passed_challenge_from_track(track):
    # По полям трека определяем был ли челлендж в этой сессии, если был, то пытаемся его определить и перевести в тэги антифрода
    if track:
        passed_challenge = track.auth_challenge_type or ('captcha' if track.is_captcha_recognized else None)
        if passed_challenge == 'phone_confirmation' and track.phone_confirmation_method is not None:
            map_phone_confirm_to_af_challenge_name = {
                CONFIRM_METHOD_BY_CALL: 'call',
                CONFIRM_METHOD_BY_FLASH_CALL: 'flash_call',
                CONFIRM_METHOD_BY_SMS: 'sms',
            }
            passed_challenge = map_phone_confirm_to_af_challenge_name[track.phone_confirmation_method]
        elif passed_challenge in ('phone', 'email'):
            passed_challenge = passed_challenge + '_hint'
    else:
        passed_challenge = None
    return passed_challenge


def get_lah_cookie_uids(cookie_lah):
    auth_history_container = try_parse_cookie_lah(cookie_lah)
    return list({auth_history_item.uid for auth_history_item in auth_history_container})


def log_auth_event_for_antifraud(env, uid, retpath, track, authid, login_id):
    region_info = Region(ip=env.user_ip)
    data_to_send = {
        'uid': uid,
        'external_id': 'track-{}'.format(track.track_id) if track else 'req-{}'.format(env.request_id),
        'auth_source': track.auth_source if track else None,
        'channel': 'auth',
        'sub_channel': settings.ANTIFRAUD_AUTH_SUB_CHANNEL,
        'retpath': retpath,
        'user_agent': env.user_agent,
        'ip': env.user_ip,
        'AS': region_info.AS_list[0] if region_info.AS_list else None,
        'authid': authid,
        'login_id': login_id,
        'yandexuid': env.cookies.get('yandexuid'),
        'status': 'OK',
        'challenge': get_passed_challenge_from_track(track),
        'lah_uids': ' '.join(map(str, get_lah_cookie_uids(env.cookies.get('lah')))),
        'service_id': 'login',
        'cookie_auth': 'true',
    }
    if track:
        data_to_send.update(account_manager.get_device_params_from_track(track))
    logger = AntifraudLogger(**data_to_send)
    logger.log()


def log_auth_event(env, ttl, status, uid, authid=None, retpath=None,
                   extra_kwargs=None, extra_comment_kwargs=None, auth_type=None,
                   ):
    """
    Пишет в журнал сведения об успешном создании, изменении или удалении авторизации.

    Параметры
    @param env: объект ApiEnvironment
    @param ttl: время жизни сессии, выданной пользователю
    @param status: состояние сессии
    @param uid: UID пользователя
    @param authid: идентификатор сессии
    @param auth_type: тип авторизации
    @param retpath: URL, куда нужно перенаправить пользователя после авторизации
    """
    comment = {
        'ttl': ttl,
        'aid': authid,
    }

    kwargs = build_auth_log_kwargs(env, status, uid, comment=comment, auth_type=auth_type, retpath=retpath,
                                   extra_kwargs=extra_kwargs, extra_comment_kwargs=extra_comment_kwargs,
                                   )
    to_authlog(kwargs)


def log_web_auth_updates(env, users, ttl, authid, exclude_uid=None, retpath='',
                         modification_uid=None, modification_extra_comment_kwargs=None,
                         ):
    """
    @param env: объект ApiEnvironment
    @param users: список сведений о пользователях отданных ЧЯ
    @param ttl: время жизни обновлённой сессии
    @param authid: идентификатор сессии
    @param exclude_uid: UID пользователя, не требующего записи в журнал
    @param retpath: URL, куда нужно перенаправить пользователя после авторизации
    """
    for _, user in iteritems(users):
        if not is_user_session_valid(user):
            continue
        uid = user['uid']
        if exclude_uid and uid == exclude_uid:
            continue
        extra_comment_kwargs = None
        if modification_uid and modification_uid == uid:
            extra_comment_kwargs = modification_extra_comment_kwargs
        extra_kwargs = {'login': user['login']}

        log_auth_event(
            env=env,
            ttl=ttl,
            status=SESSION_UPDATE,
            uid=uid,
            authid=authid,
            retpath=retpath,
            # при update кук это всегда происходит на вебе и важно только это,
            # только запись 'ses_create' может иметь значение, отличное от 'web'
            auth_type=authtypes.AUTH_TYPE_WEB,
            extra_kwargs=extra_kwargs,
            extra_comment_kwargs=extra_comment_kwargs,
        )


def log_auth_challenge_shown(env, uid, auth_type=None, retpath='',
                             time=None):
    """
    Пишет в журнал сведения о показе челленджа при авторизации.
    """
    kwargs = build_auth_log_kwargs(env, 'challenge_shown', uid, auth_type=auth_type, retpath=retpath,
                                   time=time)
    to_authlog(kwargs)


def get_ttl_by_session_policy(session_policy):
    session_policy_to_ttl = {
        AUTHORIZATION_SESSION_POLICY_SESSIONAL: settings.SESSIONAL_COOKIE_TTL,
        AUTHORIZATION_SESSION_POLICY_SHORT: settings.SHORT_COOKIE_TTL,
        AUTHORIZATION_SESSION_POLICY_PERMANENT: settings.PERMANENT_COOKIE_TTL,
    }

    return session_policy_to_ttl.get(
        session_policy,
        settings.SESSIONAL_COOKIE_TTL,
    )


def get_session_policy_by_ttl(ttl):
    ttl_to_session_policy = {
        settings.SESSIONAL_COOKIE_TTL: AUTHORIZATION_SESSION_POLICY_SESSIONAL,
        settings.SHORT_COOKIE_TTL: AUTHORIZATION_SESSION_POLICY_SHORT,
        settings.PERMANENT_COOKIE_TTL: AUTHORIZATION_SESSION_POLICY_PERMANENT,
    }

    return ttl_to_session_policy.get(
        ttl,
        settings.SESSIONAL_COOKIE_TTL,
    )


def check_track_for_logout(track, logout_datetime):
    if track:
        is_logout_occured = (
            not track.is_web_sessions_logout and
            is_logouted_after(
                logout_datetime,
                unixtime_to_datetime(track.logout_checkpoint_timestamp),
            )
        )
        if is_logout_occured:
            raise AccountGlobalLogoutError()


def bb_response_to_session(bb_response, env, extra_sessguard_host=None, retpath=None):
    session = {}
    if 'new-session' in bb_response:
        session['session'] = bb_response['new-session']
    if 'new-sslsession' in bb_response:
        session['sslsession'] = bb_response['new-sslsession']
    service_guard_cookie = None
    if 'new-sessguards' in bb_response:
        passport_host = build_passport_domain(env.host)
        passport_sessguard = bb_response['new-sessguards'].get(passport_host, {}).get('sessguard')
        if passport_sessguard is not None:
            session['sessguard'] = passport_sessguard
        if extra_sessguard_host is not None and not (
            extra_sessguard_host == passport_host or
            extra_sessguard_host.endswith('.' + passport_host)
        ):
            try:
                service_guard_cookie = bb_response['new-sessguards'][extra_sessguard_host]['sessguard']
            except KeyError:
                pass
    if service_guard_cookie:
        service_guard_cookie_dumped = dump_cookie(
            'sessguard',
            service_guard_cookie['value'],
            expires=service_guard_cookie['expires'],
            domain=service_guard_cookie['domain'],
            httponly=True,
            secure=True,
            path='/',
            samesite=get_samesite_policy(env),
        )
        service_guard_container = SessGuardContainer(
            dict(
                cookies=[
                    service_guard_cookie_dumped,
                ],
                retpath=retpath,
            ),
        )
    else:
        service_guard_container = None

    return session, service_guard_container


def log_new_session(session, extra_sessguard_container=None):
    # залогируем куку целиком, для отладочных целей (информации из statbox не всегда хватает)
    debug_log.debug(
        'Got Session_id=%s, sessguard=%s, extra_sessguard_container=%s',
        mask_sessionid(session['session']['value']) if session.get('session') else '-',
        mask_sessguard(session['sessguard']['value']) if session.get('sessguard') else '-',
        mask_sessguard(extra_sessguard_container) if extra_sessguard_container else '-',
    )


def _can_browser_handle_samesite(env):
    headers = {'User-Agent': env.user_agent}
    user_agent_info = detect_user_agent(headers)
    return user_agent_info.get('SameSiteSupport', False)


def get_samesite_policy(env):
    if _can_browser_handle_samesite(env):
        # Явно говорим, что ограничения нам не нужны
        return 'None'
    else:
        # Браузер, не умеющий SameSite, и так ничего делать не будет
        return None


def authorize(
    env,
    uid,
    login,
    language,
    captcha_passed=None,
    have_password=True,
    host=None,
    keyspace=None,
    log_historydb=True,
    log_statbox=True,
    authorization_session_policy=AUTHORIZATION_SESSION_POLICY_PERMANENT,
    is_lite=False,
    track=None,
    retpath=None,
    old_session=None,
    old_ssl_session=None,
    old_session_ttl=None,
    old_sessguard=None,
    session_version=None,
    password_passed=True,
    is_yandexoid=False,
    is_betatester=False,
    social_id=None,
    extend_session=False,
    multi_session_users=None,
    auth_type=None,
    dont_change_default_uid=False,
    logout_datetime=None,
    otp_magic_passed=None,
    is_2fa_enabled=False,
    auth_challenge=None,
    auth_source=None,
    source_authid=None,
    is_session_restricted=False,
    x_token_magic_passed=None,
    login_id=None,
    need_extra_sessguard=False,
    session_scope=None,
):

    session_version = session_version or settings.BLACKBOX_SESSION_VERSION

    check_track_for_logout(track, logout_datetime)

    host = host or env.host

    # Сюда приходят уиды из трека в т.ч., которые являются строкой
    uid = int(uid)

    if multi_session_users is None:
        multi_session_users = {}

    keyspace = keyspace or get_keyspace_by_host(host)
    ttl = get_ttl_by_session_policy(authorization_session_policy)

    if not settings.IS_INTRANET:
        yateam_auth = None
    elif is_session_restricted:
        yateam_auth = False
    elif is_session_restricted is None:
        is_yandex_ip = blackbox.get_blackbox().check_ip(env.user_ip).get('yandexip')
        yateam_auth = is_yandex_ip
    else:
        yateam_auth = True

    if settings.FORCE_ISSUE_PORTAL_COOKIE_TO_LITE_USERS:
        is_lite = False

    if not session_scope:
        debug_log.warning(
            'Session_scope is not defined, fallback to Xsession. Discover the case and '
            'set session_scope properly',
        )
        session_scope = SessionScope.xsession

    if SessionScope.xsession.issubset(session_scope):
        is_scholar_session = False
    elif SessionScope.scholar.issubset(session_scope):
        is_scholar_session = True
    else:
        raise NotImplementedError()

    # TODO: встроить обработку ошибок editsession и createsession
    passport_domain = build_passport_domain(env.host)
    guard_hosts = [passport_domain]
    extra_sessguard_host = None
    if need_extra_sessguard and retpath:
        parse = urlparse(retpath)
        extra_sessguard_host = parse.hostname
        if extra_sessguard_host:
            guard_hosts.append(extra_sessguard_host)
    if extend_session:
        try:
            response = blackbox.get_blackbox().editsession(
                blackbox.BLACKBOX_EDITSESSION_OP_ADD,
                old_session,
                uid,
                env.user_ip,
                host,
                guard_hosts=guard_hosts,
                have_password=int(bool(have_password)),
                is_betatester=is_betatester,
                is_lite=is_lite,
                is_scholar=is_scholar_session,
                is_yastaff=is_yandexoid,
                keyspace=keyspace,
                lang=get_language_number(language),
                new_default=None if dont_change_default_uid else uid,
                password_check_time=time.time() if password_passed else None,
                request_id=env.request_id,
                sessguard=old_sessguard,
                social_id=social_id,
                sslsessionid=old_ssl_session,
                yateam_auth=yateam_auth,
                get_login_id=True,
            )
        except blackbox.BlackboxInvalidParamsError as ex:
            # PASSP-11214, PASSP-24286, PASSP-26157 ЧЯ уже не может работать с этой сессией
            if str(ex).startswith(BLACKBOX_ERROR_SESSION_LOGGED_OUT):
                raise SessionExpiredError(ex)
            if str(ex).startswith(BLACKBOX_ERROR_IP_NOT_YANDEX):
                raise InvalidIpError(ex)

            raise

    else:
        try:
            response = blackbox.get_blackbox().createsession(
                uid,
                env.user_ip,
                keyspace,
                ttl,
                guard_hosts=guard_hosts,
                have_password=int(bool(have_password)),
                is_betatester=is_betatester,
                is_lite=is_lite,
                is_scholar=is_scholar_session,
                is_yastaff=is_yandexoid,
                lang=get_language_number(language),
                login_id=login_id,
                password_check_time=time.time() if password_passed else None,
                request_id=env.request_id,
                social_id=social_id,
                ver=settings.BLACKBOX_SESSION_VERSION,
                yateam_auth=yateam_auth,
                get_login_id=True,
            )
        except blackbox.BlackboxInvalidParamsError as ex:
            if str(ex).startswith(BLACKBOX_ERROR_IP_NOT_YANDEX):
                raise InvalidIpError(ex)

            raise
    authid_block = response.get('authid', {})
    authid = authid_block.get('id')
    new_login_id = response.get('login_id')

    is_new_uid = uid not in multi_session_users or not is_user_session_valid(multi_session_users[uid])
    # Считаем все уиды, в т.ч. задизейбленные
    uids_count = len(multi_session_users)
    if uid not in multi_session_users:
        uids_count += 1

    if log_historydb:
        extra_comment_kwargs = {}
        if captcha_passed is not None:
            extra_comment_kwargs['cpt'] = int(captcha_passed)
        if otp_magic_passed:
            extra_comment_kwargs['mgc'] = int(otp_magic_passed)
        if x_token_magic_passed:
            extra_comment_kwargs['mgcxt'] = int(x_token_magic_passed)
        if auth_challenge:
            extra_comment_kwargs['chlng'] = auth_challenge
        if auth_source:
            extra_comment_kwargs['asrc'] = auth_source
        if source_authid:
            extra_comment_kwargs['said'] = source_authid

        log_web_auth_updates(
            env,
            multi_session_users,
            ttl,
            authid,
            retpath=retpath,
            modification_uid=uid,
            modification_extra_comment_kwargs=extra_comment_kwargs,
        )

        if is_new_uid:
            extra_kwargs = {'login': login}
            log_auth_event(
                env=env,
                ttl=ttl,
                status=SESSION_CREATE,
                uid=uid,
                authid=authid,
                retpath=retpath,
                extra_kwargs=extra_kwargs,
                extra_comment_kwargs=extra_comment_kwargs,
                # Только запись 'ses_create' может иметь значение отличное от 'web'
                auth_type=auth_type,
            )
            log_auth_event_for_antifraud(
                env=env,
                uid=uid,
                retpath=retpath,
                track=track,
                authid=authid,
                login_id=new_login_id,
            )

    if log_statbox:
        ip_country = Region(ip=env.user_ip).country
        old_session_uids = ','.join(map(str, multi_session_users)) if multi_session_users else None

        statbox = StatboxLogger(**{
            'uid': uid,
            'authid': authid,
            'source_authid': source_authid,
            'input_login': login or '',
            'mode': 'any_auth',
            'action': 'cookie_set',
            'auth_source': auth_source,
            'cookie_version': session_version,
            'ttl': ttl,
            'from': track.service if track else None,
            'yandexuid': env.cookies.get('yandexuid'),
            'track_id': track.track_id if track else None,
            'origin': track.origin if track else None,
            'ip': env.user_ip,
            'user_agent': env.user_agent,
            'person_country': track.country if track and track.country else None,
            'ip_country': ip_country['short_en_name'].lower() if ip_country else None,
            'captcha_passed': bool(captcha_passed),
            'bruteforce': track.bruteforce_status if track else None,
            'session_method': 'edit' if extend_session else 'create',
            'uids_count': uids_count,
            'old_session_uids': old_session_uids,
            'retpath': retpath,
            'is_2fa_enabled': is_2fa_enabled or None,
        })

        # Если пользователю был показан challenge и он дошел сюда,
        # то зафиксируем этот шаг в логах
        if track and track.is_auth_challenge_shown:
            statbox.bind(is_auth_challenge_shown=track.is_auth_challenge_shown)

        statbox.log()

    if track and track.surface is not None:
        credentials_logger = CredentialsLogger(**{
            'yandexuid': env.cookies.get('yandexuid'),
            'uid': uid,
            'ip': env.user_ip,
            'user_agent': env.user_agent,
            'login': login,
            'track_id': track.track_id,
            'surface': track.surface,
            'deviceid': track.device_id,
            'clientid': track.client_id,
            'auth_id': authid,
            'uids_count': uids_count,
            'is_new': not extend_session,
            'region_id': Region(ip=env.user_ip).id,
            'credential_type': 'cookie',
        })
        credentials_logger.log()

    session, service_guard_container = bb_response_to_session(
        bb_response=response,
        env=env,
        extra_sessguard_host=extra_sessguard_host,
        retpath=retpath,
    )

    log_new_session(session, service_guard_container)
    return session, service_guard_container


def authorize_oauth(client_id, client_secret, env, uid=None, track=None,
                    logout_datetime=None, login_id=None, **kwargs):
    check_track_for_logout(track, logout_datetime)
    oauth = get_oauth()
    if login_id is None and track is not None:
        login_id = track.login_id
    return oauth.token_by_uid(
        client_id=client_id,
        client_secret=client_secret,
        uid=uid or track.uid,
        user_ip=env.user_ip,
        login_id=login_id,
        passport_track_id=track.track_id if track else None,
        **kwargs
    )


def session_to_cookie(key, session, domain=None, path='/', samesite=None):
    return dump_cookie(
        key,
        session['value'],
        expires=session['expires'],
        domain=domain or session['domain'],
        httponly=True,
        secure=True,
        path=path,
        samesite=samesite,
    )


def cookie_l(l_value, domain, path='/'):
    return dump_cookie(
        'L',
        l_value,
        expires=MAX_COOKIE_EXPIRATION_TIMESTAMP,
        domain=domain,
        path=path,
    )


def cookie_yp(yp_value, domain, path='/'):
    return dump_cookie(
        'yp',
        yp_value,
        expires=MAX_COOKIE_EXPIRATION_TIMESTAMP,
        domain=domain,
        path=path,
        secure=True,
    )


def cookie_ys(ys_value, domain, path='/', samesite=None):
    return dump_cookie(
        'ys',
        ys_value,
        domain=domain,
        path=path,
        samesite=samesite,
        secure=True,
    )


def cookie_my(my_value, domain, path='/', samesite=None):
    return dump_cookie(
        'my',
        my_value,
        expires=MAX_COOKIE_EXPIRATION_TIMESTAMP,
        domain=domain,
        path=path,
        samesite=samesite,
        secure=True,
    )


def cookie_yandex_gid(yandex_gid_value, domain, path='/', samesite=None):
    return dump_cookie(
        'yandex_gid',
        yandex_gid_value,
        expires=time.time() + COOKIE_YANDEX_GID_MAX_AGE,
        domain=domain,
        path=path,
        samesite=samesite,
        secure=True,
    )


def cookie_yandexuid(yandexuid_value, domain, path='/', samesite=None):
    return dump_cookie(
        'yandexuid',
        yandexuid_value,
        expires=time.time() + settings.COOKIE_YANDEXUID_MAX_AGE,
        domain=domain,
        path=path,
        secure=True,
        samesite=samesite,
    )


def cookie_i(i_value, domain, path='/', samesite=None):
    return dump_cookie(
        'i',
        str(i_value),
        expires=time.time() + settings.COOKIE_I_MAX_AGE,
        domain=domain,
        path=path,
        secure=True,
        httponly=True,
        samesite=samesite,
    )


def cookie_yandex_login(yandex_login_value, domain, path='/', samesite=None):
    yandex_login_value = smart_text(yandex_login_value)
    if six.PY2:
        yandex_login_value = yandex_login_value.encode('utf8')
    login_part, at, domain_part = yandex_login_value.partition('@')
    quoted_yandex_login_value = ''.join([quote(login_part), at, quote(domain_part)])
    return dump_cookie(
        'yandex_login',
        quoted_yandex_login_value,
        max_age=settings.COOKIE_YANDEX_LOGIN_MAX_AGE,
        domain=domain,
        path=path,
        secure=True,
        samesite=samesite,
        sync_expires=False,
    )


def cookie_lah(lah_value, domain, path='/', samesite=None):
    return dump_cookie(
        'lah',
        lah_value,
        expires=MAX_COOKIE_EXPIRATION_TIMESTAMP if lah_value else 1,
        domain=domain,
        path=path,
        secure=True,
        httponly=True,
        samesite=samesite,
    )


def cookie_ilahu(ilahu_value, domain, path='/', samesite=None):
    return dump_cookie(
        'ilahu',
        ilahu_value,
        expires=time.time() + settings.COOKIE_ILAHU_MAX_AGE,
        domain=domain,
        path=path,
        secure=True,
        httponly=True,
        samesite=samesite,
    )


def cookie_mda2_beacon(value, domain, path='/', expires=MAX_COOKIE_EXPIRATION_TIMESTAMP, samesite=None):
    return dump_cookie(
        'mda2_beacon',
        str(value),
        expires=expires,
        domain=domain,
        path=path,
        secure=True,
        httponly=False,
        samesite=samesite,
    )


def cookie_mda2_domains(value, domain, path='/'):
    return dump_cookie(
        'mda2_domains',
        value,
        expires=MAX_COOKIE_EXPIRATION_TIMESTAMP,
        domain=domain,
        path=path,
        secure=True,
        httponly=False,
    )


def cookie_wcid(value, domain, expires=None, path='/'):
    return dump_cookie(
        'wcid',
        value,
        expires=expires or MAX_COOKIE_EXPIRATION_TIMESTAMP,
        domain=domain,
        path=path,
        secure=True,
        httponly=True,
    )


def build_auth_cookies(env, session):
    samesite = get_samesite_policy(env)
    cookies = []
    if 'session' in session:
        cookies.append(session_to_cookie('Session_id', session['session'], path='/', samesite=samesite))
    if 'sslsession' in session:
        cookies.append(session_to_cookie('sessionid2', session['sslsession'], path='/', samesite=samesite))
    if 'sessguard' in session:
        cookies.append(session_to_cookie('sessguard', session['sessguard'], path='/', samesite=samesite))
    return cookies


def build_cookie_sessionid_noauth(env, path='/'):
    domain = '.' + get_keyspace_by_host(env.host)
    samesite = get_samesite_policy(env)
    return dump_cookie(
        'Session_id',
        'noauth:%s' % get_unixtime(),
        max_age=settings.COOKIE_SESSIONID_NOAUTH_MAX_AGE,
        sync_expires=False,
        domain=domain,
        httponly=True,
        secure=True,
        path=path,
        samesite=samesite,
    )


def build_cookies_noauth(env):
    return [
        build_cookie_sessionid_noauth(env, path='/'),
        build_cookie_yandex_login(
            env,
            human_readable_login=None,
            path='/',
        ),
    ]


def build_cookie_l(env, uid, readable_login, path='/'):
    domain = '.' + get_keyspace_by_host(env.host)
    l_value = CookieL().pack(uid, readable_login)
    return cookie_l(l_value, domain, path)


def build_cookie_yp(_env, current_yp_value, display_name, domain, path='/', is_multibrowser=False, is_2fa_enabled_yp=None, is_child=False):
    yp_container = PermanentCookieYContainer()
    yp_container.parse(current_yp_value)
    if display_name is not Undefined:
        if display_name is not None:
            yp_container.insert(
                settings.COOKIE_YX_DISPLAY_NAME_FIELD,
                display_name,
                int(time.time() + settings.COOKIE_YP_DISPLAY_NAME_AGE),
            )
        else:
            yp_container.erase(settings.COOKIE_YX_DISPLAY_NAME_FIELD)
    if is_multibrowser and not yp_container.get(settings.COOKIE_YP_MULTIBROWSER_FIELD).get('value'):
        yp_container.insert(
            settings.COOKIE_YP_MULTIBROWSER_FIELD,
            settings.COOKIE_YP_MULTIBROWSER_IS_SET,
            int(time.time() + settings.COOKIE_YP_MULTIBROWSER_AGE),
        )

    is_2fa_enabled_in_cookie = yp_container.get(settings.COOKIE_YP_2FA_FIELD).get('value')

    if is_2fa_enabled_yp is not None:
        if is_2fa_enabled_yp:
            if not is_2fa_enabled_in_cookie:
                yp_container.insert(
                    settings.COOKIE_YP_2FA_FIELD,
                    settings.COOKIE_YP_2FA_IS_SET,
                    int(time.time() + settings.COOKIE_YP_2FA_AGE),
                )
        else:
            if is_2fa_enabled_in_cookie:
                yp_container.erase(settings.COOKIE_YP_2FA_FIELD)

    if is_child:
        sp_subcookie_value = yp_container.get(settings.COOKIE_YP_SP_FIELD).get('value')
        sp_subcookie_container = YpSpSubvalueContainer()
        sp_subcookie_container.parse(sp_subcookie_value)
        if sp_subcookie_container.get('family').get('value') != settings.COOKIE_YP_SP_FIELD_FAMILY_SUBFIELD_VALUE:
            sp_subcookie_container.insert('family', settings.COOKIE_YP_SP_FIELD_FAMILY_SUBFIELD_VALUE)

            yp_container.insert(
                settings.COOKIE_YP_SP_FIELD,
                sp_subcookie_container.serialize(),
                int(time.time() + settings.COOKIE_YP_SP_AGE),
            )

    yp_value = yp_container.serialize()
    return cookie_yp(yp_value, domain, path)


def build_cookie_ys(env, current_ys_value, display_name, domain, path='/', cookie_check_value=None, samesite=None):
    if samesite is None:
        samesite = get_samesite_policy(env)

    ys_container = SessionCookieYContainer()
    ys_container.parse(current_ys_value)
    if display_name is not Undefined:
        if display_name is not None:
            ys_container.insert(
                settings.COOKIE_YX_DISPLAY_NAME_FIELD,
                display_name,
            )
        else:
            ys_container.erase(settings.COOKIE_YX_DISPLAY_NAME_FIELD)

    if cookie_check_value is not None:
        ys_container.insert(
            settings.COOKIE_YS_COOKIE_CHECK_FIELD,
            str(cookie_check_value),
        )

    ys_value = ys_container.serialize()
    return cookie_ys(ys_value, domain, path, samesite=samesite)


def build_cookies_yx(env, display_name=Undefined, path='/', is_multibrowser=False, is_2fa_enabled_yp=None,
                     cookie_check_value=None, need_yp=True, need_ys=True, is_child=False):
    domain = '.' + get_keyspace_by_host(env.host)

    if display_name is Undefined:
        udn = Undefined
    else:
        udn = encode_udn(display_name)

    current_ys_value = env.cookies.get('ys', '')
    current_yp_value = env.cookies.get('yp', '')

    result = []
    if need_yp:
        result.append(
            build_cookie_yp(env, current_yp_value, udn, domain, path, is_multibrowser, is_2fa_enabled_yp, is_child),
        )
    if need_ys:
        result.append(
            build_cookie_ys(env, current_ys_value, udn, domain, path, cookie_check_value),
        )
    return result


def build_cookie_yandexuid(env, yandexuid_value=None, path='/'):
    domain = '.' + get_keyspace_by_host(env.host)
    yandexuid_value = str(yandexuid_value or env.cookies.get('yandexuid', ''))
    samesite = get_samesite_policy(env)
    return cookie_yandexuid(
        yandexuid_value,
        domain,
        path,
        samesite=samesite,
    )


def build_cookie_i(env, i_value=None, path='/'):
    domain = '.' + get_keyspace_by_host(env.host)
    i_value = i_value or env.cookies.get('i', '')
    samesite = get_samesite_policy(env)
    return cookie_i(
        i_value,
        domain,
        path,
        samesite=samesite,
    )


def build_cookie_yandex_login(env, human_readable_login, path='/'):
    domain = '.' + get_keyspace_by_host(env.host)
    samesite = get_samesite_policy(env)
    return cookie_yandex_login(
        human_readable_login or '',
        domain,
        path,
        samesite=samesite,
    )


def build_cookie_lah(env, auth_info_container, path='/'):
    domain = '.' + build_passport_domain(env.host)
    value = CookieLAH().pack(container=auth_info_container)
    samesite = get_samesite_policy(env)
    return cookie_lah(
        value,
        domain,
        path,
        samesite=samesite,
    )


def build_cookie_ilahu(env, value, path='/'):
    domain = '.' + build_passport_domain(env.host)
    samesite = get_samesite_policy(env)
    return cookie_ilahu(
        str(value),
        domain,
        path,
        samesite=samesite,
    )


def generate_cookie_mda2_beacon_value():
    return int(time.time() * 1000)


def build_cookie_mda2_beacon(env, value=None, path='/', domain=None, samesite=None, **kwargs):
    if domain is None:
        domain = '.' + build_passport_domain(env.host)
    if value is None:
        value = generate_cookie_mda2_beacon_value()
    if samesite is None:
        samesite = get_samesite_policy(env)
    return cookie_mda2_beacon(
        value=value,
        domain=domain,
        path=path,
        samesite=samesite,
        **kwargs
    )


def build_cookie_mda2_domains(env, domains_list, path='/'):
    domain = '.' + build_passport_domain(env.host)
    return cookie_mda2_domains(
        ','.join(sorted(set(domains_list))),
        domain,
        path,
    )


def build_cookie_wcid(env, webauthn_credential_external_id, expires=None, path='/'):
    domain = '.' + build_passport_domain(env.host)
    return cookie_wcid(
        webauthn_credential_external_id,
        domain,
        expires=expires,
        path=path,
    )


def build_cookie_yandex_gid(env, yandex_gid_value=None, path='/'):
    return cookie_yandex_gid(
        yandex_gid_value=yandex_gid_value or env.cookies.get(YANDEX_GID_COOKIE_NAME, ''),
        domain='.' + get_keyspace_by_host(env.host),
        path=path,
        samesite=get_samesite_policy(env),
    )


def build_cookie_my(env, my_value=None, path='/'):
    return cookie_my(
        my_value=my_value or env.cookies.get(MY_COOKIE_NAME, ''),
        domain='.' + get_keyspace_by_host(env.host),
        path=path,
        samesite=get_samesite_policy(env),
    )


def try_update_cookie_lah(env, uid, auth_method, authorization_session_policy, input_login=None, cookie_path='/'):
    auth_history_container = try_parse_cookie_lah(env.cookies.get('lah'))
    if (
        uid is not None and
        authorization_session_policy != AUTHORIZATION_SESSION_POLICY_SESSIONAL and
        # Не сохраняем логины школьников
        not (input_login and login_is_scholar(input_login))
    ):
        auth_method = auth_method or settings.AUTH_METHOD_UNKNOWN
        auth_history_container.add(
            uid=uid,
            timestamp=time.time(),
            method=settings.AUTH_METHOD_TO_ID[auth_method],
            input_login=input_login,
        )

    # Конечно, хорошо бы уже тут вычистить удалённых, но обойдёмся без тяжёлых запросов на авторизации
    if auth_history_container:
        return build_cookie_lah(
            env,
            auth_history_container,
            path=cookie_path,
        )


def update_cookie_ilahu(env, ignore_lah_for, cookie_path='/'):
    current_value = env.cookies.get('ilahu')
    if current_value and current_value.isdigit():
        current_value = int(current_value)
    else:
        current_value = 0

    new_value = max(current_value, int(time.time() + ignore_lah_for))

    return build_cookie_ilahu(
        env,
        new_value,
        path=cookie_path,
    )


def update_cookie_mda2_domains(env, new_domain, cookie_path='/'):
    current_value = env.cookies.get('mda2_domains')
    if current_value:
        domains = current_value.split(',')
    else:
        domains = []

    domains.append(get_keyspace_by_host(new_domain))

    return build_cookie_mda2_domains(
        env,
        domains,
        path=cookie_path,
    )


def build_non_auth_cookies(env, uid, human_readable_login, machine_readable_login,
                           cookie_path, ignore_cookie_l=False,
                           authorization_session_policy=AUTHORIZATION_SESSION_POLICY_PERMANENT,
                           display_name=None, is_2fa_enabled_yp=None, auth_method=None, input_login=None,
                           ignore_lah_for=None, is_child=False):
    cookies = []

    is_multibrowser = False
    try:
        old_cookie_l_value = env.cookies.get('L')
        if old_cookie_l_value:
            old_login = CookieL().unpack(old_cookie_l_value).get('login')
            # Флаг мультилогиновости браузера. Считаем что браузер мультилогиновый, если:
            # логин в куке L (который может быть либо человекочитаемым, либо машиночитаемым)
            # не совпадает с текущим авторизующимся логином
            is_multibrowser = old_login != human_readable_login and old_login != machine_readable_login
    except CookieLUnpackError:
        pass  # Если не смогли распарсить куку, мешать авторизации не нужно!

    yx_cookies = build_cookies_yx(
        env,
        display_name,
        path='/',
        is_multibrowser=is_multibrowser,
        is_2fa_enabled_yp=is_2fa_enabled_yp,
        is_child=is_child,
    )
    cookies.extend(yx_cookies)

    # Запишем в статбокс о мультибраузерности
    yp_container = PermanentCookieYContainer()
    yp_container.parse(env.cookies.get('yp', ''))
    current_yp_is_multibrowser = yp_container.get(settings.COOKIE_YP_MULTIBROWSER_FIELD).get('value', False)
    current_yp_is_multibrowser = current_yp_is_multibrowser == settings.COOKIE_YP_MULTIBROWSER_IS_SET
    if is_multibrowser or current_yp_is_multibrowser:
        StatboxLogger().log(**{
            'mode': 'any_auth',
            'action': 'multibrowser_update',
            'yandexuid': env.cookies.get('yandexuid'),
            'ip': env.user_ip,
            'uid': uid,
            'user_agent': env.user_agent,
            'old_multibrowser': current_yp_is_multibrowser,
            'new_multibrowser': is_multibrowser,
        })

    if not ignore_cookie_l:
        cookie_l = build_cookie_l(
            env,
            uid,
            human_readable_login,
            path='/',
        )
        cookies.append(cookie_l)

    cookie_yandex_login = build_cookie_yandex_login(
        env,
        human_readable_login,
        path=cookie_path,
    )
    cookies.append(cookie_yandex_login)

    cookie_lah = try_update_cookie_lah(
        env=env,
        uid=uid,
        auth_method=auth_method,
        authorization_session_policy=authorization_session_policy,
        input_login=input_login,
        cookie_path=cookie_path,
    )
    if cookie_lah:
        cookies.append(cookie_lah)

    if ignore_lah_for is not None:
        cookie_ilahu = update_cookie_ilahu(
            env=env,
            ignore_lah_for=ignore_lah_for,
            cookie_path=cookie_path,
        )
        cookies.append(cookie_ilahu)

    return cookies


def build_non_auth_cookies_from_track(env, track):
    domain = '.' + get_keyspace_by_host(env.host)
    cookies = []
    for builder, track_field in (
        (cookie_l, 'cookie_l_value'),
        (cookie_my, 'cookie_my_value'),
        (cookie_yp, 'cookie_yp_value'),
        (cookie_ys, 'cookie_ys_value'),
        (cookie_yandex_gid, 'cookie_yandex_gid_value'),
    ):
        cookie_value = getattr(track, track_field)
        if cookie_value:
            cookies.append(
                builder(cookie_value, domain),
            )

    if track.cookie_yandex_login_value:
        cookies.append(
            cookie_yandex_login(track.cookie_yandex_login_value, domain),
        )

    return cookies


def build_auth_cookies_and_session(
    env,
    track,
    account_type,
    authorization_session_policy=AUTHORIZATION_SESSION_POLICY_PERMANENT,
    ignore_non_auth_cookies=False,
    session_version=settings.BLACKBOX_SESSION_VERSION,
    is_yandexoid=False,
    is_betatester=False,
    social_id=None,
    extend_session=False,
    multi_session_users=None,
    auth_type=None,
    is_lite=None,
    display_name=None,
    logout_datetime=None,
    otp_magic_passed=None,
    is_2fa_enabled=False,
    is_2fa_enabled_yp=None,
    x_token_magic_passed=False,
    need_extra_sessguard=False,
    session_scope=None,
    is_child=False,
):
    """Авторизует пользователя в ЧЯ и подготавливает для него куки
    * Сессионная кука
    * Сессионная защищенная кука, если ЧЯ вернул ее
    * Кука ED, если это пользователь ПДД
    * Куки контейнеры YP, YS обновляются с учетом уже установленных значений
    * Кука L
    * Кука yandex_login
    """
    cookies = []

    if is_lite is None:
        is_lite = account_type == ACCOUNT_TYPE_LITE

    if session_scope is None:
        session_scope = track.session_scope

    if extend_session:
        # При добавлении аккаунта в мультикуку её ttl не меняется
        old_ttl = int(track.old_session_ttl) if track.old_session_ttl else settings.SESSIONAL_COOKIE_TTL
        authorization_session_policy = get_session_policy_by_ttl(old_ttl)

    session, service_guard_container = authorize(
        env,
        track.uid,
        track.user_entered_login or track.login,
        track.language,
        auth_challenge=track.auth_challenge_type,
        auth_source=track.auth_source,
        auth_type=auth_type,
        authorization_session_policy=authorization_session_policy,
        captcha_passed=track.is_captcha_recognized if track.is_captcha_checked else None,
        dont_change_default_uid=track.dont_change_default_uid,
        extend_session=extend_session,
        have_password=track.have_password,
        is_2fa_enabled=is_2fa_enabled,
        is_betatester=is_betatester,
        is_lite=is_lite,
        is_session_restricted=track.is_session_restricted,
        is_yandexoid=is_yandexoid,
        log_historydb=not track.use_non_auth_cookies_from_track,
        login_id=track.login_id,
        logout_datetime=logout_datetime,
        multi_session_users=multi_session_users,
        need_extra_sessguard=need_extra_sessguard,
        old_sessguard=track.old_sessguard,
        old_session=track.old_session,
        old_session_ttl=track.old_session_ttl,
        old_ssl_session=track.old_ssl_session,
        otp_magic_passed=otp_magic_passed,
        password_passed=track.is_password_passed,
        retpath=track.retpath,
        session_scope=session_scope,
        session_version=session_version,
        social_id=social_id,
        source_authid=track.source_authid,
        track=track,
        x_token_magic_passed=x_token_magic_passed,
    )

    # Expires yandex_login совпадает с expires Session_id
    session_expires = session['session']['expires']

    cookies.extend(build_auth_cookies(env, session))

    track.session = session['session']['value']
    if 'sslsession' in session:
        track.sslsession = session['sslsession']['value']
    if 'sessguard' in session:
        track.sessguard = session['sessguard']['value']
    track.session_created_at = track.session_created_at or get_unixtime()

    if not ignore_non_auth_cookies:
        if track.use_non_auth_cookies_from_track:
            cookies.extend(
                build_non_auth_cookies_from_track(
                    env,
                    track,
                ),
            )
        else:
            cookies.extend(
                build_non_auth_cookies(
                    env,
                    track.uid,
                    track.human_readable_login,
                    track.machine_readable_login,
                    '/',
                    authorization_session_policy=authorization_session_policy,
                    display_name=display_name,
                    is_2fa_enabled_yp=is_2fa_enabled_yp,
                    auth_method=getattr(track, 'auth_method', None),
                    input_login=track.user_entered_login,
                    is_child=is_child,
                ),
            )

    # Ставим куку mda2_beacon на срок жизни сессии
    cookies.append(build_cookie_mda2_beacon(env, expires=session_expires))

    return cookies, session, service_guard_container


def set_old_session_track_fields(track, old_session_info):
    track.old_session = old_session_info.session
    track.old_ssl_session = old_session_info.ssl_session
    track.old_sessguard = old_session_info.sessguard
    track.old_session_ttl = old_session_info.ttl
    track.old_session_create_timestamp = old_session_info.create_timestamp
    track.old_session_ip = old_session_info.ip
    track.old_session_age = old_session_info.age


def set_authorization_track_fields(account,
                                   track,
                                   allow_create_session,
                                   allow_create_token,
                                   password_passed,
                                   is_2fa_restore_passed=False,
                                   old_session_info=None,
                                   auth_source=None,
                                   source_authid=None,
                                   session_scope=None):
    """
    Выставляет необходимые поля в трэк для дальнейшей авторизации пользователя
    @param allow_create_session: разрешает создавать cookie
    @type allow_create_session: bool
    @param allow_create_token: разрешает создавать oauth токен
    @param password_passed: парольная авторизация или нет
    @param is_2fa_restore_passed: признак того, что было пройдено восстановление пользователя с 2ФА
    @param old_session_info: записать информацию о старой сессии
    @param auth_source: дополнительная причина, по которой происходит авторизация
    @param source_authid: ID куки на другом устройстве, послужившей источником полномочий для авторизации
    @type password_passed: bool
    @type allow_create_token: bool
    @type session_scope: SessionScope
    """
    track.allow_authorization = allow_create_session
    track.allow_oauth_authorization = allow_create_token
    track.uid = account.uid
    track.login = account.login or None
    track.human_readable_login = account.human_readable_login or None
    track.machine_readable_login = account.machine_readable_login or None

    track.is_password_passed = password_passed
    track.have_password = account.have_password
    track.is_otp_restore_passed = is_2fa_restore_passed
    if not track.language and account.person.language:
        track.language = account.person.language

    if not track.country and account.person.country:
        track.country = account.person.country

    if old_session_info:
        set_old_session_track_fields(track, old_session_info)

    if account.is_pdd:
        track.domain = account.domain.domain

    if auth_source:
        track.auth_source = auth_source
    if source_authid:
        track.source_authid = source_authid

    if session_scope:
        track.session_scope = str(session_scope)


def encode_udn(display_name, max_length=None):
    if display_name:
        if max_length is None:
            max_length = settings.COOKIE_YX_DISPLAY_NAME_LENGTH
        udn = smart_text(display_name)
        udn = udn[:max_length]
        if six.PY2:
            encoder = base64.encodestring
        else:
            encoder = base64.encodebytes
        udn = encoder(udn.encode('utf-8')).replace(b'\n', b'')
    else:
        udn = None
    return udn


def login_password_ok(login_status, password_status):
    return (
        login_status == blackbox.BLACKBOX_LOGIN_VALID_STATUS and
        password_status == blackbox.BLACKBOX_PASSWORD_VALID_STATUS
    )


def is_oauth_token_created(track):
    if track.oauth_token_created_at:
        # Игнорируем недавно случившиеся выдачи (чтобы работали ретраи получения токенов)
        return get_unixtime() - string_to_integer_unixtime(track.oauth_token_created_at) >= settings.TOKEN_REISSUE_INTERVAL
    return track.is_oauth_token_created


def is_session_created(track, session_reissue_interval=None):
    if track.session_created_at:
        # Игнорируем недавно случившиеся выдачи (чтобы работали ретраи получения кук)
        time_diff = get_unixtime() - string_to_integer_unixtime(track.session_created_at)
        if session_reissue_interval:
            return time_diff >= session_reissue_interval
        return time_diff >= settings.SESSION_REISSUE_INTERVAL
    return track.session is not None


__all__ = (
    'AUTHORIZATION_SESSION_POLICY_PERMANENT',
    'AUTHORIZATION_SESSION_POLICY_SESSIONAL',
    'AUTHORIZATION_SESSION_POLICY_SHORT',
    'to_authlog',
    'get_session_policy_by_ttl',
    'get_keyspace_by_host',
    'bb_response_to_session',
    'authorize',
    'authorize_oauth',
    'set_authorization_track_fields',
    'set_old_session_track_fields',
    'build_cookies_noauth',
    'build_auth_cookies',
    'build_cookie_i',
    'build_cookie_l',
    'build_cookie_lah',
    'build_cookie_ilahu',
    'build_cookie_mda2_beacon',
    'build_cookie_mda2_domains',
    'build_cookie_wcid',
    'build_cookie_yandex_login',
    'build_cookie_yandexuid',
    'build_cookies_yx',
    'build_cookie_ys',
    'build_cookie_yp',
    'build_auth_cookies_and_session',
    'log_auth_challenge_shown',
    'log_auth_event',
    'log_auth_event_for_antifraud',
    'log_web_auth_updates',
    'log_new_session',
    'is_session_valid',
    'is_user_session_valid',
    'build_non_auth_cookies',
    'build_non_auth_cookies_from_track',
    'user_from_multisession',
    'users_from_multisession',
    'is_user_in_multisession',
    'is_user_default_in_multisession',
    'is_user_valid_in_multisession',
    'login_password_ok',
    'is_oauth_token_created',
    'is_session_created',
    'update_cookie_mda2_domains',
    'try_update_cookie_lah',
    'user_session_scope',
)
