# -*- coding: utf-8 -*-
from collections import namedtuple
from functools import wraps
import logging
from os import urandom
import time

from flask import request
from passport.backend.api import env
from passport.backend.api.common.account_manager import (
    DEVICE_PARAM_NAME_TO_TRACK_FIELD,
    device_params_to_track_fields,
)
import passport.backend.api.validators
from passport.backend.core import frodo
from passport.backend.core.builders.blackbox.blackbox import get_blackbox
from passport.backend.core.builders.blackbox.exceptions import BaseBlackboxError
from passport.backend.core.builders.clean_web_api import (
    BaseCleanWebError,
    get_clean_web_api,
)
from passport.backend.core.conf import settings
from passport.backend.core.exceptions import UnknownUid
from passport.backend.core.models.account import Account
from passport.backend.core.tracks.track_manager import TrackManager
from passport.backend.core.utils.decorators import cached_property
from passport.backend.core.validators import (
    Invalid,
    State,
)
from passport.backend.utils.common import bytes_to_hex
from six import iteritems
from six.moves.urllib.parse import (
    quote,
    unquote,
)


ESCAPABLE_URL_CHARACTERS = {c: quote(c) for c in {'\n', '\r', ' '}}

log = logging.getLogger('passport.backend.api.common')


def chained_unquote(url, attempts=5):
    for _ in range(attempts):
        new_url = unquote(url)
        if url == new_url:
            return url
        url = new_url
    return url


def try_reload_dynamic_settings():
    settings.try_reload_dynamic_settings()


def prepare_env():
    # Сохраняем текущее время в качестве времени начала выполнения запроса
    request.start_time = time.time()
    request.env = env.APIEnvironment.from_request(request)


def increment_track_counter(track_id, counter_getter):
    if track_id is None:
        return
    with TrackManager().transaction(track_id).rollback_on_error() as track:
        counter_getter(track).incr()


def parse_args_for_track(args, exclude=None):
    exclude_set = {'consumer'}

    for form_name, track_name in iteritems(DEVICE_PARAM_NAME_TO_TRACK_FIELD):
        if form_name != track_name:
            exclude_set.add(form_name)

    if exclude:
        exclude_set.update(exclude)

    def arg_ok(arg):
        return arg not in exclude_set and args[arg] is not None
    device_info = device_params_to_track_fields(args)
    args = dict(args)
    args.update(device_info)
    return dict((arg, args[arg]) for arg in args if arg_ok(arg))


def track_to_response(track, field_list):
    """Возвращает словарь из указанных полей трека. Пропускает отсутствующие значения."""
    response = {}
    for field_name in field_list:
        value = getattr(track, field_name, None)
        if value is not None:
            response[field_name] = value
    return response


def check_spammer(account, env, frodo_args, track, consumer):
    frodo_args.update(
        consumer=consumer,
        uid=account.uid,
        account_country=account.person.country or '',
        account_language=account.person.language or '',
        account_timezone=account.person.timezone or '',
        language=account.person.language or '',
        country=account.person.country or '',
        phone_bindings_count='0',
    )

    phone_number = frodo.FrodoInfo.get_phone_number(env, frodo_args, track)
    if phone_number is not None:
        current_bindings = _get_current_phone_bindings_failsafe(phone_number)
        frodo_args.update(phone_bindings_count=str(len(current_bindings)))

    frodo_info = frodo.FrodoInfo.create(env, frodo_args, track)
    account, frodo_status = frodo.check_spammer(frodo_info, env, consumer, account)
    return account, frodo_status


def _get_current_phone_bindings_failsafe(phone_number):
    try:
        return get_blackbox().phone_bindings(
            need_current=True,
            need_history=False,
            need_unbound=False,
            phone_numbers=[phone_number.e164],
            should_ignore_binding_limit=False,
        )
    except BaseBlackboxError:
        log.error('Getting phone bindings for Frodo failed', exc_info=True)
        return []


def get_email_domain(host):
    # Подходит для регистрации, нет специальной логики по сидам (например, для галатасарая)
    for domain, email_domain in settings.DEFAULT_EMAIL_DOMAINS:
        if host == domain or host.endswith('.' + domain):
            return email_domain
    else:
        return settings.DEFAULT_EMAIL_DOMAIN


def validate_password(args, track=None):
    """
    Базовая валидация пароля, не использующая личную информацию из трека (email-ы, телефон, предыдущие пароли).
    """
    validator_args = dict(args)

    if track:
        validator_args['track_id'] = track.track_id
        validator_args['uid'] = track.uid
        if validator_args.get('login') is None:
            validator_args['login'] = track.login
        if track.is_strong_password_policy_required:
            validator_args['policy'] = 'strong'

    p = passport.backend.api.validators.Password()
    validated = p.to_python(validator_args, State(request.env))
    args['quality'] = validated['quality']
    return validated, p


def extract_tld(host, host_tlds):
    for tld in host_tlds:
        if host.endswith('.' + tld):
            return tld


def get_logout_datetime(uid):
    """
    Функция для получения времени разлогина (глобального или отзыва веб-сессий) по уиду.
    """
    try:
        data = get_blackbox().userinfo(uid, dbfields=[])
        account = Account().parse(data)
        return account.web_sessions_logout_datetime
    except UnknownUid:
        return  # После регистрации данные могут не успеть дойти до реплики


def escape_query_characters_middleware(wsgi_app):
    """
    Экранирует символы в HTTP-Query.
    """
    @wraps(wsgi_app)
    def wrapper(environ, start_response):
        qs = environ.get('QUERY_STRING')
        if qs:
            new_qs = qs
            for esc_ch, repl in ESCAPABLE_URL_CHARACTERS.items():
                new_qs = new_qs.replace(esc_ch, repl)
            if new_qs != qs:
                log.debug(u'Escape CR/LF characters in query')
            environ['QUERY_STRING'] = new_qs
        return wsgi_app(environ, start_response)
    return wrapper


def create_csrf_token():
    return bytes_to_hex(urandom(settings.CSRF_TOKEN_BYTES_COUNT))


def looks_like_yandex_email(email):
    if '@' not in email:
        return False
    login_part, domain_part = email.rsplit('@', 1)
    return domain_part.strip() in settings.NATIVE_EMAIL_DOMAINS


class CleanWebChecker(object):
    """
    Работа с Clean Web API
    """
    @cached_property
    def _clean_web_api(self):
        return get_clean_web_api()

    def check_user_data(self, first_name=None, last_name=None, full_name=None, display_name=None, public_id=None):
        """
        Возвращает
        массив полей в которых клинвеб нашел нежелательный контент
        None если при запросе случилась ошибка
        True если clean web не нашел нежелательный контент
        """
        if not first_name and not last_name and not full_name and not display_name and not public_id:
            return True
        try:
            return self._clean_web_api.check_user_data(
                key='FAKE_KEY-%s' % request.env.request_id,
                first_name=first_name,
                last_name=last_name,
                full_name=full_name,
                display_name=display_name,
                public_id=public_id,
                simple_response=True,
            )
        except BaseCleanWebError:
            return None

    @classmethod
    def _make_exception(cls, fields):
        state = State(request.env)
        error_dict = {
            field: Invalid(('invalid', 'Invalid value'), fields[field], state)
            for field in fields
        }
        return Invalid(
            '\n'.join(["%s: %s" % (k, str(v)) for k, v in sorted(error_dict.items())]),
            fields,
            state,
            error_dict=error_dict,
        )

    def check_form_values(
        self,
        form_values,
        error_class=lambda fields: CleanWebChecker._make_exception(fields),
        statbox=None,
    ):
        if not (settings.CLEAN_WEB_API_ENABLED or form_values.get('force_clean_web', False)):
            return

        Text = namedtuple('Text', ['value', 'field_names_for_error', 'field_name_for_statbox', 'field_name_for_cw'])
        texts = []

        has_firstname = bool(form_values.get('firstname'))
        has_lastname = bool(form_values.get('lastname'))
        if has_firstname:
            texts.append(Text(form_values['firstname'], ['firstname'], 'firstname', 'first_name'))
        if has_lastname:
            texts.append(Text(form_values['lastname'], ['lastname'], 'lastname', 'last_name'))
        if has_firstname and has_lastname:
            texts.append(Text(u'{} {}'.format(form_values['firstname'], form_values['lastname']), ['firstname', 'lastname'], 'fullname', 'full_name'))
        if 'display_name' in form_values and form_values['display_name']:
            texts.append(Text(form_values['display_name'].public_name, ['display_name'], 'display_name', 'display_name'))
        if bool(form_values.get('public_id')):
            texts.append(Text(form_values['public_id'], ['public_id'], 'public_id', 'public_id'))
        if len(texts) == 0:
            return
        if statbox:
            statbox.bind(
                clean_web_check_fields=','.join([t.field_name_for_statbox for t in texts]),
            )
        clean_web_response = self.check_user_data(**{t.field_name_for_cw: t.value for t in texts})
        if statbox:
            statbox.bind(
                clean_web_response=','.join(clean_web_response) if type(clean_web_response) == list else str(clean_web_response),
            )
        if clean_web_response is not None:
            if type(clean_web_response) == list:
                if statbox:
                    statbox.log(
                        action='clean_web_check_failed',
                    )
                errors = {}
                cw_error_fields = set(clean_web_response)
                # была проверка имени и фамилии вместе
                if has_firstname and has_lastname:
                    # чистый веб ругнулся на имя и фамилию вместе
                    if 'full_name' in cw_error_fields:
                        # ругается ли отдельно на имя или фамилию?
                        if 'first_name' in cw_error_fields or 'last_name' in cw_error_fields:
                            # если да, то в список ошибок надо отдать только то поле, на что ругается
                            cw_error_fields.remove('full_name')
                    else:
                        # кейс, когда имя или фамилия по отдельности не проходят проверку, но вместе - проходят
                        # например китайские фио, тогда не надо ругаться на имя и фамилию по отдельности
                        cw_error_fields -= {'first_name', 'last_name'}

                for cw_error_field in cw_error_fields:
                    for t in texts:
                        if t.field_name_for_cw != cw_error_field:
                            continue
                        for field_name in t.field_names_for_error:
                            if field_name in errors:
                                errors[field_name] += u', {}'.format(t.value)
                            else:
                                errors[field_name] = t.value

                if errors:
                    raise error_class(errors)


def get_surface_for_track(consumer, action=None):
    if not consumer:
        return None

    source = 'mobile_proxy' if consumer == settings.MOBILEPROXY_CONSUMER else 'web'
    if action is not None:
        surface = '%s_%s' % (source, action)
    else:
        surface = source
    return surface


def is_second_step_allowed(track, second_step):
    return (
        track.is_second_step_required and
        track.allowed_second_steps is not None and
        second_step in track.allowed_second_steps
    )


def should_ignore_per_ip_counters(app_id, phone):
    if app_id in settings.YANGO_APP_IDS:
        for phone_code in settings.TRUSTED_YANGO_PHONE_CODES:
            if phone.startswith(phone_code):
                return True
    return False


__all__ = (
    'prepare_env',
    'increment_track_counter',
    'parse_args_for_track',
    'track_to_response',
    'check_spammer',
    'get_email_domain',
    'chained_unquote',
    'validate_password',
    'extract_tld',
    'get_logout_datetime',
    'escape_query_characters_middleware',
    'create_csrf_token',
    'looks_like_yandex_email',
    'CleanWebChecker',
    'get_surface_for_track',
    'is_second_step_allowed',
    'try_reload_dynamic_settings',
    'should_ignore_per_ip_counters',
)
