# -*- coding: utf-8 -*-

from functools import wraps
import logging
import os.path
import re
import unicodedata

from flask import request
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers.statbox import to_statbox
from passport.backend.core.types.file import File
from passport.backend.core.validators import (
    Invalid,
    State,
)
import six

from .format_response import (
    error_response,
    format_errors,
)


log = logging.getLogger('passport.api.common')
error_log = logging.getLogger('api.requests.error')

filename_alnum_strip_re = re.compile(u'[^\\w_.-]', re.UNICODE)
windows_device_files = (
    u'CON',
    u'AUX',
    u'COM1',
    u'COM2',
    u'COM3',
    u'COM4',
    u'LPT1',
    u'LPT2',
    u'LPT3',
    u'PRN',
    u'NUL',
)


# Переписанная из werkzeug'а, т.к. оригинальная функция вырезала все не ASCII
# литеры.
# Например, от привет.jpg, оставалось только jpg.
def secure_filename(filename, file_system_encoding=None):
    """
    Pass it a filename and it will return a secure version of it.  This
    filename can then safely be stored on a regular file system and passed
    to :func:`os.path.join`.

    On windows systems the function also makes sure that the file is not
    named after one of the special device files.

    >>> secure_filename("My cool movie.mov")
    'My_cool_movie.mov'
    >>> secure_filename("../../../etc/passwd")
    'etc_passwd'
    >>> secure_filename(u'i contain cool \xfcml\xe4uts.txt')
    'i_contain_cool_umlauts.txt'

    The function might return an empty filename.  It's your responsibility
    to ensure that the filename is unique and that you generate random
    filename if the function returned an empty one.

    :param filename: the filename to secure
    """
    if file_system_encoding is None:
        file_system_encoding = settings.FILE_SYSTEM_ENCODING
    if not isinstance(filename, six.text_type):
        raise ValueError(u'Filename should be Unicode')

    # Сводим похожие по форме, но разные литеры к единой литере и отбрасываем
    # все буквы, которые нельзя записать в путевом имени файловой системы.
    filename = (unicodedata.normalize('NFKD', filename).
                encode(file_system_encoding, 'ignore').
                decode(file_system_encoding))

    # Убираем разделители путевого имени
    for sep in os.path.sep, os.path.altsep:
        if sep:
            filename = filename.replace(sep, u' ')

    # Оставляем только только букво-численные литеры.
    filename = filename_alnum_strip_re.sub(u'', u'_'.join(filename.split()))
    filename = filename.strip(u'._')

    # on nt a couple of special files are present in each folder.  We
    # have to ensure that the target file is not such a filename.  In
    # this case we prepend an underline
    if (os.name == u'nt' and filename and
            filename.split(u'.')[0].upper() in windows_device_files):
        filename = u'_' + filename

    return filename


def invalid_form_response(error):
    status = 404 if 'service' in (error.error_dict or {}) else 400
    return error_response(status, errors=format_errors(error))


def get_request_files():
    files = {}
    if request.method in ['POST', 'PUT']:
        for field, file_list in request.files.lists():
            files[field] = tuple(
                File(
                    filename=secure_filename(f.filename),
                    stream=f.stream,
                ) for f in file_list
            )
    return files


def get_request_values():
    if request.method in ['POST', 'PUT']:
        form_values = request.form.to_dict()
        get_values = request.args.to_dict()

        consumer = get_values.get('consumer')
        # consumer должен содержаться в query string, см.
        # документацию http://wiki.yandex-team.ru/passport/backend/api#novyemetody
        if consumer:
            form_values.update({'consumer': consumer})
        else:
            # Выбросим потребителя из параметров.
            if 'consumer' in form_values:
                consumer_from_body = form_values.pop('consumer')
                log.info('Consumer %s came from request body', consumer_from_body)

        from passport.backend.api.forms.base import DeviceInfoForm
        GET_ARGS = DeviceInfoForm.DEVICE_INFO_FIELD_NAMES
        for arg_name in GET_ARGS:
            arg_value = get_values.get(arg_name)
            if arg_value:
                form_values.update({arg_name: arg_value})

        values = form_values
    else:
        values = request.args.to_dict()

    return values


def validate(form, error_handler=invalid_form_response, get_request_values_func=get_request_values):
    '''
    :param form форма из forms
    :param error_handler функция, возвращающая xml-ответ в случае ошибки
    '''
    def wrapper(f):
        @wraps(f)
        def _wrapper(*args, **kwargs):
            try:
                values = get_request_values_func()
                values.update(kwargs)
                # new_args dict провалидированные параметры через форму
                new_args = form.to_python(values, State(request.env))
                log.info('Request form is valid')
                return f(new_args)
            except Invalid as e:
                log.info('Request form validation failed with error: %s', e)
                return error_handler(e)

        return _wrapper

    return wrapper


def static_statbox(events):
    '''
    Записывает логи для statbox c track_id, если он есть,
    и статичными данными из events
    :param events dict с событиями для записи в лог
    '''
    def wrapper(f):
        @wraps(f)
        def _wrapper(*args, **kwargs):
            new_events = events.copy()
            values = request.values.to_dict()
            track_id = values.get('track_id')
            mode = values.get('mode')
            if track_id:
                new_events.update({'track_id': track_id})
            if mode:
                new_events.update({'mode': mode})

            to_statbox(new_events)
            return f(*args, **kwargs)
        return _wrapper
    return wrapper


def headers_required(*headers):
    def _wrapper(func):
        @wraps(func)
        def wrapper(*args, **kwargs):
            missing_headers = []
            for header in headers:
                if not request.headers.get(header):
                    log.error('Required header is missing or empty: "%s".', header)
                    missing_headers.append(header)

            if missing_headers:
                return error_response(400, error_code='MissingHeader',
                                      error_message='Missing required headers: "%s"' % ', '.join(missing_headers))

            return func(*args, **kwargs)
        return wrapper
    return _wrapper


__all__ = (
    'validate',
    'headers_required',
    'static_statbox',
    'invalid_form_response',
    'get_request_values',
    'get_request_files',
)
