# -*- coding: utf-8 -*-
import logging

from flask import request
from passport.backend.api import exceptions
from passport.backend.core.builders import blackbox
from passport.backend.core.builders.captcha import (
    CaptchaAccessError,
    CaptchaError,
    CaptchaLocateError,
    CaptchaServerError,
    CaptchaTypeCheckError,
    CaptchaXmlParseError,
)
from passport.backend.core.builders.historydb_api import (
    HistoryDBApiInvalidResponseError,
    HistoryDBApiTemporaryError,
)
from passport.backend.core.builders.mail_apis.exceptions import (
    HuskyInvalidResponseError,
    HuskyTemporaryError,
)
from passport.backend.core.builders.oauth import OAuthTemporaryError
from passport.backend.core.builders.yasms import (
    YaSmsAccessDenied,
    YaSmsError,
    YaSmsLimitExceeded,
    YaSmsPermanentBlock,
    YaSmsSecureNumberExists,
    YaSmsSecureNumberNotAllowed,
    YaSmsTemporaryBlock,
)
from passport.backend.core.conf import settings
from passport.backend.core.dbmanager.manager import DBError
from passport.backend.core.exceptions import (
    InvalidIpHeaderError,
    ValueChanged,
    WrongHostError,
)
from passport.backend.core.grants import (
    GrantsError,
    MissingGrantsError,
)
from passport.backend.core.models.account import UnknownUid
from passport.backend.core.redis_manager.redis_manager import (
    RedisError,
    RedisWatchError,
)
from passport.backend.core.serializers.eav.exceptions import (
    EavDeletedObjectNotFound,
    EavUpdatedObjectNotFound,
)
from passport.backend.core.tracks.exceptions import (
    ConcurrentTrackOperationError,
    NodeNotFoundError,
    TrackNotFoundError,
)
from werkzeug.exceptions import InternalServerError

from .format_response import (
    error_response,
    format_errors,
)


log = logging.getLogger('passport.api.common')
error_log = logging.getLogger('api.requests.error')


def log_internal_error(err, error_details=None):
    log.error(u'Request failed with "%s", traceback written to error log.', err)
    extra = {
        'request_info': str(request),
        'request_headers': dict(request.headers),
        'request_values': request.values.to_dict(),
    }
    if error_details is not None:
        extra['error_details'] = error_details
    error_log.error(
        u'%s (%s) on %s %s' % (err.__class__.__name__, err.__class__, request.method, request.path),
        exc_info=err,
        extra=extra,
    )


def internal_error(err):
    log_internal_error(err)
    return error_response(500, error_code='internal', error_message='Internal error')


def yasms_error_handler(e):
    message = str(e)
    if isinstance(e, YaSmsLimitExceeded):
        return error_response(400, error_code='SmsSendLimitExceeded',
                              error_message='Resend reached sms limit')
    elif isinstance(e, YaSmsPermanentBlock):
        return error_response(400, error_code='YaSmsPermanentBlock',
                              error_message=message)
    elif isinstance(e, YaSmsTemporaryBlock):
        return error_response(400, error_code='SmsSendTooEarly',
                              error_message=message)
    elif isinstance(e, YaSmsSecureNumberNotAllowed):
        return error_response(400, error_code='YaSmsSecureNumberNotAllowed', error_message=message)
    elif isinstance(e, YaSmsSecureNumberExists):
        return error_response(400, error_code='YaSmsSecureNumberAlreadyPresented', error_message=message)
    elif isinstance(e, YaSmsAccessDenied):
        return error_response(500, error_code='YaSmsACL', error_message='YaSms ACL failed')

    return error_response(503, error_code='YaSmsError', error_message=message)


def error_handler(e):
    if isinstance(e, InternalServerError):
        # Flask оборачивает необработанные исключения в InternalServerError, сохраняя исходное исключение в атрибуте
        e = getattr(e, 'original_exception', e)

    if isinstance(e, MissingGrantsError):
        error = u'Access denied for ip: %s; consumer: %s; tvm_client_id: %s. Required grants: %s' % \
            (e.ip, e.consumer, e.tvm_client_id, e.missing)
        return error_response(403, error_code='AccessDenied', error_message=error)
    elif isinstance(e, GrantsError):
        error = u'Access denied for ip: %s; consumer: %s; tvm_client_id: %s' % \
            (e.ip, e.consumer, e.tvm_client_id)
        return error_response(403, error_code='AccessDenied', error_message=error)
    elif isinstance(e, blackbox.AccessDenied):
        return error_response(500, error_code='BlackboxACL', error_message='Blackbox ACL failed')
    elif isinstance(e, blackbox.BaseBlackboxError):
        return error_response(503, error_code='BlackboxFailed', error_message='Blackbox failed')
    elif isinstance(e, blackbox.BlackboxTemporaryError):
        return error_response(503, error_code='BlackboxRequest', error_message='Blackbox request failed')
    elif isinstance(e, OAuthTemporaryError):
        return error_response(503, error_code='OAuthFailed', error_message='OAuth failed')
    elif isinstance(e, InvalidIpHeaderError):
        return error_response(400,
                              error_code='InvalidHeader',
                              error_message='Invalid header in request')
    elif isinstance(e, UnknownUid):
        return error_response(404, error_code='UnknownUID', error_message='Unknown UID')
    elif isinstance(e, DBError):
        return error_response(503, error_code='DBRequest', error_message='DB request failed')
    elif isinstance(e, (EavDeletedObjectNotFound, EavUpdatedObjectNotFound)):
        # 409 Conflict
        return error_response(409, error_code='DBRequest', error_message='DB integrity error')
    elif isinstance(e, (RedisError, RedisWatchError, ConcurrentTrackOperationError)):
        return error_response(503, error_code='RedisRequest', error_message='Redis request failed')
    elif isinstance(e, TrackNotFoundError):
        return error_response(400, error_code='UnknownTrack', error_message='Unknown track_id')
    elif isinstance(e, NodeNotFoundError):
        return error_response(400, error_code='UnknownNode', error_message='Unknown node in track_id')
    elif isinstance(e, CaptchaXmlParseError):
        return error_response(503, error_code='CaptchaXml', error_message='XML from captcha parse error')
    elif isinstance(e, CaptchaTypeCheckError):
        return error_response(400, error_code='CaptchaType', error_message='Different type generated and requested')
    elif isinstance(e, CaptchaLocateError):
        return error_response(400, error_code='CaptchaLocate', error_message='Cannot locate captcha')
    elif isinstance(e, CaptchaAccessError):
        return error_response(500, error_code='CaptchaAccess', error_message='Missing access permissions')
    elif isinstance(e, CaptchaServerError):
        return error_response(503, error_code='CaptcheServer', error_message='Captcha server failed')
    elif isinstance(e, CaptchaError):
        return error_response(503, error_code='CaptchaFailed', error_message='Captcha failed')
    elif isinstance(e, ValueChanged):
        return error_response(400, errors=format_errors(e, field=e.field))
    elif isinstance(e, YaSmsError):
        return yasms_error_handler(e)
    elif isinstance(e, WrongHostError):
        return error_response(400, error_code='WrongHost', error_message='Wrong host')
    elif isinstance(e, exceptions.SmsSendTooEarlyError):
        # FIXME: Хак для успокоения тестов, которые подменяют settings
        return error_response(
            400,
            error_code='SmsSendTooEarly',
            error_message='Can\'t resend sms earlier than %d seconds from previous sending' % settings.SMS_VALIDATION_RESEND_TIMEOUT,
        )
    elif isinstance(e, exceptions.OAuthTokenAlreadyCreatedError):
        return error_response(400,
                              error_code='OAuthTokenAlreadyCreated',
                              error_message='OAuth token already created for this account')
    elif isinstance(e, exceptions.InvalidOAuthTokenError):
        return error_response(400,
                              error_code='InvalidOAuthToken',
                              error_message='OAuth token is invalid')
    elif isinstance(e, exceptions.InvalidOAuthScopeError):
        return error_response(400,
                              error_code='InvalidOAuthScope',
                              error_message='OAuth scope is invalid')
    elif isinstance(e, exceptions.AuthorizationNotAllowedError):
        return error_response(400,
                              error_code='AuthorizationNotAllowed',
                              error_message='Authorization is not allowed for this track')
    elif isinstance(e, (HuskyInvalidResponseError, HuskyTemporaryError)):
        return error_response(
            code=500,
            error_code='backend.husky_failed',
            error_message='Husky API failed',
        )
    elif isinstance(
        e,
        (
            HistoryDBApiTemporaryError,
            HistoryDBApiInvalidResponseError,
        ),
    ):
        return error_response(
            503,
            error_code='historydbapifailed',
            error_message='HistoryDB API failed',
        )
    elif isinstance(e, exceptions.BaseApiError):
        return error_response(e.status, error_code=e.code, error_message=e.message)
    else:
        return internal_error(e)


__all__ = (
    'internal_error',
    'error_handler',
    'log_internal_error',
)
