# -*- coding: utf-8 -*-
import json

from flask import (
    request,
    Response,
)
from passport.backend.core.exceptions import ValueChanged
from passport.backend.core.logging_utils.helpers import mask_sensitive_fields
from six import iteritems


class JsonResponse(Response):
    def __init__(self, *args, **kwargs):
        self.dict_ = dict(*args, **kwargs)
        data = self._build_json(self.dict_)
        super(JsonResponse, self).__init__(data, mimetype='application/json')

    def _build_json(self, *args, **kwargs):
        indent = None
        if not request.is_xhr:
            indent = 2
        return json.dumps(dict(*args, **kwargs),
                          indent=indent)


class JsonLoggedResponse(JsonResponse):
    def __init__(self, *args, **kwargs):
        self._error_code = kwargs.pop('error_code', None)
        sensitive_fields = kwargs.pop('sensitive_fields', None)
        super(JsonLoggedResponse, self).__init__(*args, **kwargs)

        self._log_data = self._build_json(
            mask_sensitive_fields(self.dict_, sensitive_fields),
        )

    def error_code(self):
        return self._error_code

    def log_data(self):
        return self._log_data


class XmlLoggedResponse(Response):
    default_mimetype = u'text/xml'

    def __init__(self, response, format_, sensitive_fields=None, error_code=None, **kwargs):
        self._error_code = error_code
        masked_response = mask_sensitive_fields(
            response,
            fields=sensitive_fields,
        )
        self.formatted_response = format_(response)
        self.formatted_masked_response = format_(masked_response)
        super(XmlLoggedResponse, self).__init__(self.formatted_response, **kwargs)

    def error_code(self):
        return self._error_code

    def log_data(self):
        return self.formatted_masked_response


def ok_response(*args, **kwargs):
    return json_response(200, *args, status='ok', **kwargs)


def error_response(code=400, error_code=None, error_message=None, errors=None,
                   sensitive_fields=None, **kwargs):
    if error_code:
        errors = [format_error(error_code, error_message)]
    return json_response(
        code,
        status='error',
        errors=errors,
        sensitive_fields=sensitive_fields,
        error_code=error_code,
        **kwargs
    )


def simple_error_response(errors, code=200, sensitive_fields=None, **kwargs):
    error_code = errors[0] if errors else None
    return json_response(
        code,
        status='error',
        errors=errors,
        sensitive_fields=sensitive_fields,
        error_code=error_code,
        **kwargs
    )


def json_response(_code, *args, **kwargs):
    response = JsonLoggedResponse(*args, **kwargs)
    response.status_code = _code
    return response


def format_error(code, message, field=None):
    return {'code': code.lower(), 'message': message, 'field': field}


def format_error_with_value(code, message, value, field=None):
    return dict(value=value, **format_error(code, message, field))


def format_errors(exc, field=None):
    errors = []

    for field, e in iteritems(exc.error_dict or {}):
        errors.extend(format_errors(e, field))
    for e in exc.error_list or []:
        errors.extend(format_errors(e, field))

    if not errors:
        if isinstance(exc, ValueChanged):
            errors.append(format_error_with_value(exc.code, exc.msg, exc.value, field))
        else:
            errors.append(format_error(exc.code, exc.msg, field))

    return errors


__all__ = (
    'format_error',
    'format_errors',
    'ok_response',
    'error_response',
    'simple_error_response',
    'json_response',
    'JsonLoggedResponse',
    'XmlLoggedResponse',
)
