# -*- coding: utf-8 -*-
# TODO: возможно, стоит унести в Core
from datetime import timedelta
import logging
import time

from passport.backend.core.builders.historydb_api import (
    get_historydb_api,
    HistoryDBApi,
    HistoryDBApiTemporaryError,
)
from passport.backend.core.conf import settings
from passport.backend.core.historydb.analyzer import EventsAnalyzer
from passport.backend.core.historydb.analyzer.events import get_account_registration_unixtime


log = logging.getLogger('passport.api.common.historydb_api')

DEFAULT_EVENTS_LIMIT = 5000


def get_historydb_events_info(account, limit=DEFAULT_EVENTS_LIMIT, analyzer_params=None, **entities_to_analyze):
    try:
        analyzer = EventsAnalyzer(**entities_to_analyze)
        events_info = analyzer.load_and_analyze_events(account, limit=limit, **(analyzer_params or {}))
        return True, events_info
    except HistoryDBApiTemporaryError:
        # HistoryDB временно недоступна
        return False, None


def auths_aggregated_runtime(account, check_depth=None, limit=None):
    """
    Обертка для ручки auths/aggregated/runtime.
    @param account: аккаунт, для которого выполняется поиск
    @param check_depth: глубина поиска в днях
    @param limit: ограничение на число авторизаций в результате
    """
    try:
        historydb = HistoryDBApi(timeout=settings.HISTORYDB_API_TIMEOUT_FOR_RESTORE)
        end_ts = time.time()
        if check_depth is not None:
            delta = timedelta(check_depth)
            start_ts = max(
                end_ts - delta.total_seconds(),
                get_account_registration_unixtime(account.registration_datetime),
            )
        else:
            start_ts = get_account_registration_unixtime(account.registration_datetime)
        result = historydb.auths_aggregated_runtime(
            account.uid,
            int(start_ts),
            int(end_ts),
            limit=limit,
        )
        return True, result['history']
    except HistoryDBApiTemporaryError:
        # HistoryDB временно недоступна
        return False, []


def find_password_in_history(account, password):
    try:
        historydb = get_historydb_api()
        start_ts = get_account_registration_unixtime(account.registration_datetime)
        end_ts = time.time()
        password_found, active_ranges = historydb.events_passwords(
            account.uid,
            password,
            int(start_ts),
            int(end_ts),
        )
        return True, password_found, active_ranges
    except HistoryDBApiTemporaryError:
        # HistoryDB временно недоступна
        return False, False, []


class AuthsAggregatedRuntimeInfo(object):
    """
    Обертка над вызовом auths_aggregated_runtime, позволяющая удобно итерироваться по результатам вызова
    """
    def __init__(self, account, check_depth=None, limit=None):
        self.auths_limit = limit
        self.api_status, self.aggregated_auths = auths_aggregated_runtime(
            account,
            check_depth=check_depth,
            limit=self.auths_limit,
        )

    def __iter__(self):
        for history_item in self.aggregated_auths:
            # обход в порядке убывания timestamp
            timestamp = history_item['timestamp']
            for auth_item_with_count in history_item['auths']:
                auth_item = auth_item_with_count['auth']
                count = auth_item_with_count['count']
                yield timestamp, auth_item, count
