# -*- coding: utf-8 -*-

import logging

from passport.backend.api.exceptions import (
    SuggestKiddishLoginError,
    SuggestKolonkishLoginError,
    SuggestNeohonishLoginError,
    SuggestPhonishLoginError,
    SuggestSocialLoginError,
    SuggestYambotLoginError,
)
from passport.backend.core import (
    types,
    validators,
)
from passport.backend.core.grants.grants_config import check_specialtest_yandex_login_grant
from passport.backend.core.portallib import is_yandex_ip
from passport.backend.core.types.account.account import (
    ACCOUNT_TYPE_KIDDISH,
    ACCOUNT_TYPE_KOLONKISH,
    ACCOUNT_TYPE_NEOPHONISH,
    ACCOUNT_TYPE_PHONISH,
    ACCOUNT_TYPE_SOCIAL,
    ACCOUNT_TYPE_YAMBOT,
)
from passport.backend.core.types.login.login import is_test_yandex_login
from passport.backend.core.validators import State


login_log = logging.getLogger('passport.api.login')


def allow_special_test_yandex_login_registration(login, ip):
    '''
    Функция, по логину и ip-адресу проверяет, можно ли
    создавать пользователя с тестовым логином для данного ip-адреса.

    Логин является тестовым, если начинается с тестовых префиксов.
    Проверяется является ли ip-адрес яндексовым или он указан в грантах.
    '''
    test_prefix = is_test_yandex_login(login)
    if test_prefix and (is_yandex_ip(str(ip)) or check_specialtest_yandex_login_grant(ip, test_prefix)):
        return True
    return False


def build_available_social_login(retries, env):
    return build_available_login(
        ACCOUNT_TYPE_SOCIAL,
        types.login.login.generate_social_login,
        retries,
        env,
        login_log,
        SuggestSocialLoginError,
    )


def build_available_kiddish_login(retries, env):
    return build_available_login(
        ACCOUNT_TYPE_KIDDISH,
        types.login.login.generate_kiddish_login,
        retries,
        env,
        login_log,
        SuggestKiddishLoginError,
    )


def build_available_kolonkish_login(retries, env):
    return build_available_login(
        ACCOUNT_TYPE_KOLONKISH,
        types.login.login.generate_kolonkish_login,
        retries,
        env,
        login_log,
        SuggestKolonkishLoginError,
    )


def build_available_phonish_login(retries, env):
    return build_available_login(
        ACCOUNT_TYPE_PHONISH,
        types.login.login.generate_phonish_login,
        retries,
        env,
        login_log,
        SuggestPhonishLoginError,
    )


def build_available_neophonish_login(retries, env):
    return build_available_login(
        ACCOUNT_TYPE_NEOPHONISH,
        types.login.login.generate_neophonish_login,
        retries,
        env,
        login_log,
        SuggestNeohonishLoginError,
    )


def build_available_yambot_login(retries, env):
    return build_available_login(
        ACCOUNT_TYPE_YAMBOT,
        types.login.login.generate_yambot_login,
        retries,
        env,
        login_log,
        SuggestYambotLoginError,
    )


def build_available_login(login_type, generate, retries, env, logger, exc_class):
    for i in range(retries):
        login = generate()
        try:
            validators.Availability(login_type=login_type).to_python({'login': login}, State(env))
            return login
        except validators.Invalid as e:
            logger.info('Build login attempt [%d/%d] login: "%s" error: "%s"', i + 1, retries, login, e)
    logger.warning('Build login failed')
    raise exc_class


__all__ = (
    'allow_special_test_yandex_login_registration',
    'build_available_social_login',
    'build_available_kiddish_login',
    'build_available_kolonkish_login',
    'build_available_neophonish_login',
    'build_available_phonish_login',
    'build_available_yambot_login',
    'build_available_login',
)
