# -*- coding: utf-8 -*-
import logging
import os
import time

from flask import (
    has_request_context,
    request,
)
from passport.backend.api.common.format_response import (
    JsonLoggedResponse,
    XmlLoggedResponse,
)
from passport.backend.api.env.env import (
    get_consumer_ip,
    get_user_agent,
)
from passport.backend.core.host.host import get_current_host
from passport.backend.core.logging_utils.helpers import (
    encode_header,
    escape_text_for_log,
    mask_sensitive_fields,
    trim_message,
)
from passport.backend.core.logging_utils.loggers import AccessLogger
from passport.backend.core.logging_utils.request_id import RequestIdManager
from passport.backend.core.serializers.logs import log_events as _log_events
from passport.backend.core.tracks.track_manager import TrackManager
from passport.backend.utils.string import smart_text
from six import iteritems


REQUEST_ID_LAST_CHARS_COUNT = 10
LOGIN_MAX_LENGTH = 50

log = logging.getLogger('passport.api.common')
error_log = logging.getLogger('api.requests.error')


def read_track(track_id):
    """
    Читает существующий трек по полученному ранее идентификатору.
    """
    return TrackManager().read(track_id)


def get_param_value(param_name, account=None):
    """
    Получаем из форм/урла, трека, из ответа ЧЯ если туда был поход в процессе запроса
    """
    if has_request_context():
        value = request.values.get(param_name) or (request.view_args and request.view_args.get(param_name))
    else:
        value = None
    if value:
        return value
    else:
        track_id = get_track_id()
        if track_id:
            try:
                track = read_track(track_id)
                value = getattr(track, param_name)
                if value:
                    return getattr(track, param_name)
            except Exception:
                pass

        if account:
            value = getattr(account, param_name)
            if value:
                return value
        return '-'


def get_track_id():
    """
    отдаём приоритет track_id, зашитому в урле
    """
    name = 'track_id'
    if has_request_context():
        return (request.view_args and request.view_args.get(name)) or request.values.get(name)


def get_log_track_id():
    track_id = get_track_id()
    return track_id or '-'


def get_external_request_id():
    if has_request_context():
        request_id = request.env.request_id
        if request_id:
            return request_id[-REQUEST_ID_LAST_CHARS_COUNT:]
    return '-'


def setup_log_prefix(account=None):
    clear_log_prefix()

    request_id = (
        ('%x' % get_current_host().get_id()).upper(),
        os.getpid(),
        request.start_time,
        get_external_request_id(),
        get_log_track_id(),
    )
    RequestIdManager.push_request_id(*request_id)

    request_id = (
        get_param_value('uid', account),
        get_param_value('login', account)[:LOGIN_MAX_LENGTH],
    )
    RequestIdManager.push_request_id(*request_id)


def clear_log_prefix():
    RequestIdManager.clear_request_id()


def format_parameters_for_log(params, trim=False):
    keys_and_values = []
    for key, value in iteritems(params):
        key_and_value = u'{}={}'.format(
            escape_text_for_log(key, errors='replace'),
            escape_text_for_log(value, errors='replace'),
        )
        keys_and_values.append(key_and_value)
    message = u', '.join(keys_and_values)
    if trim:
        message = trim_message(message)
    return smart_text(message, errors='replace')


def log_request():
    masked_args = mask_sensitive_fields(request.args.to_dict())
    masked_form = mask_sensitive_fields(request.form.to_dict())

    # Строчка в логе для упрощения грепа. Урл и параметры в одной строке.
    # request_id добавляется форматтером, поэтому здесь не упоминается
    log.info(u'Request: ' + u'\t'.join([
        request.method + ' ' + request.path,
        # GET не обрезаем, т.к. он уже ограничен by design длиной урла
        u'args: ' + format_parameters_for_log(masked_args, trim=False),
        # POST обрезаем от греха подальше
        u'form: ' + format_parameters_for_log(masked_form, trim=True),
    ]))
    log.info(u'Headers: ' + u', '.join([
        encode_header(name, value)
        for name, value in iteritems(mask_sensitive_fields(dict(request.headers)))
    ]))

    if request.env.request_id:
        log.info(u'Request id: %s', request.env.request_id)


def log_response(response):
    if isinstance(response, (JsonLoggedResponse, XmlLoggedResponse)):
        response_data = response.log_data()
    else:
        response_data = response.data
    response_data = trim_message(smart_text(response_data))

    log.info(u'Response status: %s', response.status)

    time_info = u'duration=%.6f' % (time.time() - request.start_time)

    if isinstance(response, JsonLoggedResponse):
        keys = {u'status', u'state', u'error', u'errors', u'error_message', u'code'}
        essential_fields = {key: smart_text(response.dict_[key]) for key in keys if response.dict_.get(key) is not None}
        essential_data = u', '.join(u'%s=%s' % (k, v) for k, v in iteritems(essential_fields))
        log.info(u'Response sent, %s: %s; Full: "%s"', time_info, essential_data, response_data)
    else:
        log.info(u'Response sent, %s: "%s"', time_info, response_data)

    clear_log_prefix()

    return response


def log_access(response):
    # все try-except тут, чтобы не дай бог не упасть при логировании
    error = None
    if isinstance(response, (JsonLoggedResponse, XmlLoggedResponse)):
        error = response.error_code()
    try:
        status = response.dict_['status']
    except (KeyError, AttributeError):
        status = None
    consumer = request.args.to_dict().get('consumer')
    AccessLogger().log(
        consumer=consumer,
        method=request.method,
        url=request.url,
        status_code=response.status_code,
        status=status,
        unixtime='%.3f' % request.start_time,
        duration='%.6f' % (time.time() - request.start_time),
        consumer_ip=get_consumer_ip(request),
        user_agent=get_user_agent(request),
        error=error,
    )
    return response


def log_events(user_events, initiator_uid=None, datetime_=None):
    """
    Записывает список событий в журнал с одинаковой временной отметкой.

    Входные параметры
        user_events
            Словарь, ключи которого uid'ы, а значание словарь-с-событиями.
        initiator_uid
            Идентификатор пользователя, который вызвал ручку.
        datetime_
            Делать все записи с данной временной меткой.
    """
    _log_events(
        user_events,
        request.env.user_ip,
        request.env.user_agent,
        request.env.cookies.get('yandexuid'),
        initiator_uid=initiator_uid,
        datetime_=datetime_,
    )


__all__ = (
    'setup_log_prefix',
    'log_request',
    'log_response',
    'log_access',
    'get_track_id',
)
