# -*- coding: utf-8 -*-

import logging

from passport.backend.api.common.account_manager import (
    get_device_params_from_track,
    track_to_oauth_params,
)
from passport.backend.api.common.ip import get_ip_autonomous_system
from passport.backend.api.common.yasms import (
    generate_fake_global_sms_id,
    log_sms_not_delivered,
)
from passport.backend.core.builders.antifraud import (
    BaseAntifraudApiError,
    ScoreAction,
)
from passport.backend.core.builders.phone_squatter import (
    BasePhoneSquatterError,
    get_phone_squatter,
    PhoneSquatterPhoneNumberNotTrackedError,
    PhoneSquatterPhoneNumberUntrackableError,
)
from passport.backend.core.conf import settings
from passport.backend.core.env import Environment
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.utils.common import noneless_dict
from passport.backend.utils.time import (
    get_unixtime,
    unixtime_to_datetime,
)


log = logging.getLogger(__name__)

CONFIRM_METHOD_BY_SMS = 'by_sms'
CONFIRM_METHOD_BY_CALL = 'by_call'
CONFIRM_METHOD_BY_FLASH_CALL = 'by_flash_call'


class PhoneAntifraudFeatures(object):
    def __init__(self):
        self.track = None
        self.AS = None
        self.channel = None
        self.external_id = None
        self.phone_confirmation_method = None
        self.request = None
        self.request_path = None
        self.retpath = None
        self.is_secure_phone = None
        self.scenario = None
        self.sub_channel = None
        self.t = None
        self.uid = None
        self.user_agent = None
        self.user_ip = None
        self.user_phone_number = None
        self.js_fingerprint = None
        self.device_params = None
        self.page_loading_info = None
        self.check_css_load = None
        self.check_js_load = None
        self.is_captcha_passed = None
        self.login_id = None
        self.phone_confirmation_language = None

    @classmethod
    def default(
        cls,
        sub_channel,
        user_phone_number,
    ):
        self = PhoneAntifraudFeatures()
        self.channel = 'pharma'
        self.request = 'confirm'
        self.sub_channel = sub_channel
        self.t = get_unixtime() * 1000
        self.user_phone_number = user_phone_number
        self.device_params = dict()
        return self

    def add_environment_features(self, env):
        if env.user_ip:
            self.AS = get_ip_autonomous_system(env.user_ip)
            self.user_ip = env.user_ip

        if env.user_agent:
            self.user_agent = env.user_agent

        if env.request_path:
            self.request_path = env.request_path

    def add_headers_features(self, headers):
        self.add_environment_features(self.build_env_from_headers(headers))

    def add_track_features(self, track):
        self.track = track

        self.external_id = 'track-{}'.format(track.track_id)
        if track.retpath:
            self.retpath = track.retpath
        if track.uid:
            self.uid = int(track.uid)

        self.device_params = track_to_oauth_params(get_device_params_from_track(track))

        if track.js_fingerprint:
            self.js_fingerprint = track.js_fingerprint
        if track.scenario:
            self.scenario = track.scenario
        elif track.track_type:
            # Нужен дефолт, пока фронт не начнёт везде проставлять поле
            # scenario
            self.scenario = track.track_type

        self.is_captcha_passed = track.is_captcha_checked and track.is_captcha_recognized

        if track.page_loading_info:
            self.page_loading_info = track.page_loading_info
        if track.check_css_load:
            self.check_css_load = track.check_css_load
        if track.check_js_load:
            self.check_js_load = track.check_js_load

    def add_secure_phone_flag(self):
        self.is_secure_phone = True

    def add_dict_features(self, features):
        if features:
            for feature, feature_value in features.items():
                if hasattr(self, feature) and feature_value is not None:
                    setattr(self, feature, feature_value)

    def as_score_dict(self):
        retval = dict(
            AS=self.AS,
            channel=self.channel,
            external_id=self.external_id,
            ip=str(self.user_ip) if self.user_ip else None,
            phone_confirmation_method=self.phone_confirmation_method,
            request=self.request,
            request_path=self.request_path,
            retpath=self.retpath,
            is_secure_phone=self.is_secure_phone,
            scenario=self.scenario,
            sub_channel=self.sub_channel,
            t=self.t,
            uid=self.uid,
            user_agent=self.user_agent,
            user_phone=self.user_phone_number.e164 if self.user_phone_number else None,
            js_fingerprint=self.js_fingerprint,
            page_loading_info=self.page_loading_info,
            check_css_load=self.check_css_load,
            check_js_load=self.check_js_load,
            login_id=self.login_id,
            phone_confirmation_language=self.phone_confirmation_language,
        )
        retval.update(self.device_params)
        return noneless_dict(retval)

    def score(self, antifraud, consumer, error_class, captcha_class, statbox, mask_denial=False):
        statbox.bind(scenario=self.scenario)

        try:
            score_response = antifraud.score(self.as_score_dict())
        except BaseAntifraudApiError as e:
            log.error('Antifraud request error {}: {}'.format(type(e).__name__, e))
            statbox.bind(af_request_error=trim_message(str(e)))
            return

        log.debug('Received antifraud score: {}'.format(score_response))

        statbox.bind(
            antifraud_action=score_response.action,
            antifraud_reason=score_response.reason,
            antifraud_tags=','.join(score_response.tags or []),
        )
        if score_response.action != ScoreAction.ALLOW:
            log.debug('Denying request (mask={}), antifraud action "{}"'.format(mask_denial, score_response.action))
            statbox.log(status='error', error='antifraud_score_deny', mask_denial=mask_denial)
            if self.phone_confirmation_method == CONFIRM_METHOD_BY_SMS:
                self.log_sms_not_delivered(consumer=consumer)
            raise error_class()
        elif (
            score_response.action == ScoreAction.ALLOW and
            score_response.tags and
            not self.is_captcha_passed and
            captcha_class is not None
        ):
            # На конкретные теги не смотрим: пока всегда показываем капчу
            log.debug('Requesting captcha, antifraud tags [{}]'.format(', '.join(score_response.tags)))
            statbox.log(status='error', error='antifraud_score_captcha')
            if self.track is not None:
                self.track.is_captcha_required = True
                self.track.is_captcha_checked = False
                self.track.is_captcha_recognized = False
                self.track.captcha_image_url = None
            raise captcha_class()
        else:
            statbox.log()

    def log_sms_not_delivered(self, consumer):
        log_sms_not_delivered(
            reason='antifraud',
            number=self.user_phone_number,
            global_sms_id=generate_fake_global_sms_id(),
            caller=consumer,
            client_ip=str(self.user_ip) if self.user_ip else None,
            user_agent=self.user_agent,
        )

    @staticmethod
    def build_env_from_headers(headers):
        return Environment(
            user_agent=headers.get('Ya-Client-User-Agent'),
            user_ip=headers.get('Ya-Consumer-Client-Ip'),
        )


def get_accounts_with_actual_secure_phone(accounts, request_id, ignore_errors=False):
    """
    Возвращает те аккаунты из списка, к которым секьюрный номер был привязан
    после момента последней смены владельца номера (по данным PhoneSquatter).
    У всех аккаунтов должен быть одинаковый секьюрный номер.
    """
    all_secure_phone_numbers = [account.phones.secure.number.e164 for account in accounts]
    if not all_secure_phone_numbers:
        return []
    if len(set(all_secure_phone_numbers)) > 1:
        raise ValueError('All passed accounts must have the same secure phone number')
    secure_phone_number = all_secure_phone_numbers[0]

    try:
        rv = get_phone_squatter().get_change_status(
            phone_number=secure_phone_number,
            request_id=request_id,
        )
        change_unixtime = rv.get('change_date', 0)
        change_datetime = unixtime_to_datetime(change_unixtime)
        return [
            account
            for account in accounts
            if (
                account.phones.secure.confirmed >= change_datetime or
                account.uid in settings.IGNORE_POSSIBLE_PHONE_OWNER_CHANGE_FOR_UIDS
            )
        ]
    except (PhoneSquatterPhoneNumberNotTrackedError, PhoneSquatterPhoneNumberUntrackableError):
        return []
    except BasePhoneSquatterError:
        if ignore_errors:
            # Если получили ошибку - фолбечимся на безопасный вариант (считаем, что актуальных телефонов нет)
            return []
        else:
            raise
