# -*- coding: utf-8 -*-
import base64
import binascii
from collections import namedtuple
import json
import logging

from passport.backend.core.builders.ufo_api import (
    BaseUfoApiError,
    get_ufo_api,
)
from passport.backend.core.conf import settings
from passport.backend.core.types.phone_number.phone_number import PhoneNumber


log = logging.getLogger('passport.backend.api.common.phone_karma')


PhoneKarma = namedtuple('PhoneKarma', ['unknown', 'white', 'black'])._make([-1, 0, 100])


def extract_karma(phone_stats):
    counter = phone_stats.get('phone_number_counter', 0)
    if counter >= settings.PHONE_KARMA_COUNTER_THRESHOLD:
        return PhoneKarma.black
    return PhoneKarma.white


def load_phone_stats(phone, safe=True):
    status = False
    phone_stats = {}
    try:
        ufo_api = get_ufo_api()
        phone_stats = ufo_api.phones_stats(phone)
        status = True
    except BaseUfoApiError as e:
        log.warning('Failed to load phone stats from UFO: %s: %s', type(e), e)
        if not safe:
            raise

    return status, phone_stats


def parse_phone_stats(phone_stats, safe=True):
    try:
        # В data лежит какой-то блоб, сейчас json(base64(dict)),
        # если его нет, то мы ничего о телефоне не знаем
        data = phone_stats.get('data')
        if not data:
            return {}
        return json.loads(base64.b64decode(data))
    except (TypeError, ValueError, binascii.Error) as e:
        log.warning('Failed to parse phone data-block from UFO: %s: %s', type(e), e)
        if not safe:
            raise
    return None


def get_phone_karma(phone, safe=True):
    phone_digital = phone.digital
    if (phone_digital in settings.PHONE_KARMA_WHITELIST or
            PhoneNumber.in_fakelist(phone, settings.PHONE_NUMBERS_FAKELIST) or
            any(phone_digital.startswith(prefix) for prefix in settings.PHONE_KARMA_WHITELIST_PREFIXES)):
        return PhoneKarma.white

    status, raw_phone_stats = load_phone_stats(phone, safe)
    if not status:
        return PhoneKarma.unknown
    phone_stats = parse_phone_stats(raw_phone_stats, safe)
    if phone_stats is None:
        return PhoneKarma.unknown
    return extract_karma(phone_stats)
