# -*- coding: utf-8 -*-

import logging
import time

from passport.backend.core import validators
from passport.backend.core.exceptions import ValueChanged
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.types.phone_number.phone_number import mask_for_statbox


MIN_STATBOX_PHONE_LENGTH = 8

log = logging.getLogger('passport.api')


def sanitize_phone_number(track, phone_number, country, ignore_phone_compare):
    statbox = StatboxLogger(action='sanitize_phone_number', track_id=track.track_id)
    try:
        v = validators.PhoneNumber().to_python({
            'phone_number': phone_number,
            'country': country,
        })
        parsed_number = v['phone_number']
        if parsed_number.international != phone_number and not ignore_phone_compare:
            raise ValueChanged('differentPhoneNumber',
                               'Normalized number is different to user input',
                               parsed_number.international,
                               'phone_number')
        statbox.bind(sanitize_phone_result=parsed_number.masked_format_for_statbox)
        return parsed_number
    except ValueChanged as e:
        statbox.bind(
            sanitize_phone_error=e.code,
            sanitize_phone_result=mask_for_statbox(phone_number)
        )
        track.sanitize_phone_changed_phone = True
        raise
    except validators.Invalid as e:
        statbox.bind(
            sanitize_phone_error=e.code,
            sanitize_phone_result=mask_for_statbox(phone_number)
        )
        track.sanitize_phone_error = True
        raise
    finally:
        track.sanitize_phone_count.incr()
        if track.sanitize_phone_first_call is None:
            track.sanitize_phone_first_call = time.time()
        track.sanitize_phone_last_call = time.time()

        if len(phone_number) >= MIN_STATBOX_PHONE_LENGTH:
            statbox.log()


__all__ = (
    'MIN_STATBOX_PHONE_LENGTH',
    'sanitize_phone_number',
)
