# -*- coding: utf-8 -*-

import logging

from namedlist import namedlist
from passport.backend.core.builders.tensornet import (
    BaseTensorNetError,
    get_tensornet,
)
from passport.backend.core.conf import settings
from passport.backend.core.utils.experiments import is_experiment_enabled_by_uid

from .features import (
    FeaturesBuilderProfileV1,
    FeaturesBuilderProfileV2,
    FeaturesBuilderProfileV3,
)


log = logging.getLogger('passport.backend.api.common.profile.estimate')

EstimateResult = namedlist('EstimateResult', 'status estimate model threshold', default=None)


BUILDERS_BY_VERSION = {
    1: FeaturesBuilderProfileV1,
    2: FeaturesBuilderProfileV2,
    3: FeaturesBuilderProfileV3,
}


def get_auth_model_for_uid(uid):
    # Сортируем доступные модели по убыванию деноминатора
    model_configs = sorted(
        settings.TENSORNET_MODEL_CONFIGS.items(),
        key=lambda model_and_config: model_and_config[1]['denominator'],
        reverse=True,
    )
    for model, config in model_configs:
        if is_experiment_enabled_by_uid(uid, config['denominator']):
            return model
    return model


def get_features_builder(model_name):
    builder_version = settings.TENSORNET_MODEL_CONFIGS[model_name]['features_builder_version']
    builder = BUILDERS_BY_VERSION[builder_version]()
    return builder


def estimate(uid, **kwargs):
    model = get_auth_model_for_uid(uid)
    features_builder = get_features_builder(model)
    features = features_builder.extract_features(**kwargs)
    threshold = settings.TENSORNET_MODEL_CONFIGS[model]['threshold']
    try:
        tensornet = get_tensornet(model)
        estimate = tensornet.eval(features)
        return EstimateResult(status=True, estimate=estimate, model=model, threshold=threshold)
    except BaseTensorNetError as e:
        log.error('TensorNet prediction failed (model %s): %s.', model, e)
        return EstimateResult(status=False, estimate=None, model=model, threshold=threshold)
