# -*- coding: utf-8 -*-

from passport.backend.api.common.profile.utils import (
    browser_grouper,
    get_host,
    parse_uatraits_bool_missing,
    probability,
    probability_grouped,
)
from passport.backend.core.env.env import parse_uatraits_value
from passport.backend.core.geobase import Region
from passport.backend.utils.time import (
    unixtime_to_datetime,
    Weekday,
)


# Данные извлекаются согласно коду yandex-passport-profile


class FeaturesBuilderBase(object):
    def extract_features(self, *args, **kwargs):
        raise NotImplementedError()  # pragma: no cover


class FeaturesBuilderProfileV1(FeaturesBuilderBase):

    COUNT_FEATURE_TIMERANGES = ('1d', '1w', '1m')

    # Факторы-счетчики, берутся как есть из профиля
    PROFILE_COUNT_FEATURES = (
        'succ_auth_count',
        'captcha_passed',
        'country_count',
        'city_count',
        'browser_name_count',
        'os_name_count',
        'yandexuid_count',
        'retpath_host_3_count',
        'referer_host_3_count',
    )

    PROB_FEATURE_TIMERANGES = ('1d', '1w', '1m')

    # Факторы, для которых считаются вероятности
    PROFILE_CATEG_FEATURES_WITH_PROBABILITY = (
        'browser',
        'browser_name',
        'browser_os',
        'os_name',
        'os_family',
        'ip',
        'city',
        'country',
        'day_part',
        'weekday',
        'retpath_host_2',
        'retpath_host_3',
        'referer_host_2',
        'referer_host_3',
        'yandexuid',
        'as_list',
    )

    def extract_features(self, fresh_profile, full_profile, retpath=None, referer=None):
        full_profile = full_profile or {}
        env = fresh_profile.raw_env
        ip = env['ip']
        user_agent_info = env['user_agent_info']
        yandexuid = env['yandexuid']
        yandexuid_timestamp = fresh_profile.yandexuid_timestamp
        timestamp = fresh_profile.timestamp
        region = Region(ip=ip)

        all_features = {
            'ip': ip,
            'retpath_host_2': get_host(retpath, 2),
            'retpath_host_3': get_host(retpath, 3),
            'referer_host_2': get_host(referer, 2),
            'referer_host_3': get_host(referer, 3),
            'yandexuid': yandexuid,
            'yandexuid_ts': yandexuid_timestamp,
            'yandexuid_ts_freshness': timestamp - yandexuid_timestamp if yandexuid_timestamp else -1,
            'is_yandexuid_ts_future': int(yandexuid_timestamp > timestamp) if yandexuid_timestamp else -1,
        }

        all_features.update(self.extract_geo_features(region))
        all_features.update(self.extract_unixtime_features(timestamp))
        all_features.update(self.extract_user_agent_features(user_agent_info))
        all_features.update(self.extract_profile_based_features(all_features, full_profile))

        return all_features

    def extract_user_agent_features(self, user_agent_info):
        ua_data = {
            'is_mobile': parse_uatraits_bool_missing(user_agent_info.get('isMobile')),
            'os_name': parse_uatraits_value(user_agent_info.get('OSName', user_agent_info.get('OSFamily'))),
        }
        for target_name, src_name in [
            ('browser_name', 'BrowserName'),
            ('browser_version', 'BrowserVersion'),
            ('os_family', 'OSFamily'),
            ('os_version', 'OSVersion'),
        ]:
            ua_data[target_name] = parse_uatraits_value(user_agent_info.get(src_name))
        ua_data['browser'] = str(ua_data['browser_name']) + " " + str(ua_data['browser_version'])
        ua_data['browser_os'] = str(ua_data['browser']) + " - " + str(ua_data['os_name'])
        return ua_data

    def extract_geo_features(self, region):
        as_list = region.AS_list
        return {
            'is_black_ip': 0,
            'geo_id': region.id if region.id is not None else -1,
            'country': region.country['id'] if region.country else 0,
            'city': region.city['id'] if region.city else 0,
            'as_list': ','.join(as_list) if isinstance(as_list, list) else as_list,
        }

    def extract_unixtime_features(self, unixtime):
        date_time = unixtime_to_datetime(unixtime)
        weekday = date_time.isoweekday()
        hour = date_time.hour
        month = date_time.month

        return {
            'day_part': hour // 6,  # делим сутки на 4 части (утро, день, вечер, ночь)
            'weekday': weekday,
            'is_weekend': int(weekday in (Weekday.SATURDAY, Weekday.SUNDAY)),
            'hour': hour,
            'month': month,
        }

    def extract_profile_based_features(self, categ_values, full_profile):
        features = {}
        for feature in self.PROFILE_COUNT_FEATURES:
            for timerange in self.COUNT_FEATURE_TIMERANGES:
                feature_name = '%s_%s' % (feature, timerange)
                features[feature_name] = full_profile.get(feature_name, 0)

        for feature in self.PROFILE_CATEG_FEATURES_WITH_PROBABILITY:
            for timerange in self.PROB_FEATURE_TIMERANGES:
                target_feature_name = '%s_prob_%s' % (feature, timerange)
                value_dict_name = '%s_freq_%s' % (feature, timerange)
                value_dict = full_profile.get(value_dict_name)
                categ_value = categ_values[feature]
                if feature == 'as_list' and categ_value is not None:
                    categ_value = categ_value.split(',')
                features[target_feature_name] = probability(categ_value, value_dict)

                if feature == 'browser':
                    value = probability_grouped(browser_grouper)(categ_value, value_dict)
                    features['browser_grouped_prob_%s' % timerange] = value

        return features


class FeaturesBuilderProfileV2(FeaturesBuilderProfileV1):
    SU_PROB_FEATURE_TIMERANGES = ('1d', '1w', '1m', '3m')

    FEATURES_WITH_PROBABILITY_FOR_SU = (
        'ip',
        'country',
        'city',
        'as_list',
        'day_part',
        'weekday',
        'browser',
        'browser_name',
        'is_mobile',
        'browser_os',
    )

    PROB_FEATURE_TIMERANGES = ('1d', '1w', '1m', '3m')

    def extract_profile_based_features(self, categ_values, full_profile):
        features = super(FeaturesBuilderProfileV2, self).extract_profile_based_features(categ_values, full_profile)

        for feature in self.FEATURES_WITH_PROBABILITY_FOR_SU:
            for timerange in self.SU_PROB_FEATURE_TIMERANGES:
                target_feature_name = 'su_%s_prob_%s' % (feature, timerange)
                value_dict_name = 'su_%s_freq_%s' % (feature, timerange)
                value_dict = full_profile.get(value_dict_name)
                categ_value = categ_values[feature]
                if feature == 'as_list' and categ_value is not None:
                    categ_value = categ_value.split(',')
                features[target_feature_name] = probability(categ_value, value_dict)

        return features


class FeaturesBuilderProfileV3(FeaturesBuilderProfileV2):
    COUNT_FEATURE_TIMERANGES = ('1d',)
    SU_PROB_FEATURE_TIMERANGES = ('1d', '6m')
    PROB_FEATURE_TIMERANGES = ('1d', '6m')
