# -*- coding: utf-8 -*-
from functools import total_ordering

from passport.backend.core.utils.version import parse_am_version
from passport.backend.core.validators.validators import DEVICE_ID_REGEX
import six
from six.moves import zip_longest
from six.moves.urllib.parse import urlparse
from statbox_bindings2.string_utils.misc import canonize_vhost


# Скопипащено как есть из yandex-passport-profile


def get_host(url, parts=None):
    if not url:
        return
    try:
        host = canonize_vhost(urlparse(url).netloc)
        if six.PY3:
            host = host.decode('utf8')
        if parts:
            # отрезаем домен нужного уровня
            host = '.'.join(host.split('.')[-1 * parts:])

        return host
    except ValueError:
        return


@total_ordering
class Version(object):
    @staticmethod
    def parse(version):
        if not version:
            return None

        try:
            version_parts = [int(part) for part in version.split('.')]
        except ValueError:
            return None

        return Version(version_parts)

    def __init__(self, version_parts):
        self.version_parts = version_parts

    def __gt__(self, other):
        if not isinstance(other, Version):
            raise ValueError('Cannot compare Version with %s' % type(other))

        for version_part1, version_part2 in zip_longest(self.version_parts, other.version_parts):
            if version_part1 != version_part2:
                return version_part1 is not None and (version_part2 is None or version_part1 > version_part2)
        return False

    def __eq__(self, other):
        if not isinstance(other, Version):
            raise ValueError('Cannot compare Version with %s' % type(other))

        for version_part1, version_part2 in zip_longest(self.version_parts, other.version_parts):
            if version_part1 != version_part2:
                return False
        return True


def parse_browser(browser):
    browser_parts = browser.split(' ')
    if len(browser_parts) == 2:
        browser_name = browser_parts[0]
        browser_version = Version.parse(browser_parts[1])
        return browser_name, browser_version
    elif len(browser_parts) == 1:
        return browser_parts[0], None
    else:
        return None, None


def parse_uatraits_bool_missing(value):
    if value == 'Unknown':
        return -1
    return int(bool(value))


def probability_grouped(grouper):
    def wrapper(value, value_freq):
        return probability(value, value_freq, grouper)
    return wrapper


def browser_grouper(value, value_freq):
    current_browser_name, current_browser_version = parse_browser(value)
    if current_browser_name and current_browser_version:
        count = 0
        for browser, browser_count in value_freq:
            browser_name, browser_version = parse_browser(browser)
            if (browser_name == current_browser_name and
                    current_browser_version is not None and browser_version is not None and
                    current_browser_version >= browser_version):
                count += browser_count
        return count
    else:
        return 0


def probability(value, value_freq, grouper=None, round_digits=4):
    """
    Посчитать вероятность появления заданного значения (набора значений), используя известное распределение.
    Распределение (value_freq) задается как список пар (значение, количество значений).
    grouper - опциональный способ определения числа вхождений значения.
    """
    if not value_freq:
        return 0

    value_freq_dict = dict(value_freq)
    if isinstance(value, (list, tuple)):
        result = float(sum([
            grouper(elem, value_freq) if grouper is not None else value_freq_dict.get(elem, 0)
            for elem in value
        ])) / sum(value_freq_dict.values())
    else:
        value_count = grouper(value, value_freq) if grouper is not None else value_freq_dict.get(value, 0)
        result = float(value_count) / sum(value_freq_dict.values())

    return round(result, round_digits)


def extract_mobile_params_from_track(track):
    mobile_params = dict()
    if track.account_manager_version is not None:
        mobile_params.update(
            am_version='.'.join([str(x) for x in parse_am_version(track.account_manager_version)]),
        )
    if track.device_id is not None and DEVICE_ID_REGEX.match(track.device_id) is not None:
        mobile_params.update(
            device_id=track.device_id,
        )
    if track.cloud_token is not None:
        mobile_params.update(
            cloud_token=track.cloud_token,
        )
    return mobile_params
