# -*- coding: utf-8 -*-

import logging

from passport.backend.core.avatars import upload_avatar
from passport.backend.core.builders.social_api import (
    BaseSocialApiError,
    get_social_api,
)
from passport.backend.core.builders.social_broker import get_max_size_avatar
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.helpers import trim_message
from passport.backend.core.types.social_business_info import BusinessInfo


log = logging.getLogger(__name__)


def get_profiles_with_person_by_uid(uid, expand_provider=False):
    try:
        profiles = get_social_api().get_profiles_by_uid(
            uid,
            person=True,
            expand_provider=expand_provider,
        )
        return True, profiles
    except BaseSocialApiError:
        return False, []


def get_unique_social_profiles_by_task_ids(task_ids):
    """
    По списку task_id, найти уникальные соц. профили (соц. аккаунты, привязанные к какому-либо аккаунту Яндекса).
    В ответе возвращает кортеж:
     - признак успешности вызовов социального API;
     - список данных аккаунтов в соц. сетях. Данные получены по task_id (возможны дубликаты);
     - список данных соц. профилей (т.е. соц. аккаунтов, привязанных к какому-либо аккаунту в Яндексе), элементы
     списка уникальны (уникальным является соответствие соц. аккаунта и аккаунта в Яндексе).
    """
    social_api = get_social_api()
    social_account_info = set()
    api_calls_succeeded = True

    social_accounts_list = []
    for task_id in task_ids:
        try:
            # Сначала по task_id получаем данные, собранные брокером
            task_data = social_api.get_task_data(task_id)
            user_id = task_data['profile']['userid']
            provider_id = task_data['profile']['provider']['id']
            business_info = BusinessInfo.from_dict(task_data['profile'].get('business'))

            social_account_info.add((provider_id, user_id, business_info))
            social_accounts_list.append(task_data['profile'])
        except BaseSocialApiError:
            api_calls_succeeded = False
    all_profiles = []
    for provider_id, user_id, business_info in social_account_info:
        try:
            profiles = social_api.get_profiles(
                user_id,
                provider_id=provider_id,
                person=True,
                business_info=business_info,
            )
            all_profiles.extend(profiles)
        except BaseSocialApiError:
            api_calls_succeeded = False
    return api_calls_succeeded, social_accounts_list, all_profiles


def is_social_auth_available(account):
    return not (
        account.is_strong_password_required or
        account.totp_secret.is_set
    )


def try_get_social_profiles_for_auth(account):
    profiles = []
    if not settings.IS_INTRANET and is_social_auth_available(account):
        try:
            all_profiles = get_social_api().get_profiles(uid=account.uid, allow_auth=True)
        except BaseSocialApiError as e:
            log.warning('Failed to get social profiles: %s', trim_message(str(e)))
            all_profiles = []
        profiles = [
            profile
            for profile in all_profiles
            if profile['provider_code'] in settings.AUTH_ALLOWED_PROVIDERS
        ]
    return profiles


def get_max_size_avatar_from_profile(profile):
    avatars = profile.get('avatar')
    if avatars:
        if not isinstance(avatars, dict):
            raise SocialInvalidAvatarError()
        max_size_avatar = get_max_size_avatar(avatars)
        if not max_size_avatar:
            raise SocialInvalidAvatarError()
        return max_size_avatar


def get_max_size_avatar_from_profile_fail_safe(*args, **kwargs):
    try:
        return get_max_size_avatar_from_profile(*args, **kwargs)
    except SocialInvalidAvatarError as e:
        log.warning('Social avatar has unexpected format! Error: %s', trim_message(str(e)))


def set_avatar_from_social_profile(uid, profile):
    log.info('Setting user avatar from social profile...')
    max_size_avatar = get_max_size_avatar_from_profile(profile)
    if max_size_avatar:
        upload_avatar(uid, avatar_url=max_size_avatar)


class SocialInvalidAvatarError(Exception):
    pass
