# -*- coding: utf-8 -*-
from flask import request
from passport.backend.core.builders import blackbox
from passport.backend.core.conf import settings
from passport.backend.core.geobase import Region
from passport.backend.core.language_detect import (
    LanguageData,
    MobileLanguageData,
    OctopusLanguageData,
)
from passport.backend.core.portallib import is_yandex_ip
from passport.backend.core.suggest.login_suggester import LoginSuggester
from passport.backend.core.types.login.login import (
    is_test_yandex_login,
    login_is_kiddish,
    login_is_kolonkish,
    login_is_neophonish,
    login_is_phonish,
    login_is_social,
    login_is_yambot,
    normalize_login,
)
import pytz

from .common import extract_tld


def get_language_suggest():
    language_data = LanguageData(
        ip=str(request.env.user_ip),
        cookiemy=request.env.cookies.get('my', ''),
        host=request.env.host,
        accept_language=request.env.accept_language or '',
    )
    return language_data.language


def get_mobile_language_suggest():
    language_data = MobileLanguageData(
        ip=str(request.env.user_ip),
        accept_language=request.env.accept_language or '',
    )
    return language_data.language


def get_octopus_language_suggest(display_language):
    language_data = OctopusLanguageData(
        ip=str(request.env.user_ip),
        display_language=display_language or '',
    )
    return language_data.language


def get_countries_suggest():
    ret = list()

    if request.env.host:
        tld = extract_tld(request.env.host, settings.PASSPORT_TLDS)
        if tld:
            country = Region(tld=tld).country
            if country is not None and country['short_en_name']:
                ret.append(country['short_en_name'].lower())

    user_ip = request.env.user_ip
    country = Region(ip=user_ip).country
    if country is not None and country['short_en_name']:
        if country['short_en_name'].lower() not in ret:
            ret.append(country['short_en_name'].lower())

    if not ret:
        ret = [settings.DEFAULT_COUNTRY]
    return ret


def safe_detect_timezone(user_ip):
    timezone = Region(ip=user_ip).timezone
    if not timezone:
        return

    try:
        return pytz.timezone(timezone)
    except pytz.UnknownTimeZoneError:
        return


def sieve_available_logins(original_login, logins):
    """Выбираем из списка переданных логинов только пригодные для регистрации"""
    available_logins = []
    from_yandex_nets = is_yandex_ip(str(request.env.user_ip))
    # В отличие от валидатора Availability, не проверяем гранты на каждый префикс
    for login in logins:
        if (
            login_is_kiddish(login) or
            login_is_kolonkish(login) or
            login_is_neophonish(login) or
            login_is_phonish(login) or
            login_is_social(login) or
            login_is_yambot(login)
        ):
            continue

        if is_test_yandex_login(login) and not from_yandex_nets:
            continue

        # Могли сгенерить сами такой же
        user_normalized_login = normalize_login(original_login)
        if user_normalized_login == normalize_login(login):
            continue

        available_logins.append(login)

    if not available_logins:
        return []

    response = blackbox.get_blackbox().loginoccupation(
        available_logins,
        ignore_stoplist=False,
        is_pdd=False,
    )

    available_logins = [
        login
        for login in available_logins
        if response[login]['status'] == 'free'
    ]

    return available_logins


def get_login_suggestions(original_login, login, firstname, lastname, language):
    """
    Генерим логины, проверяем их в ЧЯ на незанятость пачками,
    пока не наберем нужное количество
    :rtype: list
    """
    suggester = LoginSuggester(
        first_name=firstname,
        last_name=lastname,
        login=login,
        language=language,
    )
    suggestions = []

    for _ in range(settings.LOGIN_SUGGEST_MAX_ITERATIONS):
        pack = suggester.next_pack()
        if not pack:
            break
        pack = sieve_available_logins(original_login, pack)
        suggestions.extend(pack)
        if len(suggestions) >= settings.LOGIN_SUGGESTIONS_LIMIT:
            break

    return suggestions[:settings.LOGIN_SUGGESTIONS_LIMIT]
