# -*- coding: utf-8 -*-
from collections import OrderedDict
from copy import deepcopy
import logging

import elementflow
from flask import (
    request,
    Response,
)
from passport.backend.api.common.errors import log_internal_error
from passport.backend.api.common.format_response import XmlLoggedResponse
from passport.backend.api.common.logs import setup_log_prefix
from passport.backend.api.email_validator.exceptions import (
    EmailAlreadyConfirmedError,
    EmailAlreadySentError,
    EmailGenericError,
    EmailGrantsError,
    EmailIncorrectKeyError,
    EmailIsNativeError,
    EmailIsNotRpopError,
    EmailIsNotUnsafeError,
)
from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    BaseBundleError,
    SessionidInvalidError,
    UnhandledBundleError,
    ValidationFailedError,
)
from passport.backend.api.views.bundle.mixins import BundleAccountGetterMixin
from passport.backend.core.grants import (
    get_grants_config,
    Grants,
    MissingRequiredGrantsError,
    MissingTicketError,
    UnknownConsumerError,
)
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.types.email.email import mask_email_for_statbox
from passport.backend.core.utils.decorators import cached_property
from passport.backend.utils.string import smart_text
from six import (
    BytesIO,
    iteritems,
)


UNSAFE_ADDRESS_GRANT = 'email_validator.unsafe'
EMPTY_RESPONSE = Response(
    b'<?xml version="1.0" encoding="windows-1251"?>\n<page>\n</page>',
    headers={
        'Content-Type': 'text/xml; charset=windows-1251',
    },
)


logger = logging.getLogger(__name__)


def determine_consumers_by_ip(grants_config, ip):
    return grants_config.get_consumers(ip)


def determine_validator_error_response(exception, raw_values):
    '''
    У email-валидатора свой специфичный вариант сообщений об ошибках,
    на который не очень просто отобразить существующие бандловые ошибки.
    К примеру, здесь нужно дополнять сообщения об ошибках значениями
    из полей формы.

    Здесь мы пытаемся обернуть ошибку в подходящую из числа уже реализованных
    в существующем валидаторе, если же нет - выкидываем безликую ошибку,
    по кодам в которой сможем понять что пошло не так.
    '''
    if isinstance(exception, SessionidInvalidError):
        return ValidatorResponse({
            'tag': 'validator-invalid-argument',
            'attrs': OrderedDict([
                ('sessionid', raw_values['sessionid']),
                ('address', raw_values.get('email', '')),
            ]),
        })

    if isinstance(exception, BaseBundleError):
        # Некоторые ошибки надо подавать особым образом,
        # добавляя в ответ поля из формы. Обрабатываем
        # только самый первый код.
        specific_errors = {
            EmailIsNativeError.error: ('invalid-argument', []),
            EmailGrantsError.error: ('no-grants', ['missing']),
            EmailAlreadyConfirmedError.error: ('key-already-validated', ['uid', 'address']),
            EmailIncorrectKeyError.error: ('key-error', ['uid']),
            EmailIsNotUnsafeError.error: ('invalid-argument', ['uid', 'address']),
            EmailIsNotRpopError.error: ('invalid-argument', ['uid', 'address']),
            EmailGenericError.error: ('error', []),
            EmailAlreadySentError.error: ('already-sent-error', ['retry-in']),
        }

        for error_code in exception.errors:
            if error_code in specific_errors:
                validator_error_code, exc_fields = specific_errors[error_code]

                specific_response = {
                    'tag': 'validator-%s' % validator_error_code,
                    'attrs': OrderedDict([
                        (field, smart_text(getattr(exception, field, '')))
                        for field in exc_fields
                    ]),
                }

                message = getattr(exception, 'message')
                if message:
                    specific_response['text'] = message
                return ValidatorResponse(specific_response)

    if hasattr(exception, 'errors'):
        # Определенные ошибки валидации форм надо обрабатывать отдельно.
        specific_form_errors = {
            ('key.empty', 'key.invalid'): 'Bad key!',
            ('uid.invalid',): 'Bad uid: %(uid)s',
            ('email.invalid',): 'Bad e-mail format: %(email)s',
        }

        for error_code in exception.errors:
            for codes, message in iteritems(specific_form_errors):
                if error_code in codes:
                    return ValidatorResponse({
                        'tag': 'validator-invalid-argument',
                        'text': message % raw_values,
                    })

    # Если пропущено хотя бы одно из обязательных полей,
    # то тоже нужно выдать пустой ответ.
    if isinstance(exception, ValidationFailedError):
        are_fields_missing = [
            e.endswith('.empty')
            for e in exception.errors
        ]
        if are_fields_missing:
            return EMPTY_RESPONSE


def dispatch_operations(operations):
    def dispatch_request():
        op_code = request.values.get('op')
        operation = operations.get(op_code)

        if not op_code or not operation:
            logger.warning(u'"{}" is an unknown operation code'.format(op_code))
            return EMPTY_RESPONSE

        return operation().dispatch_request()
    return dispatch_request


def validator_xml_response(values):
    body = BytesIO()
    with elementflow.xml(body, 'page', indent=True) as page:
        page.element(
            values['tag'],
            text=values.get('text', ''),
            attrs=values.get('attrs', {}),
        )
    xml = body.getvalue()

    # FIXME: Жуткий хак, потому что elementflow не позволяет задать кодировку получаемого XML.
    xml = xml.replace(
        b'<?xml version="1.0" encoding="utf-8"?>',
        b'<?xml version="1.0" encoding="windows-1251"?>',
    )
    return xml


class ValidatorResponse(XmlLoggedResponse):
    charset = 'windows-1251'

    def __init__(self, response, format_=validator_xml_response, sensitive_fields=None):
        data = deepcopy(response)
        self.formatted_response = format_(data)

        if 'address' in data.get('attrs', {}):
            data['attrs']['address'] = mask_email_for_statbox(data['attrs']['address'])

        self.formatted_masked_response = format_(data)
        self._error_code = response.get('tag')
        super(XmlLoggedResponse, self).__init__(self.formatted_response)


class BaseEmailValidatorView(BaseBundleView,
                             BundleAccountGetterMixin):

    def process_basic_form(self):
        super(BaseEmailValidatorView, self).process_basic_form()

        if self.form_values.get('unsafe') is False:
            self.check_grant(lambda: [UNSAFE_ADDRESS_GRANT])

    @cached_property
    def grants(self):
        return Grants()

    @cached_property
    def grants_config(self):
        return get_grants_config()

    def check_grant(self, grant_source):
        required = grant_source()
        try:
            if not self.consumer:
                possible_consumers = determine_consumers_by_ip(
                    self.grants_config,
                    request.env.consumer_ip,
                )
                self.grants.check_access(
                    request.env.consumer_ip,
                    consumers=possible_consumers,
                    required_grants=required,
                )
            else:
                super(BaseEmailValidatorView, self).check_grant(grant_source)
        except (UnknownConsumerError, MissingRequiredGrantsError, MissingTicketError):
            raise EmailGrantsError(required)

    @cached_property
    def statbox(self):
        return StatboxLogger(
            mode='legacy_email_validator',
            consumer=self.consumer,
        )

    def ok_response(self, **values):
        return ValidatorResponse(
            values,
            validator_xml_response,
            sensitive_fields=['attrs.address'],
        )

    def process_root_form(self):
        self.consumer = self.all_values.get('from')

    def respond_error(self, exception):
        setup_log_prefix(self.account)

        # В некоторых случаях, нам необходимо выдать очень специфический
        # ответ на вполне простые ошибки.
        specific_error_response = determine_validator_error_response(
            exception,
            self.all_values,
        )
        if specific_error_response:
            return specific_error_response

        # Обработаем ошибки, получив предусмотренное view-исключение
        processed_error = self.process_error(exception)
        errors = processed_error.errors

        # Запишем в лог необработанное исключение
        if isinstance(processed_error, UnhandledBundleError):
            log_internal_error(exception)

        return ValidatorResponse({
            'tag': 'validator-error',
            'text': 'Bad things happened: %s' % ', '.join(errors),
        })

    def respond_success(self):
        """
        Генерирует и отдаёт успешный ответ с данными из response_values.

        @return: Объект ValidatorResponse.
        """
        if self.state:
            self.state.update_response(self.response_values)

        setup_log_prefix(self.account)
        return self.ok_response(**self.response_values)

    def _get_session_from_form_and_host(self):
        """
        Костыль для случая передачи куки Session_id и хоста в форме
        """
        sessionid_cookie = self.form_values.get('sessionid')
        host = self.form_values.get('host')
        return sessionid_cookie, None, host
