# -*- coding: utf-8 -*-

from datetime import datetime
import logging

from passport.backend.api.email_validator.exceptions import (
    EmailAlreadyConfirmedError,
    EmailAlreadySentError,
    EmailGenericError,
    EmailIncorrectKeyError,
    EmailIsNativeError,
    EmailLimitPerProfileReachedError,
)
from passport.backend.api.email_validator.utils import send_validation_message
from passport.backend.core.conf import settings
from passport.backend.core.counters import (
    email_per_uid_and_address_counter,
    email_per_uid_counter,
)
from passport.backend.core.models.email import Email
from passport.backend.core.models.persistent_track import (
    PersistentTrack,
    TRACK_TYPE_EMAIL_CONFIRMATION_CODE,
)
from passport.backend.core.runner.context_managers import CREATE
from passport.backend.utils.common import generate_random_code


log = logging.getLogger('passport.email_validator.mixins')


class EmailValidatorMixin(object):
    def create_email_confirmation_track(self, email, ttl=None):
        short_code = generate_random_code(settings.EMAIL_VALIDATOR_SHORT_CODE_LENGTH)
        confirmation_track = PersistentTrack.create(
            self.account.uid,
            TRACK_TYPE_EMAIL_CONFIRMATION_CODE,
            expires_after=ttl or settings.EMAIL_VALIDATOR_CONFIRMATION_CODE_TTL,
            address=email.address,
            short_code=short_code,
        )
        events = dict(action='validator')
        with CREATE(confirmation_track, self.request.env, events):
            pass
        return confirmation_track

    def check_email_confirmation_code(self, code):
        bb_response = self.blackbox.get_track(
            self.account.uid,
            code,
        )

        track = PersistentTrack().parse(bb_response)
        if not track.type == TRACK_TYPE_EMAIL_CONFIRMATION_CODE or not track.content:
            raise EmailIncorrectKeyError(self.account.uid, code)

        email = self.account.emails.find(track.content.get('address'))
        if not email:
            raise EmailIncorrectKeyError(self.account.uid, code)

        if email.confirmed_at:
            raise EmailAlreadyConfirmedError(
                email.address,
                self.account.uid,
                code,
            )

        return email

    def create_confirmed_email(self, address, is_unsafe=False):
        return Email(
            address=address,
            created_at=datetime.now(),
            confirmed_at=datetime.now(),
            bound_at=datetime.now(),
            is_unsafe=is_unsafe,
        )

    def check_email_not_native(self, address):
        uid, email_address = self.blackbox.get_single_email(
            self.consumer_ip,
            address,
            uid=self.account.uid,
        )

        email = Email().parse(email_address) if email_address else None
        if email and email.is_native:
            raise EmailIsNativeError(address)

    def _email_counters_hit_limit(self, email_address):
        return (
            email_per_uid_counter.hit_limit(self.account.uid) or
            email_per_uid_and_address_counter.hit_limit(email_address, self.account.uid)
        )

    def _increase_email_counters(self, email_address):
        email_per_uid_and_address_counter.incr(email_address, self.account.uid)
        email_per_uid_counter.incr(self.account.uid)

    def check_if_too_many_validation_messages_sent(self, email):
        if self._email_counters_hit_limit(email.address):
            raise EmailAlreadySentError(email.address)
        self._increase_email_counters(email.address)

    def send_validation_message(self, account, address, code,
                                short_code=None, tld=None,
                                language=None, retpath=None, code_only=False):
        send_validation_message(
            account,
            address,
            code,
            short_code=short_code,
            tld=tld,
            lang=language,
            retpath=retpath,
        )

    def validate_by_email(self, address, force=False, tld=None,
                          language=None, retpath=None, is_unsafe=True, code_only=False):

        self.check_email_not_native(address)

        email = self.account.emails.find(address)

        if not email:
            if len(self.account.emails) >= settings.MAX_EMAILS_PER_PROFILE:
                raise EmailLimitPerProfileReachedError()
            email = Email()
            email.address = address
            email.created_at = datetime.now()

            # В старой версии unsafe всегда устанавливался True, но по
            # просьбе spleenjack@ добавляем возможность установить этот
            # флаг и сюда.
            email.is_unsafe = is_unsafe

            log.debug(
                u'Email %s added as unconfirmed for uid=%s.',
                address,
                self.account.uid,
            )
            self.account.emails.add(email)
        else:
            if email.confirmed_at:
                raise EmailGenericError()

        if not force:
            self.check_if_too_many_validation_messages_sent(email)

        confirmation_track = self.create_email_confirmation_track(email)
        self.send_validation_message(
            self.account,
            email.address,
            confirmation_track.track_id,
            short_code=confirmation_track.content['short_code'],
            tld=tld,
            language=language,
            retpath=retpath,
            code_only=code_only,
        )
        return confirmation_track
