# -*- coding: utf-8 -*-
from passport.backend.core.conf import settings
from passport.backend.core.mailer.utils import (
    get_tld_by_country,
    login_shadower,
    MailInfo,
    make_email_context,
    send_mail_for_account,
)
from passport.backend.core.models.account import get_preferred_language
from six.moves.urllib.parse import (
    urlencode,
    urlsplit,
    urlunparse,
)


def construct_validation_url(domain, validation_code, retpath=None):
    path_info = urlsplit(settings.PASSPORT_BASE_URL_TEMPLATE % dict(tld=domain))

    query = {
        'key': validation_code,
        'mode': 'changeemails',
    }
    if retpath:
        query['retpath'] = retpath

    return urlunparse((
        path_info[0],
        path_info[1],
        'passport',
        '',
        urlencode(query),
        '',
    ))


def send_validation_message(account, address, code,
                            short_code=None, tld=None,
                            lang=None, retpath=None):
    language = get_preferred_language(account, lang)
    phrases = settings.translations.VALIDATOR[language]
    user_tld = tld or get_tld_by_country(account.person.country)

    url = construct_validation_url(
        user_tld,
        code,
        retpath=retpath,
    )

    context = make_email_context(
        language=language,
        account=account,
        context={
            'VALIDATION_URL': url,
            'TLD': user_tld,
            'SHORT_CODE': short_code,
        },
    )

    send_mail_for_account(
        'mail/email_validation_link_message.html',
        MailInfo(
            subject=phrases['subject_validation'],
            from_=phrases['from_name'],
            tld=user_tld,
        ),
        context,
        account,
        context_shadower=login_shadower,
        is_plain_text=False,
        send_to_native=False,
        send_to_external=False,
        specific_email=address,
    )


def send_address_validated_message(account, address, tld=None, lang=None):
    language = lang or get_preferred_language(account)
    phrases = settings.translations.VALIDATOR[language]
    user_tld = tld or get_tld_by_country(account.person.country)

    context = make_email_context(
        language=language,
        account=account,
        context={
            'ADDRESS': address,
        },
    )

    send_mail_for_account(
        'mail/email_confirmation_message.html',
        MailInfo(
            subject=phrases['subject_confirmation'],
            from_=phrases['from_name'],
            tld=user_tld,
        ),
        context,
        account,
        context_shadower=login_shadower,
        is_plain_text=False,
        send_to_native=True,
        send_to_external=True,
    )


def send_address_deleted_message(account, email, tld=None, lang=None):
    if not email.is_confirmed:
        return

    language = lang or get_preferred_language(account)
    phrases = settings.translations.VALIDATOR[language]
    user_tld = tld or get_tld_by_country(account.person.country)

    context = make_email_context(
        language=language,
        account=account,
        context={
            'TLD': user_tld,
            'ADDRESS': email.address,
        },
    )

    info = MailInfo(
        subject=phrases['subject_deletion'],
        from_=phrases['from_name'],
        tld=user_tld,
    )

    send_mail_for_account(
        'mail/email_deletion_message.html',
        info,
        context,
        account,
        context_shadower=login_shadower,
        is_plain_text=False,
    )

    send_mail_for_account(
        'mail/email_deletion_message.html',
        info,
        context,
        account,
        context_shadower=login_shadower,
        is_plain_text=False,
        send_to_native=False,
        send_to_external=False,
        specific_email=email.address,
    )
