# -*- coding: utf-8 -*-

from datetime import datetime
import logging

from passport.backend.api.email_validator.base import (
    BaseEmailValidatorView,
    dispatch_operations,
)
from passport.backend.api.email_validator.exceptions import (
    EmailIsNotRpopError,
    EmailIsNotUnsafeError,
    EmailLimitPerProfileReachedError,
)
from passport.backend.api.email_validator.forms import (
    AddRPOPForm,
    ConfirmEmailByCodeForm,
    DeleteAllEmailsForm,
    DeleteEmailForm,
    DeleteRpopForm,
    DeleteUnsafeForm,
    GetConfirmationCodeForm,
    ValidateByEmailForm,
)
from passport.backend.api.email_validator.mixins import EmailValidatorMixin
from passport.backend.api.email_validator.utils import (
    send_address_deleted_message,
    send_address_validated_message,
)
from passport.backend.api.views.bundle.mixins.kolmogor import KolmogorMixin
from passport.backend.api.views.bundle.mixins.push import BundlePushMixin
from passport.backend.core.conf import settings
from passport.backend.core.models.email import Email
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.types.email.email import mask_email_for_statbox


log = logging.getLogger('passport.backend.api.email_validator.views')


class AddRPOPView(
    BaseEmailValidatorView,
    EmailValidatorMixin,
    BundlePushMixin,
    KolmogorMixin,
):
    basic_form = AddRPOPForm
    required_grants = [
        'email_validator.api_addrpop',
    ]

    def get_ok_response(self):
        return {
            'tag': 'validator-rpop-added',
            'attrs': {
                'uid': str(self.form_values['uid']),
                'address': self.form_values['email'],
            },
        }

    def process_request(self):
        self.process_basic_form()
        uid = self.form_values['uid']
        address = self.form_values['email']

        self.get_account_by_uid(
            uid,
            emails=True,
            email_attributes='all',
        )
        self.check_email_not_native(address)

        events = dict(action='validator')

        with UPDATE(self.account, self.request.env, events):
            email = self.account.emails.find(address)

            if not email:
                if len(self.account.emails) >= settings.MAX_EMAILS_PER_PROFILE:
                    raise EmailLimitPerProfileReachedError()
                email = Email()
                email.address = address
                email.created_at = datetime.now()
                email.is_unsafe = True
                log.debug(u'Email %s added as rpop for uid=%s.', address, uid)

            if not email.is_rpop:
                email.is_rpop = True
                self.send_account_modification_push(event_name='collector_add')
            else:
                log.debug(u'Address %s is already RPOP.', address)

            if not email.confirmed_at:
                email.confirmed_at = datetime.now()

            if not email.bound_at:
                email.bound_at = email.confirmed_at

            self.account.emails.add(email)

        self.statbox.log(
            action='rpop_added',
            address=mask_email_for_statbox(address),
            uid=uid,
        )
        self.response_values = self.get_ok_response()


class GetConfirmationCodeView(BaseEmailValidatorView,
                              EmailValidatorMixin):
    basic_form = GetConfirmationCodeForm
    required_grants = [
        'email_validator.api_getkey',
    ]

    def process_request(self):
        self.process_basic_form()
        address = self.form_values['email']
        unsafe = self.form_values['unsafe']

        self.get_account_from_uid_or_session(
            check_host_header=False,
            get_session_from_headers=False,
            emails=True,
            ignore_grants=True,
            email_attributes='all',
        )
        self.check_email_not_native(address)

        events = dict(action='validator')
        email = self.account.emails.find(address)

        if not email:
            if len(self.account.emails) >= settings.MAX_EMAILS_PER_PROFILE:
                raise EmailLimitPerProfileReachedError()
            with UPDATE(self.account, self.request.env, events):
                email = Email()
                email.address = address
                email.is_unsafe = unsafe
                email.created_at = datetime.now()

                log.debug(
                    u'Email %s added as unconfirmed for uid=%s.',
                    address,
                    self.account.uid,
                )
                self.account.emails.add(email)

        self.statbox.log(
            action='get_key',
            address=mask_email_for_statbox(address),
            uid=self.account.uid,
            is_unsafe=unsafe,
            sessionid=bool(self.form_values.get('sessionid')),
        )

        confirmation_track = self.create_email_confirmation_track(email)
        self.response_values = {
            'tag': 'validation-key',
            'text': confirmation_track.track_id,
        }


class ConfirmEmailByCodeView(BaseEmailValidatorView,
                             EmailValidatorMixin):
    basic_form = ConfirmEmailByCodeForm
    required_grants = [
        'email_validator.api_confirm',
    ]

    def process_request(self):
        self.process_basic_form()
        self.get_account_from_uid_or_session(
            check_host_header=False,
            get_session_from_headers=False,
            ignore_grants=True,
            emails=True,
            email_attributes='all',
        )

        key = self.form_values['key']
        unsafe = self.form_values['unsafe']

        email = self.check_email_confirmation_code(key)
        events = dict(action='validator')

        with UPDATE(self.account, self.request.env, events):
            email.confirmed_at = email.bound_at = datetime.now()
            if unsafe:
                email.is_unsafe = unsafe
            self.account.emails.add(email)

        send_address_validated_message(
            self.account,
            email.address,
        )

        self.statbox.log(
            action='confirmed_by_code',
            address=mask_email_for_statbox(email.address),
            uid=self.account.uid,
            keystring=key,
            is_unsafe=unsafe,
            sessionid=bool(self.form_values.get('sessionid')),
        )

        self.response_values = {
            'tag': 'validator-key-ok',
            'attrs': {
                'address': email.address,
                'uid': str(self.account.uid),
            },
        }


class ValidateByEmailView(BaseEmailValidatorView,
                          EmailValidatorMixin):
    basic_form = ValidateByEmailForm
    required_grants = [
        'email_validator.api_validate',
    ]

    def process_request(self):
        self.process_basic_form()
        address = self.form_values['email']

        self.get_account_from_uid_or_session(
            check_host_header=False,
            get_session_from_headers=False,
            ignore_grants=True,
            emails=True,
            email_attributes='all',
        )
        email_created = address not in self.account.emails

        events = dict(action='validator')
        with UPDATE(self.account, self.request.env, events):
            confirmation_track = self.validate_by_email(
                address,
                force=self.form_values['force'],
                is_unsafe=False,
                tld=self.form_values['domain'],
                language=self.form_values['lang'],
                retpath=self.form_values['retpath'],
            )

        self.statbox.log(
            action='validation_email_sent',
            address=mask_email_for_statbox(address),
            uid=self.account.uid,
            keystring=confirmation_track.track_id,
            created=email_created,
            unsafe=False,
            sessionid=bool(self.form_values['sessionid']),
        )
        self.response_values = {
            'tag': 'validator-request-sent-ok',
        }


class DeleteEmailView(BaseEmailValidatorView):
    basic_form = DeleteEmailForm
    required_grants = [
        'email_validator.api_delete',
    ]

    def check_eligible_for_deletion(self, email):
        pass

    def process_request(self):
        self.process_basic_form()
        uid = self.form_values['uid']
        address = self.form_values['email']

        self.get_account_by_uid(
            uid=uid,
            enabled_required=False,
            emails=True,
            email_attributes='all',
        )

        self.response_values = {
            'tag': 'validator-address-deleted',
            'attrs': {
                'address': address,
                'uid': str(self.account.uid),
            },
        }

        email = self.account.emails.find(address)
        if not email:
            return
        self.check_eligible_for_deletion(email)

        events = dict(action='validator')
        with UPDATE(self.account, self.request.env, events):
            self.account.emails.pop(email.address)

        send_address_deleted_message(self.account, email)
        self.statbox.log(
            action='delete',
            address=mask_email_for_statbox(email.address),
            uid=uid,
        )


class DeleteUnsafeEmailView(DeleteEmailView):
    basic_form = DeleteUnsafeForm
    required_grants = [
        'email_validator.api_deleteunsafe',
    ]

    def check_eligible_for_deletion(self, email):
        if not email.is_unsafe:
            raise EmailIsNotUnsafeError(email.address, self.account.uid)


class DeleteRpopEmailView(DeleteEmailView):
    basic_form = DeleteRpopForm
    required_grants = [
        'email_validator.api_deleterpop',
    ]

    def check_eligible_for_deletion(self, email):
        if not email.is_rpop:
            raise EmailIsNotRpopError(email.address, self.account.uid)


class DeleteAllEmailsView(BaseEmailValidatorView):
    basic_form = DeleteAllEmailsForm
    required_grants = [
        'email_validator.api_deleteemails',
    ]

    def process_request(self):
        self.process_basic_form()

        self.get_account_from_session(
            get_session_from_headers=False,
            check_host_header=False,
            emails=True,
            email_attributes='all',
        )

        events = dict(action='validator')
        with UPDATE(self.account, self.request.env, events):
            self.account.emails = None

        self.statbox.log(
            action='deleteall',
            uid=self.account.uid,
        )

        self.response_values = {
            'tag': 'validator-emails-deleted',
        }


# Специальный роутер для определения необходимой вьюхи на конкретный запрос.
# Связано это с тем, что вместо 404 на запрос к несуществующему методу надо
# отдавать пустой ответ.
dispatch_validator_request = dispatch_operations({
    'addrpop': AddRPOPView,
    'getkey': GetConfirmationCodeView,
    'confirm': ConfirmEmailByCodeView,
    'delete': DeleteEmailView,
    'deleteemails': DeleteAllEmailsView,
    'deleterpop': DeleteRpopEmailView,
    'deleteunsafe': DeleteUnsafeEmailView,
    'validate': ValidateByEmailView,
})
