# -*- coding: utf-8 -*-
import random
import string

from passport.backend.api.exceptions import (
    InvalidHostError,
    MissingHeaderError,
)
from passport.backend.core.cookies.utils import parse_cookie
from passport.backend.core.env import Environment
from passport.backend.core.grants.grants_config import (
    check_substitute_consumer_ip_grant,
    check_substitute_user_ip_grant,
)
from passport.backend.core.portallib import get_net
from passport.backend.utils.string import (
    smart_bytes,
    smart_text,
)
from six.moves.urllib.parse import urlparse


REQUEST_ID_ALPHABET = string.digits + string.ascii_letters
REQUEST_ID_LENGTH = 10


def _generate_request_id(length):
    # g - generated, чтобы было сразу понятно, что request id не прилетел снаружи
    prefix = 'g-'
    return prefix + ''.join(random.choice(REQUEST_ID_ALPHABET) for _ in range(length - len(prefix)))


def _get_header(headers, name, alternate_name, required=False, default=None, ensure_ascii=False):
    header = headers.get(name, headers.get(alternate_name, default))
    if required and header is None:
        raise MissingHeaderError('Required header "%s"("%s") is missing in request headers' % (name, alternate_name))
    if header and ensure_ascii:
        # Выкидываем все не-ascii символы. Это более-менее законная операция,
        # см. последний абзац https://tools.ietf.org/html/rfc7230#section-3.2.4
        header = smart_text(
            smart_bytes(header).decode('ascii', 'ignore').encode('ascii'),
        )
    return header


def _get_and_validate_host(headers):
    host = _get_header(headers, 'HTTP_YA_CLIENT_HOST', 'HTTP_HOST')
    # rsplit в условии для того, что данные для аутентификации не должны присутствовать в Ya-Client-Host
    if host and host.lower() != urlparse('//' + host).netloc.rsplit('@', 1)[-1]:
        raise InvalidHostError('Header Ya-Client-Host is invalid')
    return host


def _list_to_multivalue_dict(list_of_pairs):
    result = {}
    for key, value in list_of_pairs:
        result.setdefault(key, []).append(value)
    return result


def get_consumer_ip(request):
    return _get_header(request.environ, 'HTTP_X_REAL_IP', 'REMOTE_ADDR')


def get_user_agent(request):
    return _get_header(request.environ, 'HTTP_YA_CLIENT_USER_AGENT', 'HTTP_USER_AGENT', ensure_ascii=True)


class APIEnvironment(Environment):
    @classmethod
    def from_request(cls, request):
        consumer_ip = get_consumer_ip(request)

        # Некоторые паспортные прокси (например, YaSMS) могут прокидывать IP потребителя в специальном заголовке
        consumer_ip_from_proxy = request.environ.get('HTTP_YA_CONSUMER_REAL_IP')
        if consumer_ip_from_proxy and check_substitute_consumer_ip_grant(consumer_ip):
            consumer_ip = consumer_ip_from_proxy

        user_ip_raw = request.environ.get('HTTP_YA_CONSUMER_CLIENT_IP', consumer_ip)
        user_ip = None

        # Если запрос пришёл из mobileproxy через украинский туннель, то нам нужно заменить user_ip
        # на переданный в хедере. Чтобы убедиться, что это наш случай, проверим, является ли user_ip
        # адресом туннельного балансера.
        # На этом этапе нет нужды пропускать user_ip_raw через IPReg, так как после туннельного балансера
        # запрос не проходит через недоверенные прокси.
        user_ip_from_proxy = request.environ.get('HTTP_X_YPROXY_HEADER_IP')
        if user_ip_from_proxy and check_substitute_user_ip_grant(user_ip_raw):
            user_ip_raw = user_ip_from_proxy

        if user_ip_raw:
            user_ip = get_net(user_ip_raw, request.headers)['real_ip']

        ya_client_cookies = request.environ.get('HTTP_YA_CLIENT_COOKIE')

        if ya_client_cookies:
            cookies = parse_cookie(ya_client_cookies)
            cookies_all = parse_cookie(ya_client_cookies, cls=_list_to_multivalue_dict)
        else:
            cookies = {}
            cookies_all = {}

        request_id = _get_header(request.environ, 'HTTP_YA_CLIENT_X_REQUEST_ID', 'HTTP_X_REQUEST_ID')
        if not request_id:
            request_id = _generate_request_id(REQUEST_ID_LENGTH)

        return cls(
            cookies=cookies,
            cookies_all=cookies_all,
            consumer_ip=consumer_ip,
            user_ip=user_ip,
            user_agent=get_user_agent(request),
            host=_get_and_validate_host(request.environ),
            accept_language=_get_header(request.environ, 'HTTP_YA_CLIENT_ACCEPT_LANGUAGE', 'HTTP_ACCEPT_LANGUAGE'),
            referer=_get_header(request.environ, 'HTTP_YA_CLIENT_REFERER', 'HTTP_REFERER'),
            authorization=_get_header(request.environ, 'HTTP_YA_CONSUMER_AUTHORIZATION', 'HTTP_AUTHORIZATION'),
            service_ticket=request.environ.get('HTTP_X_YA_SERVICE_TICKET', None),
            user_ticket=request.environ.get('HTTP_X_YA_USER_TICKET', None),
            request_id=request_id,
            request_path=request.path,
        )
