# -*- coding: utf-8 -*-
from passport.backend.core import validators

from .base import UidForm


class AdminActionForm(UidForm):
    admin_name = validators.String(if_missing=None, not_empty=True, strip=True)
    comment = validators.String(if_missing=None, not_empty=True, strip=True)

    # Этот валидатор должен быть во всех наследниках
    chained_validators = [
        validators.RequireSet([('admin_name', 'comment')], allow_empty=True),
    ]


class Karma(AdminActionForm):
    prefix = validators.KarmaPrefix()
    suffix = validators.KarmaSuffix()

    chained_validators = [
        validators.RequireSome([
            'prefix',
            'suffix'
        ], min_=1, max_=2),
        validators.RequireSet([('admin_name', 'comment')], allow_empty=True),
    ]


class Account(AdminActionForm):
    is_enabled = validators.StringBool(if_missing=None, not_empty=True, strip=True)
    is_enabled_app_password = validators.StringBool(if_missing=None, not_empty=True, strip=True)

    chained_validators = AdminActionForm.chained_validators + [
        validators.RequireSome(['is_enabled', 'is_enabled_app_password'], min_=1, max_=2),
    ]


class Person(UidForm):
    pre_validators = [
        validators.GroupDisplayName(),
    ]

    force_clean_web = validators.StringBool(if_missing=False)
    firstname = validators.AntiFraudFirstName(if_missing=None)
    lastname = validators.AntiFraudLastName(if_missing=None)
    display_name = validators.DisplayName(if_missing=None)
    gender = validators.Gender(if_missing=None)
    birthday = validators.Birthday(if_missing=None)
    country = validators.CountryCode(if_missing=None, not_empty=False)
    city = validators.Regex('^[^<>\&]{,50}$', if_missing=None)
    language = validators.PortalLanguage(if_missing=None)  # TODO: переименовать language в person_language
    timezone = validators.Timezone(if_missing=None)
    contact_phone_number = validators.String(if_missing=None, not_empty=True,
                                             strip=True)

    chained_validators = [
        validators.RequireSome([
            'firstname',
            'lastname',
            'display_name',
            'gender',
            'birthday',
            'country',
            'city',
            'language',
            'timezone',
            'contact_phone_number',
        ], min_=1, max_=10),
        validators.PhoneNumber(phone_number_key='contact_phone_number',
                               country_key=None)
    ]


__all__ = (
    'Account',
    'Karma',
    'Person',
)
