# -*- coding: utf-8 -*-

from passport.backend.core import validators
from passport.backend.core.types.account.account import (
    ACCOUNT_TYPE_LITE,
    ACCOUNT_TYPE_SOCIAL,
)

from .base import (
    RequiredTrackedConsumerForm,
    SetPassword,
)


PHONE_VALIDATION_METHOD = 'phone'
CAPTCHA_VALIDATION_METHOD = 'captcha'


class AccountRegisterModeForm(validators.Schema):
    mode = validators.String(not_empty=True, strip=True)


class AccountRegisterBase(RequiredTrackedConsumerForm):
    login = validators.Login()

    force_clean_web = validators.StringBool(if_missing=False)
    firstname = validators.AntiFraudFirstName(not_empty=True)
    lastname = validators.AntiFraudLastName(not_empty=True)

    # язык, который выбрал пользователь или автоопределённый
    language = validators.PortalLanguage()

    country = validators.CountryCode()

    gender = validators.Gender(if_missing=None)
    birthday = validators.Birthday(if_missing=None)
    timezone = validators.Timezone(if_missing=None)

    eula_accepted = validators.StringBool(not_empty=True, strip=True)


class AccountRegisterRequireConfirmedPhone(AccountRegisterBase):
    password = validators.String(not_empty=True)

    chained_validators = [
        validators.Availability(),
    ]


class AccountRegisterRequireConfirmedPhoneAndEmail(AccountRegisterBase):
    login = validators.LiteLogin()
    password = validators.String(not_empty=True)

    chained_validators = [
        validators.Availability(login_type=ACCOUNT_TYPE_LITE),
    ]


class AccountRegisterSocial(AccountRegisterBase):
    login = validators.SocialLogin(if_missing=None)
    provider = validators.SocialProvider(not_empty=True)

    chained_validators = [
        validators.Availability(login_type=ACCOUNT_TYPE_SOCIAL),
    ]


class AccountRegisterSelectAlternative(RequiredTrackedConsumerForm):
    validation_method = validators.OneOf(
        [
            PHONE_VALIDATION_METHOD,
            CAPTCHA_VALIDATION_METHOD,
        ],
        not_empty=True,
    )


class AccountRegisterAlternativeBase(AccountRegisterBase):
    password = validators.String(not_empty=True)

    tmx_session = validators.String(if_missing=None)

    plus_promo_code = validators.String(if_missing=None, strip=True)

    unsubscribe_from_maillists = validators.StringBool(if_missing=False)
    origin = validators.String(if_missing=None)
    app_id = validators.String(if_missing=None)

    chained_validators = [
        validators.Availability(),
    ]


class AccountRegisterAlternativeWithConfirmedPhone(AccountRegisterAlternativeBase):
    allow_to_take_busy_alias = validators.StringBool(if_missing=False)


class AccountRegisterAlternativeWithHint(AccountRegisterAlternativeBase):
    # язык отображаемого интерфейса, автоопределённый
    display_language = validators.DisplayLanguage()

    hint_question_id = validators.HintQuestionId()
    hint_question = validators.HintQuestion()
    hint_answer = validators.HintAnswer(if_missing=None)

    chained_validators = AccountRegisterAlternativeBase.chained_validators + [
        validators.RequireSet(
            allowed_sets=[
                ['hint_question_id', 'hint_answer'],
                ['hint_question', 'hint_answer'],
            ],
        ),
    ]


class AccountRegisterWithHintAndEmail(AccountRegisterBase):
    login = validators.LiteLogin()
    # язык отображаемого интерфейса, автоопределённый
    display_language = validators.DisplayLanguage()

    password = validators.String(not_empty=True)

    hint_question_id = validators.HintQuestionId()
    hint_question = validators.HintQuestion()
    hint_answer = validators.HintAnswer(if_missing=None)

    chained_validators = [
        validators.Availability(login_type=ACCOUNT_TYPE_LITE),
        validators.RequireSet(
            allowed_sets=[
                ['hint_question_id', 'hint_answer'],
                ['hint_question', 'hint_answer'],
            ],
        ),
    ]


class AccountRegisterPhonishForm(RequiredTrackedConsumerForm):
    app_id = validators.String(if_missing=None, strip=True)


class AccountRegisterUncompleted(AccountRegisterBase):
    eula_accepted = None

    chained_validators = [
        validators.Availability(),
    ]


class SetPasswordUncompleted(SetPassword):
    eula_accepted = validators.StringBool(not_empty=True, strip=True)


__all__ = (
    'PHONE_VALIDATION_METHOD',
    'CAPTCHA_VALIDATION_METHOD',
    'AccountRegisterModeForm',
    'AccountRegisterRequireConfirmedPhone',
    'AccountRegisterRequireConfirmedPhoneAndEmail',
    'AccountRegisterSocial',
    'AccountRegisterSelectAlternative',
    'AccountRegisterAlternativeWithConfirmedPhone',
    'AccountRegisterAlternativeWithHint',
    'AccountRegisterWithHintAndEmail',
    'AccountRegisterPhonishForm',
    'AccountRegisterUncompleted',
    'SetPasswordUncompleted',
)
