# -*- coding: utf-8 -*-
from functools import partial
import logging

import elementflow
from flask import request
from passport.backend.api.common.decorators import (
    invalid_form_response,
    validate,
)
from passport.backend.api.common.errors import internal_error
from passport.backend.api.common.format_response import XmlLoggedResponse
from passport.backend.api.exceptions import UnknownLoginError
from passport.backend.core.builders import blackbox
from passport.backend.core.dbmanager.manager import DBError
from passport.backend.core.exceptions import InvalidIpHeaderError
from passport.backend.core.grants import GrantsError
from passport.backend.core.models.account import UnknownUid
from passport.backend.utils.string import smart_text
from six import (
    BytesIO,
    iteritems,
)


log = logging.getLogger('passport.api.common')

HTML_ERROR_RESPONSE = u"""
<!DOCTYPE html>
<html>
<head>
<meta http-equiv="Content-Type" content="text/html; charset=UTF-8">
</head>
<body>
%s
</body>
</html>
""" % ('<a href="mailto:passport-admin@yandex-team.ru">\nNo Grants\n</a>\n' * 222)


class LegacyXmlLoggedResponse(XmlLoggedResponse):
    """
    Собирает ответ как xml-документ
    Элементы документа могут быть вложенными до одного уровня
    Все элементы, кроме корневого не имеют атрибутов
    """
    default_mimetype = u'application/xml'

    def __init__(self, root_element, job_or_status_attr, **data):
        sensitive_fields = data.pop('sensitive_fields', None)
        cleaned_data = self._clean_data(data)
        format_response = partial(
            dict_to_xml,
            root_element,
            job_or_status_attr,
        )
        super(LegacyXmlLoggedResponse, self).__init__(
            cleaned_data,
            format_=format_response,
            sensitive_fields=sensitive_fields,
        )

    def _clean_data(self, data):
        cleaned_data = dict()
        for key, value in data.items():
            if value is not None:
                if isinstance(value, (list, set)):
                    cleaned_data[key] = [(k, smart_text(v)) for k, v in value if v is not None]
                else:
                    cleaned_data[key] = smart_text(value)
        return cleaned_data


class ListLegacyXmlLoggedResponse(XmlLoggedResponse):
    """
    Собирает ответ как xml-документ
    Элементы документа, кроме корневого, представляют плоский список
    Элементы имеют атрибуты
    см. list_to_xml()
    """

    default_mimetype = u'application/xml'

    def __init__(self, root_element_name, root_attrs, element_list):
        cleaned_data = self._clean_data(element_list)
        format_response = partial(
            list_to_xml,
            root_element_name,
            root_attrs,
        )
        super(ListLegacyXmlLoggedResponse, self).__init__(
            cleaned_data,
            format_=format_response,
        )

    def _clean_data(self, data):
        """
        Исключает из обработки атрибуты, значения которых None
        Безопасно переводит значения атрибутов в строки
        """
        cleaned_data = list()
        for key, attributes in data:
            cleaned_attributes = dict(
                (key, smart_text(value))
                for key, value in attributes.items()
                if value is not None
            )
            cleaned_data.append((key, cleaned_attributes))

        return cleaned_data


def list_to_xml(root_tag, root_attrs, data):
    """
    Преобразует переданный список в xml-документ с указанным корневым элементом
    :param root_tag: Имя тега корневого элемента
    :param root_attrs: Словарь атрибутов корневого элемента, ключи и значения -- строки
    :param data: список коретежей вида (имя_элемента, словарь_параметров_элемента)
    :return: Текст xml-документа

    >>> list_to_xml(
        'foo',
        {'bar': 'spam'},
        [
            ('item', {'id': '1', 'text': 'item-text'}),
            ('extra', {'id': '2', 'size': '25'}),
        ],
    )
    '''
    <?xml version="1.0" encoding="utf-8" ?>
    <foo bar="spam">
        <item id=1>item-text</item>
        <extra id=2 size=25 />
    </foo>
    '''
    """
    body = BytesIO()
    with elementflow.xml(body, root_tag, indent=False, attrs=root_attrs) as page:
        for key, attrs in data:
            text = attrs.pop('text', None)
            page.element(key, attrs=attrs, text=text)

    xml = body.getvalue()
    return xml


def dict_to_xml(root_tag, root_attrs, data):
    """
    Преобразует специальный словарь в xml документ.

    Параметры

        root_tag
            тег корневого элемента

        root_attrs
            словарь с атрибутами для корневого элемента

        data
            словарь, ключи -- это теги сыновей корневого элемента,
            значения -- строка или список/множество пар (тег, текст).

    Пример

    dict_to_xml(
        u'foo',
        {u'bar': u'spam'},
        {
            u'alpha': u'a',
            u'beta': [(u'gamma', 'g'), (u'zeta', u'z')],
        },
    )

    '''
    <?xml version="1.0" encoding="utf-8" ?>
    <foo bar="spam">
        <alpha>a</alpha>
        <beta>
            <gamma>g</gamma>
            <zeta>z</zeta>
        </beta>
    </foo>
    '''

    """
    body = BytesIO()
    with elementflow.xml(body, root_tag, indent=False, attrs=root_attrs) as page:
        for key, value in iteritems(data):
            if isinstance(value, (list, set)):
                with page.container(key):
                    for k, v in value:
                        page.element(k, text=v)
            else:
                page.element(key, text=value)
    xml = body.getvalue()
    return xml


def xml_response(root_element, job_or_status_attr, **data):
    """
    :param root_element: имя корневого тэга
    :type unicode
    :param job_or_status_attrs: атрибут корневого элемента job or status
    :type dict
    """
    response = LegacyXmlLoggedResponse(root_element, job_or_status_attr, **data)
    response.status_code = 200
    return response


def mailhost_xml_response(*args, **kwargs):
    response = ListLegacyXmlLoggedResponse(*args, **kwargs)
    response.status_code = 200
    return response


def invalid_xml_response(**data):
    return xml_response(u'result', {u'status': u'error'}, **data)


def admsubscribe_invalid_xml_response(**data):
    return xml_response(u'page', {u'job': u'rejected'}, **data)


def mailhost_ok_response():
    return mailhost_xml_response('doc', {}, [('status', dict(id=0, text='OK'))])


def mailhost_error_response(error):
    return mailhost_xml_response('doc', {}, [('status', dict(id=1, text=error))])


def mailhost_exception_response(error):
    return mailhost_xml_response('doc', {}, [
        ('exception', dict(id='1', text='UNKNOWN')),
        ('error', dict(text=error)),
    ])


def error_handler(e):
    if isinstance(e, GrantsError):
        return HTML_ERROR_RESPONSE
    elif isinstance(e, blackbox.AccessDenied):
        return invalid_xml_response(error='interror', text=e)
    elif isinstance(e, blackbox.BaseBlackboxError):
        return invalid_xml_response(error='interror', text=e)
    elif isinstance(e, UnknownUid):
        return invalid_xml_response(error=u'unknownuid', text=u'Пользователь с таким UID не найден.')
    elif isinstance(e, DBError):
        return invalid_xml_response(error='interror', text=e)
    elif isinstance(e, InvalidIpHeaderError):
        return invalid_xml_response(error='interror', text=e)
    else:
        return internal_error(e)


def admsubscribe_error_handler(e):
    if isinstance(e, GrantsError):
        return HTML_ERROR_RESPONSE
    elif isinstance(e, blackbox.AccessDenied):
        return admsubscribe_invalid_xml_response(error=u'interror', message=e)
    elif isinstance(e, blackbox.BaseBlackboxError):
        return admsubscribe_invalid_xml_response(error=u'interror', message=e)
    elif isinstance(e, UnknownUid):
        return admsubscribe_invalid_xml_response(error=u'unknownuid', message=e)
    elif isinstance(e, UnknownLoginError):
        return admsubscribe_invalid_xml_response(error=u'unknownlogin', message=e)
    elif isinstance(e, DBError):
        return admsubscribe_invalid_xml_response(error=u'interror', message=e)
    elif isinstance(e, InvalidIpHeaderError):
        return admsubscribe_invalid_xml_response(error='interror', text=e)
    else:
        return internal_error(e)


def admreg_error_handler(e):
    if isinstance(e, GrantsError):
        return HTML_ERROR_RESPONSE
    elif isinstance(e, (InvalidIpHeaderError, DBError, blackbox.BaseBlackboxError)):
        return '500: %s' % e, 500
    else:
        return internal_error(e)


def mailhost_error_handler(e):
    if isinstance(e, GrantsError):
        return HTML_ERROR_RESPONSE
    elif isinstance(e, DBError):
        return mailhost_exception_response(e)
    elif isinstance(e, blackbox.BaseBlackboxError):
        return mailhost_exception_response(e)
    else:
        return internal_error(e)


def get_request_values():
    return request.values.to_dict()


def validate_legacy(form, error_handler=invalid_form_response):
    return validate(form, error_handler=error_handler, get_request_values_func=get_request_values)
