# -*- coding: utf-8 -*-
from passport.backend.api.validators import FieldOnlyForSids
from passport.backend.core import validators
from passport.backend.core.services import (
    SERVICES,
    sids,
)

from .validators import SetService


class LegacyForm(validators.Schema):
    pass


class AdmRegLegacyForm(validators.Schema):
    pass


LegacyForm.add_field(
    'from',
    validators.compound.All(
        validators=[
            validators.OneOf(
                SERVICES,
                if_missing=None,
                messages={'notIn': u'недопустимое значение параметра from'},
            ),
            validators.String(if_missing=None, strip=True, not_empty=True),
        ],
    ),
)

LegacyForm.add_field(
    'sid',
    validators.compound.All(
        validators=[
            validators.OneOf(
                sids,
                if_missing=None,
                messages={'notIn': u'недопустимое значение параметра sid'},
            ),
            validators.Int(if_missing=None, strip=True, not_empty=True),
        ],
    ),
)

AdmRegLegacyForm.add_field(
    'from',
    validators.compound.All(
        validators=[
            validators.OneOf(
                SERVICES,
                if_missing=None,
                if_invalid=None,
                messages={'notIn': u'недопустимое значение параметра from'},
            ),
            validators.String(if_missing=None, strip=True, not_empty=True),
        ],
    ),
)

AdmRegLegacyForm.add_field(
    'sid',
    validators.compound.All(
        validators=[
            validators.OneOf(
                sids,
                if_missing=None,
                if_invalid=None,
                messages={'notIn': u'недопустимое значение параметра sid'},
            ),
            validators.Int(if_missing=None, strip=True, not_empty=True),
        ],
    ),
)


class AdmLoginRule(LegacyForm):
    uid = validators.Uid()
    login_rule = validators.Int(not_empty=True)
    need_change_pass = validators.StringBool(if_missing=None)

    chained_validators = [
        SetService(),
        FieldOnlyForSids('need_change_pass', [8], False),
    ]


def required_for_subscribe(x):
    return True if not x.get('unsubscribe') else False


class AdmSubscribe(LegacyForm):
    uid = validators.Uid(if_missing=None)
    login = validators.String(if_missing=None, not_empty=True)
    yastaff_login = validators.String(if_missing=None, not_empty=True, strip=True)
    unsubscribe = validators.StringBool(if_missing=False)
    wmode = validators.Int(if_missing=None)

    chained_validators = [
        validators.RequireSome(['uid', 'login'], max_=2,
                               messages={'empty': u'недостаточно параметров'}),
        SetService(),
        FieldOnlyForSids('wmode', [42], False),
        FieldOnlyForSids(
            'yastaff_login',
            [669],
            required_checker=required_for_subscribe,
        ),
    ]


class AdmKarma(validators.Schema):
    uid = validators.Uid()
    prefix = validators.KarmaPrefix()
    karma = validators.KarmaSuffix()
    chained_validators = [
        validators.RequireSome(['karma', 'prefix'],
                               messages={'empty': u'недостаточно параметров'})
    ]


class AdmBlock(LegacyForm):
    uid = validators.Uid()
    chained_validators = [
        SetService(required=False)
    ]


class AdmChangeReg(LegacyForm):
    pre_validators = [
        validators.GroupDisplayName(),
    ]

    uid = validators.Uid(messages={'empty': 'Some of required fields are not specified'})
    birth_date = validators.Birthday(if_missing=None)
    timezone = validators.Timezone(if_missing=None)
    display_name = validators.DisplayName(if_missing=None)
    lang = validators.PortalLanguage(if_missing=None)
    iname = validators.FirstName(if_missing=None)
    fname = validators.LastName(if_missing=None)
    sex = validators.Gender(if_missing=None)

    chained_validators = [
        SetService(),
        validators.RequireSome(
            ['birth_date', 'timezone', 'display_name', 'lang', 'iname', 'fname', 'sex'], min_=1, max_=7)
    ]


class AdmSimpleReg(LegacyForm):
    login = validators.YandexTeamLogin()
    maillist = validators.StringBool(if_missing=False)

    chained_validators = [
        SetService(required=False),
    ]


class AdmReg(AdmRegLegacyForm):
    login = validators.Login()
    iname = validators.FirstName(not_empty=True)
    fname = validators.LastName(not_empty=True)
    passwd = validators.String(not_empty=True)
    passwd2 = validators.String(not_empty=True)
    ignore_stoplist = validators.StringBool(if_missing=False)
    ena = validators.StringBool(if_missing=True)
    yastaff_login = validators.String(if_missing=None, strip=True, not_empty=True)

    chained_validators = [
        SetService(required=False),
        FieldOnlyForSids('yastaff_login', [669]),
        validators.Password(password_field='passwd'),
        validators.FieldsMatch('passwd', 'passwd2', validate_partial_form=False),
    ]


class MailHost(LegacyForm):

    OPERATION_CREATE = 'create'
    OPERATION_SET_PRIORITY = 'setprio'
    OPERATION_DELETE = 'delete'
    OPERATION_ASSIGN = 'assign'
    OPERATION_FIND = 'find'

    OPERATION_CHOICE = (
        OPERATION_CREATE,
        OPERATION_SET_PRIORITY,
        OPERATION_DELETE,
        OPERATION_ASSIGN,
        OPERATION_FIND
    )

    OPERATIONS_REQUIRE_NEGATIVE_PRIORITY = (
        OPERATION_CREATE,
        OPERATION_SET_PRIORITY,
    )

    op = validators.OneOf(OPERATION_CHOICE, not_empty=True, strip=True)
    prio = validators.Number(if_missing=None, not_empty=True)
    mx = validators.String(if_missing=None, not_empty=True)

    # Это значение-строку задает потребитель - Почта
    db_id = validators.String(if_missing=None, not_empty=True)

    suid = validators.Number(if_missing=None, not_empty=True)
    old_db_id = validators.String(if_missing=None, not_empty=True)

    # При разных операциях требуются различные наборы параметров
    chained_validators = [
        validators.RequireIfEquals(['db_id', 'prio', 'mx'], 'op', 'create'),
        validators.RequireIfEquals(['db_id', 'prio'], 'op', 'setprio'),
        validators.RequireIfEquals(['db_id'], 'op', 'delete'),
        validators.RequireIfEquals(['db_id', 'suid'], 'op', 'assign'),
        validators.RequireIfEquals(['prio'], 'op', 'find'),
    ]
