# -*- coding: utf-8 -*-

from datetime import datetime
from functools import wraps
import logging
from operator import itemgetter

from flask import request
from passport.backend.api.common.account import (
    fill_person_from_args,
    set_impossible_password,
    set_password_with_experiment,
)
from passport.backend.api.common.format_response import format_errors
from passport.backend.api.common.grants import add_grants_for_service
from passport.backend.api.common.logs import setup_log_prefix
from passport.backend.api.common.mail import is_mail_occupied_by_another_user
from passport.backend.api.common.suggest import get_login_suggestions
from passport.backend.api.exceptions import UnknownLoginError
from passport.backend.core import (
    frodo,
    validators,
)
from passport.backend.core.builders.blackbox import (
    BLACKBOX_FIND_BY_PHONE_ALIAS_FORCE_ON,
    get_blackbox,
)
from passport.backend.core.builders.blackbox.utils import user_exists
from passport.backend.core.conf import settings
from passport.backend.core.grants import (
    get_grants,
    get_grants_config,
)
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.models.account import (
    Account,
    UnknownUid,
)
from passport.backend.core.models.mailhost import MailHost
from passport.backend.core.models.person import Person
from passport.backend.core.runner.context_managers import (
    CREATE,
    DELETE,
    UPDATE,
)
from passport.backend.core.services import get_service
from passport.backend.core.subscription import (
    add_subscription,
    can_be_subscribed,
    delete_subscription,
    SubscriptionImpossibleError,
    SubscriptionNotAllowedError,
    SubscriptionRequiresAccountWithLoginError,
    SubscriptionRequiresAccountWithPasswordError,
)
from passport.backend.utils.string import smart_text
from six import string_types

from . import forms
from .common import (
    admsubscribe_invalid_xml_response,
    invalid_xml_response,
    mailhost_error_response,
    mailhost_ok_response,
    mailhost_xml_response,
    validate_legacy as validate,
    xml_response,
)


log = logging.getLogger('passport.api.legacy.views')


def only_for_service(service, grant, else_grants=None):
    if isinstance(grant, string_types):
        grant = [grant]

    def _check(*args, **kwargs):
        return grant if service == args[0].get('from') else else_grants(*args, **kwargs)

    return _check


def grants_legacy(required_grants, optional_grants=None):
    def grants_wrapper(f):
        @wraps(f)
        def _wrapper(*args, **kwargs):
            get_grants().check_access(
                request.env.consumer_ip,
                get_grants_config().get_consumers(
                    request.env.consumer_ip,
                    args[0].get('from') if args else None,
                ),
                required_grants,
                optional_grants,
                grants_args=args,
                grants_kwargs=kwargs,
                service_ticket=request.env.service_ticket,
            )
            return f(*args, **kwargs)
        return _wrapper
    return grants_wrapper


def missing_params_xml_response(error, **data):
    return invalid_xml_response(error='nofield', text=smart_text(error), **data)


def valid_xml_response(**data):
    return xml_response(u'result', {u'status': u'ok'}, **data)


def admsubscribe_missing_params_xml_response(error, **data):
    return admsubscribe_invalid_xml_response(error='nofield',
                                             message=error, **data)


def admsubscribe_valid_xml_response(**data):
    return xml_response(u'page', {u'job': u'accepted'},
                        **data)


def admchangereg_error_xml_response(error, **data):
    if 'display_name' in error.error_dict:
        code = 'baddisplayname'
    elif 'sex' in error.error_dict:
        code = 'badsex'
    elif 'timezone' in error.error_dict:
        code = 'badtimezone'
    elif 'lang' in error.error_dict:
        code = 'badlang'
    else:
        code = 'nofield'
    return invalid_xml_response(error=code, text=error)


def admsimplereg_error_xml_response(error, **data):

    code = 'interr'

    if 'login' in error.error_dict:
        if error.args[1].get('login'):
            code = 'badlogin'
        else:
            code = 'nologin'

    elif 'maillist' in error.error_dict:
        code = 'badmaillist'

    return xml_response('page', {u'status': 'error'},
                        login=error.args[1].get('login', ''),
                        error=code,
                        error_text=error.args[0])


def admreg_invalid_params(error, **data):
    errors = format_errors(error)
    return u'500: %s' % '\n'.join(['%(field)s: %(message)s (%(code)s)' % e for e in errors]), 500


@validate(forms.AdmLoginRule(), missing_params_xml_response)
@grants_legacy([only_for_service('passport', 'password.is_changing_required', add_grants_for_service(['update']))])
def admloginrule(params):
    """
    :param params список параметров функции, провалидированных формой
    :type dict
    """
    uid = params['uid']
    service = params['service']
    need_change_pass = params['need_change_pass']

    data = get_blackbox().userinfo(uid=uid)

    account = Account().parse(data)
    setup_log_prefix(account)

    events = {
        'action': 'admloginrule',
        'from': service.slug,
    }

    if service.sid not in account.subscriptions:
        error_msg = u'У пользователя нет подписки на указанный сервис.'
        return invalid_xml_response(error=u'nosubscription', text=error_msg)

    if service.sid == 8 and need_change_pass is None:
        error_msg = u'Не указано поле need_change_pass.'
        return invalid_xml_response(error='nofield', text=error_msg)

    with UPDATE(account, request.env, events):
        # ставить флаг о смене пароля при первом входе можно только на
        # сервисе passport, sid=8
        if service.sid == 8:
            # Принудительная смена пароля не имеет смысла для пользователя
            # со включенным 2FA. (PASSP-10198)
            if account.totp_secret.is_set:
                error_msg = u'У пользователя включен 2FA, смена пароля ' \
                            u'не будет иметь смысла.'
                return invalid_xml_response(error='interror', text=error_msg)

            account.password.setup_password_changing_requirement(is_required=need_change_pass)
            return valid_xml_response(uid=uid, sid=service.sid,
                                      login_rule=u'1' if need_change_pass else u'0')

        else:
            account.subscriptions[service.sid].login_rule = params['login_rule']

    return valid_xml_response(uid=uid, sid=service.sid,
                              login_rule=params['login_rule'])


@validate(forms.AdmSubscribe(), admsubscribe_missing_params_xml_response)
@grants_legacy([
    add_grants_for_service(['create', 'update', 'delete']),
])
def admsubscribe(params):
    """
    :param params список параметров функции, провалидированных формой
    :type dict
    """
    uid = params['uid']
    service = params['service']

    bb = get_blackbox()
    data = bb.userinfo(
        uid=uid,
        login=params['login'],
        find_by_phone_alias=BLACKBOX_FIND_BY_PHONE_ALIAS_FORCE_ON,
        force_show_mail_subscription=True,
    )

    try:
        account = Account().parse(data)
        setup_log_prefix(account)
    except UnknownUid:
        if uid is not None:
            raise UnknownUid(u'Пользователь с таким UID не найден.')
        if params['login']:
            raise UnknownLoginError(u'Пользователь с указанным логином не найден.')

    if params['login'] and (account.is_pdd or account.is_lite):
        raise UnknownLoginError(u'Пользователь с указанным логином не найден.')

    unsubscribe = params['unsubscribe']
    mode = 'unsubscribe' if unsubscribe else 'subscribe'

    is_subscribed = account.is_subscribed(service)

    if unsubscribe:
        # У пользователя нет подписки. Нечего отписывать.
        if not is_subscribed:
            return admsubscribe_valid_xml_response(uid=account.uid, sid=service.sid,
                                                   status='nothingtodo', mode=mode)

        if account.is_pdd and service.slug == 'mail':
            return admsubscribe_invalid_xml_response(mode=mode, error='badsid',
                                                     message=u'Отписка запрещена для ПДД')

    else:
        # Пользователь уже подписан
        # Позволяем обновлять подписку на сидах: 42 ('wwwdgt')
        if service.sid not in [42] and is_subscribed:
            return admsubscribe_valid_xml_response(uid=account.uid, sid=service.sid,
                                                   status='nothingtodo', mode=mode)

        if service.slug == 'mail':
            if is_mail_occupied_by_another_user(bb, account):
                return admsubscribe_invalid_xml_response(mode=mode, error='emailalreadyinuse',
                                                         message=u'Другой пользователь уже использует этот email')

        try:
            can_be_subscribed(account, service)
        except SubscriptionNotAllowedError:
            return admsubscribe_invalid_xml_response(mode=mode, error='badsid',
                                                     message=u'Подписка запрещена для ПДД')
        except SubscriptionRequiresAccountWithLoginError:
            return admsubscribe_invalid_xml_response(mode=mode, error='accountwithloginrequired',
                                                     message=u'Подписка запрещена для пользователей без логина')
        except SubscriptionRequiresAccountWithPasswordError:
            return admsubscribe_invalid_xml_response(mode=mode, error='accountwithpasswordrequired',
                                                     message=u'Подписка запрещена для пользователей без пароля')
        except SubscriptionImpossibleError:
            return admsubscribe_invalid_xml_response(
                mode=mode,
                error='interr',
                message=u'Подписка на данный сид невозможна',
            )

        if params['yastaff_login'] and user_exists(params['yastaff_login'], sid=669):
            return admsubscribe_invalid_xml_response(mode=mode, error='occupied',
                                                     message=u'yastaff_login="%s" уже занят' % params['yastaff_login'])

    events = {'action': 'admsubscribe', 'from': service.slug}

    with UPDATE(account, request.env, events):
        if unsubscribe:
            delete_subscription(account, service)
        else:
            # Для galatasaray такой костыль
            if service.sid == 61 and not account.is_normal:
                return admsubscribe_invalid_xml_response(
                    mode=mode,
                    error='interr',
                    message=u'Подписка на данный сид невозможна для пользователя с uid %s' % account.uid,
                )
            add_subscription(
                account,
                service=service,
                login_rule=params.get('login_rule'),
                login=params.get('yastaff_login'),
                host_id=None if service.sid == 2 else params.get('wmode'),
            )
    return admsubscribe_valid_xml_response(status=u'ok', mode=mode,
                                           uid=account.uid, sid=service.sid)


@validate(forms.AdmKarma(), missing_params_xml_response)
@grants_legacy(["karma"])
def admkarma(params):
    """
    :param params список параметров функции, проваледированных формой
    :type dict
    """
    uid = params['uid']
    prefix = params['prefix']
    value = params['karma']

    data = get_blackbox().userinfo(uid=uid)

    account = Account().parse(data)
    setup_log_prefix(account)

    events = {
        'action': 'admkarma',
    }

    with UPDATE(account, request.env, events):
        # если приходило значение karma, то старый метод записывал
        # значение как есть и не сохранял префикс - поддержала
        if prefix is not None:
            account.karma.prefix = int(prefix)
        if value is not None:
            account.karma.suffix = int(value)

    return valid_xml_response(uid=uid, karma=value, prefix=prefix)


@validate(forms.AdmBlock(), missing_params_xml_response)
@grants_legacy([
    "karma",
    "account.is_enabled",
    add_grants_for_service(['update']),
])
def admblock(params):
    """
    :param params список параметров функции, провалидированных формой
    :type dict
    """
    uid = params['uid']
    service = params['service']

    data = get_blackbox().userinfo(uid=uid)

    account = Account().parse(data)
    setup_log_prefix(account)

    if not account.is_enabled:
        log.info('Account is already blocked')
        raise UnknownUid()

    events = {
        'action': 'admblock',
    }

    if service is not None:
        events.update({'from': service.slug})

    if service is not None and service.sid not in account.subscriptions:
        error_msg = u'У пользователя нет подписки на указанный сервис.'
        return invalid_xml_response(error=u'nosubscription', text=error_msg)

    with UPDATE(account, request.env, events):
        account.karma.prefix = 3

        account.is_enabled = 0

        if service is not None:
            account.subscriptions[service.sid].login_rule = 0
            return valid_xml_response(uid=uid, ena=0, login_rule=0, sid=service.sid)

    return valid_xml_response(uid=uid, ena=0)


@validate(forms.AdmChangeReg(), admchangereg_error_xml_response)
@grants_legacy(['admchangereg'])
def admchangereg(params):
    data = get_blackbox().userinfo(uid=params['uid'])

    account = Account().parse(data)
    setup_log_prefix(account)

    data = {
        'firstname': params['iname'],
        'lastname': params['fname'],
        'display_name': params['display_name'],
        'gender': params['sex'],
        'birthday': params['birth_date'],
        'language': params['lang'],
        'timezone': params['timezone'],
    }

    with UPDATE(account, request.env, {'action': 'admchangereg', 'from': params['from']}):
        account.person = fill_person_from_args(Person(account), data)

    response = dict(
        (key, params[key]) for key in [
            'uid', 'sid', 'iname', 'fname', 'display_name', 'sex', 'birth_date', 'lang', 'timezone']
    )

    return valid_xml_response(**response)


@validate(forms.AdmSimpleReg(), admsimplereg_error_xml_response)
@grants_legacy(['admsimplereg'])
def admsimplereg(params):

    login = params['login']

    try:
        validators.Availability().to_python(params, validators.State(request.env))
    except validators.Invalid:
        return xml_response(
            u'page',
            {'status': 'error'},
            login=params['login'],
            error='occupied',
            error_text='That username is already taken on Yandex. Try a different one.',
        )

    with CREATE(Account(), request.env, {'action': 'admsimplereg'}) as account:
        account.parse({
            'is_changing_required': False,
            'userinfo.reg_date.uid': datetime.now(),
            'karma': 0,
            'subscriptions': {
                8: {'sid': 8, 'host_id': 12, 'login': login},
            },
        })
        account.set_portal_alias(login)
        set_impossible_password(account)
        add_subscription(account, get_service(sid=2))
        if params['maillist']:
            # указываем, что аккаунт является рассылкой
            account.is_maillist = True
        else:
            # указываем, что аккаунт является сотрудником
            account.is_employee = True

    setup_log_prefix(account)

    return xml_response(
        u'page',
        {'status': 'ok', 'uid': str(account.uid)},
        login=account.login,
    )


@validate(forms.AdmReg(), admreg_invalid_params)
@grants_legacy(['admreg'])
def admreg(params):
    service = params['service']
    service_data = {}

    if service:
        service_data = {'from': service.slug}

    data = {
        'accounts.ena.uid': params['ena'],
        'person.firstname': params['iname'],
        'person.lastname': params['fname'],
        'person.country': settings.DEFAULT_COUNTRY,
        'is_changing_required': False,
        'userinfo.reg_date.uid': datetime.now(),
        'karma': 0,
        'subscriptions': {
            8: {'sid': 8, 'host_id': 12, 'login': params['login']},
        },
    }

    if params['yastaff_login'] and user_exists(params['yastaff_login'], sid=669):
        return '500: yastaff login %s already subscribed on sid=669' % params['yastaff_login'], 500

    try:
        validators.Availability().to_python(params, validators.State(request.env))
    except validators.Invalid:
        logins = get_login_suggestions(
            original_login=params['login'],
            firstname=params['iname'],
            lastname=params['fname'],
            login=params['login'],
            language=None,
        )

        suggest_data = {'variants': [('login', login) for login in logins]}
        suggest_data.update(service_data)
        return xml_response(
            u'page',
            {'status': 'occupied'},
            iname=params['iname'],
            fname=params['fname'],
            login=params['login'],
            **suggest_data
        )

    events = {'action': 'account_create_admreg'}

    with CREATE(Account(), request.env, events) as account:
        account.parse(data)
        account.set_portal_alias(params['login'])
        account.is_enabled = params['ena']
        set_password_with_experiment(
            account,
            params['passwd'],
            params['quality'],
            login=params['login'],
        )
        account.password.is_creating_required = True
        if service is not None:
            add_subscription(account, service, login=params.get('yastaff_login'))
            events['from'] = service.slug
        # Всегда подписываем на почту
        if service is None or service.sid != 2:
            add_subscription(account, get_service(sid=2))

    setup_log_prefix(account)

    query_params_to_frodo_params = {
        'login': 'login',
        'iname': 'firstname',
        'fname': 'lastname',
        'passwd': 'password',
        'from': 'service',
    }
    frodo_args = dict((query_params_to_frodo_params[key], value) for key, value in params.items()
                      if key in query_params_to_frodo_params and value is not None)
    frodo_args['action'] = 'admreg'
    frodo_args['quality'] = params['quality']

    frodo_info = frodo.FrodoInfo.create(request.env, frodo_args)
    account, _ = frodo.check_spammer(frodo_info, request.env, params['from'], account)

    statbox = StatboxLogger(**{
        'action': 'account_created',
        'mode': 'admreg',
        'uid': account.uid,
        'login': account.normalized_login,
        'karma': account.karma.value,
        'country': account.person.country,
    })

    statbox.log()

    return xml_response(
        u'page',
        {'status': 'ok', 'uid': str(account.uid)},
        iname=account.person.firstname,
        fname=account.person.lastname,
        login=account.login,
        **service_data
    )


def build_map_by_key(host_list, key_name):
    return dict(
        (host[key_name], host)
        for host in host_list
    )


def mailhost_incorrect_params_error_xml_response(error):
    """
    Обработаем ошибки формы и покажем xml как если бы это делал perl
    :param error: Исключение formencode.Invalid
    :return: Сформированный xml-ответ
    """
    # Указана неизвестная операция или параметр пуст
    if error.error_dict.get('op'):
        return mailhost_error_response('unknown op')

    # Не передан параметр db_id
    elif error.error_dict.get('db_id'):
        return mailhost_error_response('missing parameter db_id')

    # Не передан параметр prio
    elif error.error_dict.get('prio'):
        return mailhost_error_response('missing parameter prio')

    # Не передан параметр mx
    elif error.error_dict.get('mx'):
        return mailhost_error_response('missing parameter mx')

    # Не передан параметр suid или некорректное число
    elif error.error_dict.get('suid'):
        return mailhost_error_response('missing parameter suid')

    raise ValueError('Unknown form error')  # pragma: no cover


def serialize_mailhost(host):
    """Подготовим атрибуты xml-элемента"""
    result = dict(host)
    # Опечатка 'hostd_id' пришла из perl, а еще раньше из ЧЯ
    result['hostd_id'] = host['host_id']
    del result['host_id']
    return result


@validate(forms.MailHost(), mailhost_incorrect_params_error_xml_response)
@grants_legacy(['mailhost'])
def mailhost(args):
    """
    Замена perl-ручке mailhost

    Делает два запроса в ЧЯ - выбирает почтовые хосты для обычных пользователей
    и потом для ПДД пользователей

    Выполняет одну из операций:
      * Добавить новую запись в таблицу hosts; если хост с таким db_id уже есть, ответить ОК
      * Сменить приоритет хоста по переданному db_id; если хост не найден, ответить ОК
      * Удалить хост по переданному db_id; если хост не найден, ответить ОК
      * Найти хост по переданному db_id и записать его host_id в поле subscription.mail.host_id
        для аккаунта с указанным suid & sid=2
      * Найти все хосты, приоритет которых строго выше указанного и вернуть всю информацию о них

    См https://beta.wiki.yandex-team.ru/maildba/passportinterface/
    """
    db_id = args['db_id']
    operation = args['op']
    priority = args['prio']
    if operation in forms.MailHost.OPERATIONS_REQUIRE_NEGATIVE_PRIORITY and priority >= 0:
        return mailhost_error_response('prio must be negative')

    mail_sid = get_service(slug='mail').sid
    bb = get_blackbox()
    # Попросим из ЧЯ все известные hosts
    bb_response = bb.get_hosts(sid=mail_sid)
    by_db_id = build_map_by_key(bb_response, 'db_id')
    by_host_id = build_map_by_key(bb_response, 'host_id')
    # Попросим из ЧЯ все известные domains_hosts
    bb_response = bb.get_hosts(sid=mail_sid, is_pdd=True)
    domain_hosts_by_db_id = build_map_by_key(bb_response, 'db_id')
    domain_hosts_by_host_id = build_map_by_key(bb_response, 'host_id')

    # Соберем общие словари
    by_db_id.update(domain_hosts_by_db_id)
    by_host_id.update(domain_hosts_by_host_id)

    if operation == forms.MailHost.OPERATION_CREATE:
        if db_id in by_db_id:
            return mailhost_ok_response()

        host = MailHost()
        # TODO: Завернуть в транзакцию в будущем
        with CREATE(host, request.env, dict(action='mailhost')):
            host.db_id = db_id
            host.priority = priority
            host.mx = args['mx']
            host.sid = mail_sid  # Аналогичный хардкод есть в cgi-bin/DBAF/Safe.pm:ProcessMailHost()

        return mailhost_ok_response()

    elif operation == forms.MailHost.OPERATION_SET_PRIORITY:
        if db_id not in by_db_id:
            return mailhost_ok_response()

        host = MailHost().parse(by_db_id[db_id])
        with UPDATE(host, request.env, dict(action='mailhost')):
            host.priority = priority

        return mailhost_ok_response()

    elif operation == forms.MailHost.OPERATION_DELETE:
        if db_id not in by_db_id:
            return mailhost_ok_response()

        host = MailHost().parse(by_db_id[db_id])
        with DELETE(host, request.env, dict(action='mailhost')):
            # Никаких дополнительных действий не требуется
            pass

        return mailhost_ok_response()

    elif operation == forms.MailHost.OPERATION_ASSIGN:
        if db_id not in by_db_id:
            return mailhost_error_response('db_id=%s isn\'t exist' % db_id)

        old_db_id = args['old_db_id']
        if old_db_id and old_db_id not in by_db_id:
            return mailhost_error_response('old_db_id=%s isn\'t exist' % old_db_id)

        suid = args['suid']
        bb_response = bb.userinfo(sid=mail_sid, suid=suid)
        try:
            account = Account().parse(bb_response)
            setup_log_prefix(account)
        except UnknownUid:
            return mailhost_ok_response()

        mail_subscription = account.subscriptions.get(mail_sid)
        if not mail_subscription:
            # Пользователь не подписан на Почту
            return mailhost_error_response('account without mail subscription')

        # режим устарел, изменить почтовый host_id больше нельзя

        return mailhost_ok_response()

    elif operation == forms.MailHost.OPERATION_FIND:
        host_list = filter(
            lambda host: int(host['prio']) > priority,
            sorted(
                by_host_id.values(),
                key=itemgetter('host_id'),
            ),
        )
        element_list = [('status', dict(id=0, text='OK'))]
        element_list.extend(
            ('entry', serialize_mailhost(host))
            for host in host_list
        )
        return mailhost_xml_response(
            'doc',
            {},
            element_list,
        )

    # Форма не должна допустить этого
    raise ValueError('Unknown operation value %s' % operation)  # pragma: no cover
