# -*- coding: utf-8 -*-
import logging

import elementflow
from flask import request
from flask.views import View
from passport.backend.api.common.errors import log_internal_error
from passport.backend.api.common.format_response import (
    JsonLoggedResponse,
    XmlLoggedResponse,
)
from passport.backend.core.builders.blackbox.blackbox import BlackboxTemporaryError
from passport.backend.core.builders.oauth import BaseOAuthError
from passport.backend.utils.string import smart_text
from six import (
    BytesIO,
    iteritems,
)
from werkzeug.datastructures import (
    ImmutableMultiDict,
    MultiDict,
)

from .errors import MobileProxyError


log = logging.getLogger(__name__)


class MobileProxyXmlSuccessfulResponse(XmlLoggedResponse):
    def __init__(self, root, **attrs):
        out = BytesIO()
        flow = elementflow.xml(out, root)
        with flow:
            for key, value in iteritems(attrs):
                flow.element(key, text=value)
        response = out.getvalue()
        super(MobileProxyXmlSuccessfulResponse, self).__init__(
            response=response,
            format_=smart_text,
            status=200,
        )


class MobileProxyXmlErrorResponse(XmlLoggedResponse):
    def __init__(self, error, status_code=400):
        out = BytesIO()
        flow = elementflow.xml(out, 'error', {'code': str(status_code)})
        with flow:
            flow.text(error)
        response = out.getvalue()
        super(MobileProxyXmlErrorResponse, self).__init__(
            response=response,
            format_=smart_text,
            status=status_code,
        )


class MobileProxyJSONSuccessfulResponse(JsonLoggedResponse):
    def __init__(self, *args, **kwargs):
        super(MobileProxyJSONSuccessfulResponse, self).__init__(*args, status='ok', **kwargs)


class MobileProxyJSONErrorResponse(JsonLoggedResponse):
    def __init__(self, status_code=400, *args, **kwargs):
        super(MobileProxyJSONErrorResponse, self).__init__(*args, status='error', **kwargs)
        self.status_code = status_code


class BaseMobileProxyView(View):
    @classmethod
    def as_view(cls, name=None, *args, **kwargs):
        name = name or cls.__name__
        return super(BaseMobileProxyView, cls).as_view(name, *args, **kwargs)

    def dispatch_request(self, **kwargs):
        try:
            return self.process_request()
        except MobileProxyError as e:
            return MobileProxyXmlErrorResponse(e.error, e.status)
        except (BlackboxTemporaryError, BaseOAuthError):
            return MobileProxyXmlErrorResponse('service temporarily unavailable', 500)
        except Exception as e:
            log_internal_error(e)
            return MobileProxyXmlErrorResponse('unknown error', 500)

    def process_request(self):
        raise NotImplementedError()  # pragma: no cover

    @staticmethod
    def get_required_param(name):
        value = request.values.get(name)
        if value is None:
            raise MobileProxyError(400, 'missing parameter \'%s\'' % name)
        elif not value:
            raise MobileProxyError(400, 'parameter \'%s\' is empty' % name)
        else:
            return value

    def add_request_params(self, to_form=False, **kwargs):
        source = 'form' if to_form else 'args'
        args = MultiDict(getattr(request, source))
        for arg_name, arg_value in iteritems(kwargs):
            args[arg_name] = arg_value
        setattr(request, source, ImmutableMultiDict(args))
