# -*- coding: utf-8 -*-
from flask import request
from passport.backend.api.common.oauth import get_token_issue_time
from passport.backend.api.views.bundle.constants import X_TOKEN_OAUTH_SCOPE
from passport.backend.core.builders.blackbox import (
    Blackbox,
    BLACKBOX_OAUTH_VALID_STATUS,
    get_alias,
    get_attribute,
    get_blackbox,
)
from passport.backend.core.conf import settings
from passport.backend.core.utils.decorators import cached_property

from .base import (
    BaseMobileProxyView,
    MobileProxyJSONSuccessfulResponse,
)
from .errors import MobileProxyError


class BaseMobileProxyBlackboxView(BaseMobileProxyView):
    """
    Общий класс для ручек, ходящих только в ЧЯ
    """
    @property
    def oauth_token(self):
        return self.get_required_param('token')

    @cached_property
    def blackbox(self):
        return get_blackbox()

    def check_token(self, require_xtoken=False):
        bb_response = self.blackbox.oauth(self.oauth_token, ip=request.env.user_ip)
        if bb_response['status'] != BLACKBOX_OAUTH_VALID_STATUS or not bb_response.get('uid'):
            raise MobileProxyError(
                401,
                'Oauth token invalid',
                'Blackbox error: %s' % bb_response.get('error'),
            )
        if require_xtoken and X_TOKEN_OAUTH_SCOPE not in bb_response['oauth']['scope']:
            raise MobileProxyError(
                400,
                'Invalid token: X-Token scope missing',
                'Scopes: %s' % bb_response['oauth']['scope'],
            )
        return bb_response


class MobileProxyGetLoginView(BaseMobileProxyBlackboxView):
    def process_request(self):
        bb_response = self.check_token()
        return MobileProxyJSONSuccessfulResponse(login=bb_response['display_login'])


class MobileProxyUserInfoView(BaseMobileProxyBlackboxView):
    def process_request(self):
        bb_response = self.check_token(require_xtoken=True)
        return MobileProxyJSONSuccessfulResponse(
            uid=bb_response['uid'],
            login=bb_response['display_login'],
            display_name=bb_response['display_name']['name'],
            default_avatar=settings.GET_AVATAR_URL % (
                bb_response['display_name']['avatar']['default'],
                request.values.get('size', 'normal'),
            ),
            is_avatar_empty=bb_response['display_name']['avatar']['empty'],
            staff=bool(get_alias(bb_response, 'yandexoid')),
            betatester=bool(get_attribute(bb_response, 'account.is_betatester')),
            has_plus=bool(get_attribute(bb_response, 'account.have_plus')),
            x_token_issued_at=get_token_issue_time(bb_response['oauth']),
        )
