# -*- coding: utf-8 -*-
from flask import request
from passport.backend.core.builders.blackbox import (
    BLACKBOX_SESSIONID_NEED_RESET_STATUS,
    BLACKBOX_SESSIONID_VALID_STATUS,
    get_blackbox,
)
from passport.backend.core.builders.oauth import get_oauth
from passport.backend.core.logging_utils.helpers import mask_sessionid
from passport.backend.core.logging_utils.loggers.statbox import to_statbox
from passport.backend.core.utils.decorators import cached_property

from .base import (
    BaseMobileProxyView,
    MobileProxyJSONErrorResponse,
    MobileProxyJSONSuccessfulResponse,
)


DEVICE_INFO_FIELDS = (
    'uuid',  # uuid приложения
    'deviceid',  # deviceid (0, если запрещен)
    'app_id',  # id приложения (package name для android, bundle id для айфона, что-то-там - для WP)
    'app_platform',  # платформа
    'manufacturer',  # вендор устройства
    'model',  # модель устройства
    'app_version',  # версия приложения
    'am_version',  # версия account manager
    'ifv',  # id for vendor, только для iOS
    'device_name',  # имя, заданное владельцем (только для iOS)
    'device_id',  # универсальный идентификатор устройства
)


class BaseMobileProxyOAuthView(BaseMobileProxyView):
    """
    Общий класс для ручек, ходящих в OAuth
    """
    @cached_property
    def oauth(self):
        return get_oauth()

    @staticmethod
    def get_common_oauth_kwargs():
        kwargs = dict(
            user_ip=request.env.user_ip,
            request_id=request.env.request_id,  # OAuth он не нужен, но будет проще грепать логи
        )
        for field in DEVICE_INFO_FIELDS:
            value = request.values.get(field)
            if value:
                kwargs[field] = value
        return kwargs


class MobileProxySocialTokenView(BaseMobileProxyOAuthView):
    @cached_property
    def blackbox(self):
        return get_blackbox()

    def process_request(self):
        """
        Раньше прокси ходил в oauth/token?grant_type=sessionid, а затем в blackbox?method=oauth.
        Теперь будем ходить в blackbox?method=sessionid, а затем в oauth?grant_type=passport_assertion.
        Цель - сэкономить один запрос в ЧЯ.
        """
        client_id = self.get_required_param('client_id')
        client_secret = self.get_required_param('client_secret')

        to_statbox({
            'mode': 'check_cookies',
            'host': self.get_required_param('host'),
            'client_id': client_id,
            'app_id': request.values.get('app_id'),
            'have_sessguard': request.values.get('sessguard') is not None,
            'sessionid': mask_sessionid(self.get_required_param('sessionid')),
        })

        bb_response = self.blackbox.sessionid(
            sessionid=self.get_required_param('sessionid'),
            host=self.get_required_param('host'),
            ip=request.env.user_ip,
            get_login_id=True,
        )
        if bb_response['status'] not in (
            BLACKBOX_SESSIONID_VALID_STATUS,
            BLACKBOX_SESSIONID_NEED_RESET_STATUS,
        ):
            # Эмулируем ответ OAuth grant_type=sessionid, поэтому нужно отдать JSON
            return MobileProxyJSONErrorResponse(
                error='invalid_grant',
                description='Session not valid',
            )
        display_name = bb_response['display_name']
        uid = bb_response['uid']
        login_id = bb_response['login_id']

        oauth_response = self.oauth.token_by_uid(
            client_id=client_id,
            client_secret=client_secret,
            uid=uid,
            login_id=login_id,
            **self.get_common_oauth_kwargs()
        )
        if not oauth_response.get('access_token'):
            # Тут тоже проксируем JSON-ответ сервиса
            return MobileProxyJSONErrorResponse(**oauth_response)

        return MobileProxyJSONSuccessfulResponse(
            oauth=dict(oauth_response, uid=uid),
            display_name=display_name,
        )
