# -*- coding: utf-8 -*-
import logging

from flask import (
    current_app,
    request,
)
from passport.backend.api.common.errors import error_handler
from passport.backend.api.env import APIEnvironment
from passport.backend.core.builders.blackbox import (
    BLACKBOX_OAUTH_VALID_STATUS,
    get_blackbox,
)
from passport.backend.core.conf import settings
from werkzeug.datastructures import EnvironHeaders

from .base import BaseMobileProxyView
from .errors import MobileProxyError


URL_PREFIX = '/mobileproxy/'


log = logging.getLogger(__name__)


class MobileProxySimplePassportView(BaseMobileProxyView):
    """
    Отрывает от урла префикс, проставляет нужные паспорту заголовки и
    прозрачно проксирует в аналогичную ручку паспортного АПИ.
    """
    def get_proxied_path(self, path):
        if not path.startswith(URL_PREFIX):
            raise RuntimeError('Urls misconfigured: mobile proxy prefix not found')  # pragma: no cover
        return path.replace(URL_PREFIX, '/')

    def get_proxied_view_and_kwargs(self):
        proxied_path = self.get_proxied_path(request.path)
        log.debug('Proxying to %s', proxied_path)

        urls = current_app.url_map.bind_to_environ(request.environ)
        # если соответствия не найдётся - match сам кинет нужный HTTPException
        endpoint, kwargs = urls.match(proxied_path, request.method)

        return current_app.view_functions[endpoint], kwargs

    def set_headers(self):
        """
        На первом этапе (nginx мобпрокси -> nginx паспорта -> питон):
          - IP пользователя nginx мобпрокси прокидывает как Ya-Consumer-Client-Ip паспортному nginx и питону.
          - Схему - аналогично.
          - IP потребителя паспортный nginx возьмёт в Remote-Addr и прокинет как X-Real-Ip питону.
        На втором этапе (nginx паспорта -> nginx паспорта -> питон) - всё то же самое (мобпроксёвый nginx
        переедет на отдельные паспортные машины)
        """
        # Выставляем все Ya-хедера (АПИ при проверке наличия обязательных хедеров не смотрит
        # на фолбеки вроде Host, Authorization, etc)
        new_environ = dict(
            request.environ,
            HTTP_YA_CONSUMER_REAL_IP=str(request.env.consumer_ip),
            HTTP_YA_CONSUMER_CLIENT_IP=str(request.env.user_ip),
        )
        # Если хедер Ya-Client-Host не передан, то пишем в него нечто на каком-либо
        # безкуковом домене (для имитации старого поведения)
        new_environ.setdefault('HTTP_YA_CLIENT_HOST', 'registrator.mobile.yandex.net')
        if request.env.authorization:
            new_environ['HTTP_YA_CONSUMER_AUTHORIZATION'] = request.env.authorization
        for header in ('User-Agent', 'Cookie', 'Accept-Language', 'Referer', 'X-Forwarded-For', 'X-Request-Id'):
            name = header.upper().replace('-', '_')
            value = request.environ.get('HTTP_%s' % name)
            if value is not None:
                new_environ.setdefault('HTTP_YA_CLIENT_%s' % name, value)

        request.environ = new_environ
        request.headers = EnvironHeaders(new_environ)
        # И ещё раз пересоберём env, чтобы учесть изменения в хедерах
        request.env = APIEnvironment.from_request(request)

    def process_request(self):
        self.set_headers()
        self.add_request_params(consumer=settings.MOBILEPROXY_CONSUMER)
        view, kwargs = self.get_proxied_view_and_kwargs()
        try:
            return view(**kwargs)
        except Exception as e:
            return error_handler(e)


class MobileProxyGetAccountFullInfoView(MobileProxySimplePassportView):
    def process_request(self):
        # Эти поля нужны не во всех сценариях, а получать их дорого, поэтому по дефолту их не отдаём
        for param in ('need_social_profiles', 'need_display_name_variants'):
            if not request.args.get(param):
                self.add_request_params(**{param: False})

        # Эти поля АМу не нужны - не разрешаем их отдавать
        self.add_request_params(
            need_phones=False,
            need_emails=False,
            need_question=False,
            need_additional_account_data=False,
            need_family_info=False,
            need_family_members=False,
        )
        return super(MobileProxyGetAccountFullInfoView, self).process_request()


class MobileProxyUidPassportView(MobileProxySimplePassportView):
    @property
    def oauth_token(self):
        return request.environ.get('HTTP_YA_OAUTH_HEADER')

    def get_oauth_info(self):
        if not self.oauth_token:
            raise MobileProxyError(400, 'Oauth token not set')

        bb_response = get_blackbox().oauth(self.oauth_token, ip=request.env.user_ip)
        if bb_response['status'] != BLACKBOX_OAUTH_VALID_STATUS or not bb_response.get('uid'):
            raise MobileProxyError(401, 'Oauth token invalid', 'Blackbox error: %s' % bb_response.get('error'))

        # TODO: начать тут требовать x-токен
        return bb_response

    def get_proxied_path(self, path):
        new_path = super(MobileProxyUidPassportView, self).get_proxied_path(path)
        info = self.get_oauth_info()
        return new_path.replace('account/', 'account/%s/' % info['uid'], 1)


class MobileProxyMobileRestoreLoginView(MobileProxySimplePassportView):
    def get_proxied_path(self, path):
        return '/1/bundle/restore/login/check_names/simple/'

    def process_request(self):
        if not request.form.get('allow_disabled'):
            self.add_request_params(to_form=True, allow_disabled='false')
        return super(MobileProxyMobileRestoreLoginView, self).process_request()


class MobileProxyCompleteSubmitView(MobileProxySimplePassportView):
    def process_request(self):
        if not request.form.get('can_handle_neophonish'):
            self.add_request_params(to_form=True, can_handle_neophonish='false')
        return super(MobileProxyCompleteSubmitView, self).process_request()


class MobileProxyStartCommitMagicView(MobileProxySimplePassportView):
    def process_request(self):
        if not request.form.get('need_oauth_token'):
            self.add_request_params(to_form=True, need_oauth_token='true')
        return super(MobileProxyStartCommitMagicView, self).process_request()


class MobileProxyPhoneBundleViewWithMaskedNumber(MobileProxySimplePassportView):
    def process_request(self):
        self.add_request_params(to_form=True, return_unmasked_number='false')
        return super(MobileProxyPhoneBundleViewWithMaskedNumber, self).process_request()


class MobileProxyCacheablePassportView(MobileProxySimplePassportView):
    """
    Обеспечивает кэширование ответа: позволяет отправлять ответ с пустым телом, если данные не изменились с момента
    предыдущего запроса. Подробнее: https://ru.wikipedia.org/wiki/HTTP_ETag
    """
    def process_request(self):
        response = super(MobileProxyCacheablePassportView, self).process_request()
        response.add_etag()
        return response.make_conditional(request)
