# -*- coding: utf-8 -*-
import logging
import os

from passport.backend.core.password.password_quality import (
    PasswordQualifier,
    STANDARD_SEQUENCES,
)
from passport.backend.core.password.policy import PASSWORD_PROHIBITED_SYMBOLS_RE


PASSWORD_QUALITY_LOWER_BOUND = 10
PASSWORD_QUALITY_UPPER_BOUND = 100


def configure_add_passwords_command(commander):
    commander.add_command(
        'add',
        add_passwords,
    ).add_argument(
        '-d', '--dir',
        dest='base_dir',
        type=str,
        default='../',
    ).add_argument(
        'file_name',
        metavar='FILE_WITH_NEW_PASSWORDS',
        type=str,
    )


def get_blacklist(blacklist_path):
    passwords = []
    with open(blacklist_path) as f:
        for line in f:
            line = line.strip()
            if not line:
                continue
            passwords.append(line)
    return passwords


def filter_passwords_by_quality(passwords, lower_bound, upper_bound, password_qualifier=None):
    password_qualifier = password_qualifier or PasswordQualifier(STANDARD_SEQUENCES, [])
    filtered_passwords = []
    for password in passwords:
        if PASSWORD_PROHIBITED_SYMBOLS_RE.search(password):
            continue
        quality = password_qualifier.get_quality(password, [], [])['quality']
        if quality < lower_bound or quality > upper_bound:
            continue
        filtered_passwords.append(password)
    return filtered_passwords


def add_passwords(base_dir, file_name):
    blacklist_path = os.path.join(base_dir, 'blacklist.txt')
    current_passwords = get_blacklist(blacklist_path)
    new_passwords = get_blacklist(file_name)

    passwords = set(current_passwords).union(new_passwords)
    passwords = filter_passwords_by_quality(
        passwords,
        PASSWORD_QUALITY_LOWER_BOUND,
        PASSWORD_QUALITY_UPPER_BOUND,
    )

    if sorted(current_passwords) == sorted(passwords):
        logging.info('Current blacklist is identical with new one')
        return

    logging.info('Updating %s...', blacklist_path)
    with open(blacklist_path, 'w') as f:
        f.write('\n'.join(sorted(passwords)) + '\n')

    logging.info('Password blacklist updated.  Old size: %s. New size: %s', len(current_passwords), len(passwords))
