# -*- coding: utf-8 -*-

import codecs
import json
import logging
import os
import re
import sys

from passport.backend.api.settings.settings import (
    DISPLAY_LANGUAGES,
    SSL_CA_CERT,
    translations,
)
import requests


def fetch(url, project_id, keyset_id, msgid_re, trim_keys=False):
    msgid_re = re.compile(msgid_re, re.UNICODE)
    logging.info('Fetching %s', url)
    resp = requests.get(
        url,
        params={
            'keyset-id': keyset_id,
            'project-id': project_id,
            'safe': '1',
        },
        verify=SSL_CA_CERT,
    )
    json_data = json.loads(resp.content)
    result = {}
    for msgid, data in json_data['keysets'][keyset_id]['keys'].items():
        match = msgid_re.match(msgid)
        if match is not None:
            logging.info('Adding %s (key: %s)', msgid, match.groups()[0])
            for lang in DISPLAY_LANGUAGES:
                translation = data['translations'].get(lang, {}).get('form', '')
                if lang not in result:
                    result[lang] = {}
                tr_key = match.groups()[0]
                if trim_keys:
                    tr_key = tr_key.strip()
                result[lang][tr_key] = translation
        else:
            logging.debug('Ignoring %s', msgid)
    return result


def update_translations(trim_keys, base_dir, url):
    for name, params in translations.TRANSLATIONS.items():
        data = fetch(url=url, trim_keys=trim_keys, **params)
        if set(data) != set(DISPLAY_LANGUAGES):
            logging.error('Translations for %s doesn\'t match DISPLAY_LANGUAGES', name)
            sys.exit(1)
        fname = os.path.join(base_dir, '%s.json' % name)
        with codecs.open(fname, 'w', 'utf-8') as f:
            json.dump(data, f, indent=4, ensure_ascii=False, sort_keys=True)
            f.write('\n')


def configure_update_translations_command(commander):
    logging.basicConfig(format='%(message)s', level=logging.INFO)

    commander.add_command(
        'update-translations',
        update_translations,
    ).add_argument(
        '-t', '--trim-keys',
        dest='trim_keys',
        action='store_true',
        default=False,
        help='Trim leading spaces from keys.',
    ).add_argument(
        '-b', '--base-dir',
        dest='base_dir',
        type=str,
        default='../',
    ).add_argument(
        '-u', '--url',
        dest='url',
        type=str,
        default=translations.TANKER_BASE_URL + '/keysets/tjson/',
    )
