# -*- coding: utf-8 -*-
import jinja2
from jinja2 import nodes
import jinja2.ext
from passport.backend.core.conf import settings
from six import iteritems


@jinja2.contextfunction
def l(context, keyset, key, language=None):
    """Shotcut от localization"""
    """Используется в шаблонах для получения перевода по кейсету и ключу"""
    if not language:
        language = context['language']
    return getattr(settings.translations, keyset, {}).get(language, {}).get(key, '')


def get_template(template_text, **kwargs):
    """Просто, но нужно для тестов, например"""
    return jinja2.Template(template_text, autoescape=False, **kwargs)


@jinja2.contextfunction
def T(context, text):
    """Shotcut от Template"""
    """Используется в шаблонах для промежуточного процессирования подшаблонов, например переводов"""
    template = get_template(text, variable_start_string='%', variable_end_string='%')
    return template.render(**context)


def escape_percents(string):
    return string.replace('%', '&#37;')


def escape_domain(domain, style=''):
    return domain.replace('.', '<span style=\'%s\'>.</span>' % style)


def span(login, style=''):
    return '<span style=\'%s\'>%s</span>' % (style, login)


def yandex_login(login):
    return span(login[0], 'color: #ff0000;') + span(login[1:])


def markdown_email(email):
    mailbox, hostname = email.split('@')
    hostname = '<span>.</span>'.join(hostname.split('.'))
    mailbox = '<span>%s</span>' % mailbox
    return '%s@%s' % (mailbox, hostname)


@jinja2.contextfunction
def gettext(context, key, keyset=None, language=None):
    # Не берём запасной keyset из контекста, иначе будет возможно для одного
    # перевода в шаблоне передавать разные keyset'ы.
    if keyset is None:
        keyset = context.environment.tanker_default_keyset

    if language is None:
        language = context.get('_tanker_default_language')
    if language is None:
        language = context.get('language')
    if language is None:
        language = context.environment.tanker_default_language

    return getattr(settings.translations, keyset, {}).get(language, {}).get(key, '')


def tags():
    """Эта функция используется для экспорта своих функций в шаблонные функции"""
    return dict(
        T=T,
        l=l,
        span=span,
        yandex_login=yandex_login,
        escape_domain=escape_domain,
        escape_percents=escape_percents,
        gettext=gettext,
        markdown_email=markdown_email,
    )


class TankerI18nExtension(jinja2.ext.i18n):
    def __init__(self, environment):
        super(TankerI18nExtension, self).__init__(environment)
        environment.extend(
            tanker_default_keyset=None,
            tanker_default_language=None,
        )

    def _make_node(self, singular, plural, variables, plural_expr,
                   vars_referenced, num_called_num):
        """Generates a useful node from the data provided."""
        # no variables referenced?  no need to escape for old style
        # gettext invocations only if there are vars.
        if not vars_referenced and not self.environment.newstyle_gettext:
            singular = singular.replace('%%', '%')
            if plural:
                plural = plural.replace('%%', '%')

        keyset = nodes.Const(variables.pop('_keyset', None))
        language = nodes.Const(variables.pop('_language', None))

        # singular only:
        if plural_expr is None:
            gettext = nodes.Name('gettext', 'load')
            node = nodes.Call(
                gettext,
                [nodes.Const(singular)],
                [
                    nodes.Keyword('keyset', keyset),
                    nodes.Keyword('language', language),
                ],
                None,
                None,
            )

        # singular and plural
        else:
            ngettext = nodes.Name('ngettext', 'load')
            node = nodes.Call(
                ngettext,
                [
                    nodes.Const(singular),
                    nodes.Const(plural),
                    plural_expr
                ],
                [
                    nodes.Keyword('keyset', keyset),
                    nodes.Keyword('language', language),
                ],
                None,
                None,
            )

        # in case newstyle gettext is used, the method is powerful
        # enough to handle the variable expansion and autoescape
        # handling itself
        if self.environment.newstyle_gettext:
            for key, value in iteritems(variables):
                # the function adds that later anyways in case num was
                # called num, so just skip it.
                if num_called_num and key == 'num':
                    continue
                node.kwargs.append(nodes.Keyword(key, value))

        # otherwise do that here
        else:
            # mark the return value as safe if we are in an
            # environment with autoescaping turned on
            node = nodes.MarkSafeIfAutoescape(node)
            if variables:
                node = nodes.Mod(node, nodes.Dict([
                    nodes.Pair(nodes.Const(key), value)
                    for key, value in variables.items()
                ]))
        return nodes.Output([node])


def extensions():
    return [TankerI18nExtension]
