# -*- coding: utf-8 -*-

from passport.backend.core import validators
from passport.backend.core.builders.blackbox.utils import user_exists
from passport.backend.core.logging_utils.loggers.statbox import to_statbox
from passport.backend.core.validators import _


class FieldOnlyForSids(validators.FormValidator):
    '''
    Проверяет, что указанное поле встречается только в сочетании
    с конкретным sid'ом
    '''

    messages = dict(
        empty='Please enter a value',
        notEmpty='You cannot enter a value here',
    )

    def __init__(self, param, sids, required=True, required_checker=None):
        self._param = param
        self._sids = sids
        self._required = required
        self._required_checker = required_checker

    def _to_python(self, value_dict, state):
        if value_dict.get('service') is None:
            value = value_dict.get(self._param)
            if value is not None:
                message = self.message('notEmpty', state)
                raise validators.Invalid(
                    message,
                    value,
                    state,
                    error_dict={self._param: validators.Invalid(message, value, state)},
                )

            return value_dict

        sid = value_dict.get('service').sid

        value = value_dict.get(self._param)
        # сделано для admsubscribe: параметры не надо
        # проверять при отписке от сервиса (поле формы unsubscribe)
        if self._required_checker:
            required = self._required_checker(value_dict)
        else:
            required = self._required

        if sid in self._sids and required and value is None:
            message = self.message('empty', state)
            raise validators.Invalid(
                message,
                value,
                state,
                error_dict={self._param: validators.Invalid(message, value, state)},
            )
        elif sid not in self._sids and value is not None:
            message = self.message('notEmpty', state)
            raise validators.Invalid(
                message,
                value,
                state,
                error_dict={self._param: validators.Invalid(message, value, state)},
            )
        return value_dict


class AvailabilityOnSid(validators.FormValidator):
    messages = {
        'notAvailable': u'Login `%(login)s` is not available on sid=%(sid)s',
    }

    def __init__(self, sid, login_field):
        self._sid = sid
        self._login_field = login_field

    def _to_python(self, value_dict, state):
        login = value_dict.get(self._login_field)
        if login is None:
            return value_dict
        if user_exists(login, sid=self._sid):
            message = self.message('notAvailable', state, sid=self._sid, login=login)
            raise validators.Invalid(
                message,
                value_dict,
                state,
                error_dict={
                    self._login_field: validators.Invalid(message, value_dict, state),
                },
            )
        return value_dict


class TrackFlagsList(validators.FormValidator):

    messages = {
        'badFlagValue': _('One or more flags have non-boolean value'),
    }
    flag_validator = validators.StringBool(not_empty=True)

    def _to_python(self, value_dict, state):
        new_value_dict = dict(value_dict)

        error_dict = {}

        for key in value_dict:
            try:
                new_value_dict[key] = self.flag_validator.to_python(value_dict[key])
            except validators.Invalid as e:
                error_dict[key] = e

        if error_dict:
            raise validators.Invalid(
                self.message('badFlagValue', state),
                value_dict,
                state,
                error_dict=error_dict,
            )

        return new_value_dict


class Password(validators.Password):
    QUALITY_FIELDS_TO_STATBOX = [
        'quality',
        'length',
        'classes_number',
        'sequences_number',
        'is_sequence',
        'is_word',
        'is_additional_word',
        'additional_subwords_number',
    ]

    STATBOX_QUALITY_FIELDS_MAPPER = {
        'quality': 'password_quality',
    }

    STATBOX_ERROR_CODES_MAPPER = {
        'weak': 'weak',
        'likeLogin': 'like_login',
        'likeOldPassword': 'like_old_password',
        'foundInHistory': 'found_in_history',
        'likePhoneNumber': 'like_phone_number',
    }

    def get_statbox_params(self, error_codes, value_dict, state):
        params = {
            'track_id': value_dict['track_id'],
            'action': 'password_validation_error',
            'policy': value_dict.get(self._policy_name_field, 'basic'),
        }
        quality = state.password_quality or {}
        for key, value in quality.items():
            if key not in self.QUALITY_FIELDS_TO_STATBOX:
                continue
            statbox_field = self.STATBOX_QUALITY_FIELDS_MAPPER.get(key, key)
            params[statbox_field] = value
        for code in error_codes:
            statbox_field = self.STATBOX_ERROR_CODES_MAPPER[code]
            params[statbox_field] = True

        return params

    def _to_python(self, value_dict, state):
        try:
            return super(Password, self)._to_python(value_dict, state)
        except validators.Invalid as e:
            errors = e.error_dict[self._password_field].error_list
            filtered_error_codes = [error.code for error in errors if error.code in self.STATBOX_ERROR_CODES_MAPPER]
            if filtered_error_codes and value_dict.get('track_id'):
                to_statbox(self.get_statbox_params(filtered_error_codes, value_dict, state))
            raise e
