# -*- coding: utf-8 -*-
import logging

from flask import request
from passport.backend.api import forms
from passport.backend.api.common.account import fill_person_from_args
from passport.backend.api.common.common import CleanWebChecker
from passport.backend.api.common.decorators import validate
from passport.backend.api.common.format_response import ok_response
from passport.backend.api.common.grants import add_grants
from passport.backend.api.common.logs import setup_log_prefix
from passport.backend.core.builders import blackbox
from passport.backend.core.models.account import Account
from passport.backend.core.models.person import Person
from passport.backend.core.runner.context_managers import UPDATE

from .grants import grants


log = logging.getLogger('passport.api.views')


def is_admin_action(args):
    return (
        args['admin_name'] is not None and
        args['comment'] is not None
    )


def admin_action_to_events(args, events):
    events.update(
        admin=args['admin_name'],
        comment=args['comment'],
    )


def mark_admin_action(args, events):
    if is_admin_action(args):
        admin_action_to_events(args, events)


@validate(forms.Karma())
@grants([
    'karma',
    add_grants(['admin.log_action'], if_arg='admin_name'),
])
def karma(args):
    prefix = args['prefix']
    suffix = args['suffix']
    uid = args['uid']

    data = blackbox.get_blackbox().userinfo(uid=uid)

    acc = Account().parse(data)
    setup_log_prefix(acc)

    events = {
        'action': 'karma',
        'consumer': args['consumer'],
    }

    mark_admin_action(args, events)

    with UPDATE(acc, request.env, events):
        if prefix is not None:
            acc.karma.prefix = int(prefix)
        if suffix is not None:
            acc.karma.suffix = int(suffix)

    return ok_response()


@validate(forms.Account())
@grants([
    add_grants(['account.is_enabled'], if_arg='is_enabled', default_values=[None]),
    add_grants(['admin.log_action'], if_arg='admin_name'),
    add_grants(['account.enable_app_password__full_control'], if_arg='is_enabled_app_password', default_values=[None])
])
def account(args):
    data = blackbox.get_blackbox().userinfo(uid=args['uid'])

    acc = Account().parse(data)
    setup_log_prefix(acc)

    events = {
        'action': 'account',
        'consumer': args['consumer'],
    }

    mark_admin_action(args, events)
    with UPDATE(acc, request.env, events):
        if args['is_enabled'] is not None:
            acc.is_enabled = int(args['is_enabled'])
        if args['is_enabled_app_password'] is not None:
            acc.enable_app_password = args['is_enabled_app_password']

    return ok_response()


@validate(forms.Person())
@grants(['person'])
def person(args):
    CleanWebChecker().check_form_values(args)
    data = blackbox.get_blackbox().userinfo(uid=args['uid'])

    account = Account().parse(data)
    setup_log_prefix(account)

    contact_phone_number = args['contact_phone_number']
    with UPDATE(account, request.env, {'action': 'person', 'consumer': args['consumer']}):
        account.person = fill_person_from_args(Person(account), args)

        if contact_phone_number:
            contact_phone_number = contact_phone_number.e164
            account.person.contact_phone_number = contact_phone_number

    return ok_response()


__all__ = (
    'karma',
    'account',
    'person',
)
