# -*- coding: utf-8 -*-
import logging

from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    AccountInvalidTypeError,
    ActionImpossibleError,
)
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleFederalMixin,
)
from passport.backend.api.views.bundle.mixins.common import BundleAdminActionMixin
from passport.backend.core.runner.context_managers import UPDATE

from .forms import AccountOptionsForm


log = logging.getLogger('passport.api.view.bundle.account.change_options')

ACCOUNT_OPTIONS_BASE_GRANT = 'account_options.base'
ACCOUNT_OPTIONS_ANY_GRANT = 'account_options.any'
ACCOUNT_OPTIONS_NORMAL_GRANT = 'account_options.normal'
ACCOUNT_OPTIONS_PDD_GRANT = 'account_options.pdd'

ACCOUNT_IS_ENABLED_GRANT = 'account.is_enabled'
ACCOUNT_IS_MAILLIST_GRANT = 'account.is_maillist'
ACCOUNT_IS_APP_PASSWORD_ENABLED_GRANT = 'account.enable_app_password__full_control'
ACCOUNT_IS_SHARED_GRANT = 'account.is_shared'
LOG_ADMIN_ACTION_GRANT = 'admin.log_action'


class AccountOptionsViewV1(BaseBundleView, BundleAccountGetterMixin, BundleAdminActionMixin, BundleFederalMixin):

    require_track = False
    required_grants = [ACCOUNT_OPTIONS_BASE_GRANT]

    basic_form = AccountOptionsForm

    grants_for_account_type = {
        'any': ACCOUNT_OPTIONS_ANY_GRANT,
        'normal': ACCOUNT_OPTIONS_NORMAL_GRANT,
        'pdd': ACCOUNT_OPTIONS_PDD_GRANT,
    }

    def process_request(self, *args, **kwargs):
        self.process_basic_form()
        if self.is_admin_action:
            self.check_grant(LOG_ADMIN_ACTION_GRANT)

        self.get_account_by_uid(
            self.form_values['uid'],
            enabled_required=False,
        )

        self.check_grants_for_account_type()

        is_enabled = self.form_values.get('is_enabled')
        is_app_password_enabled = self.form_values.get('is_app_password_enabled')
        is_shared = self.form_values.get('is_shared')
        is_maillist = self.form_values.get('is_maillist')

        events = {
            'action': 'account',
            'consumer': self.consumer,
        }

        self.mark_admin_action(events)

        with UPDATE(self.account, self.request.env, events):
            if is_enabled is not None:
                self.check_grant(ACCOUNT_IS_ENABLED_GRANT)
                self.account.is_enabled = is_enabled

            if is_app_password_enabled is not None:
                self.check_grant(ACCOUNT_IS_APP_PASSWORD_ENABLED_GRANT)
                self.account.enable_app_password = is_app_password_enabled

            if is_shared is not None:
                self.check_grant(ACCOUNT_IS_SHARED_GRANT)
                self.account.is_shared = is_shared

            if is_maillist is not None:
                self.check_grant(ACCOUNT_IS_MAILLIST_GRANT)
                # Сделать "подпиской" можно только ПДДшника, но не
                # админа ПДД-домена/Коннекта.
                if not self.account.is_pdd or self.account.is_pdd_admin or self.account.is_connect_admin:
                    raise AccountInvalidTypeError()
                if not is_maillist and self.get_saml_settings(domain_id=self.account.domain.id, only_enabled=True):
                    raise ActionImpossibleError()

                self.account.is_maillist = is_maillist
                self.account.is_employee = not is_maillist
