# -*- coding: utf-8 -*-
import logging

from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    AccountInvalidTypeError,
    AccountNotSubscribedError,
    DomainInvalidTypeError,
    DomainNotFoundError,
)
from passport.backend.api.views.bundle.mixins import BundleAccountGetterMixin
from passport.backend.api.yasms.api import SaveSimplePhone
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers import StatboxLogger
from passport.backend.core.models.alias import (
    AltDomainAlias,
    BankPhoneNumberAlias,
    PddAlias,
)
from passport.backend.core.models.domain import Domain
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.services import Service
from passport.backend.core.utils.decorators import cached_property

from .exceptions import (
    AliasExistsError,
    AliasNotAvailableError,
    AliasNotFoundError,
    MaxNumbersOfAliasesExceeded,
    PhoneForAliasNotBoundError,
)
from .forms import (
    AccountAliasCreateForm,
    AccountAliasDeleteForm,
    AccountBankPhoneNumberAliasCreateForm,
    AccountBankPhoneNumberAliasDeleteForm,
    AccountPddAliasLoginForm,
    AccountPDDCreateForm,
)


log = logging.getLogger('passport.api.view.bundle.account.aliases')

ALIASES_BASE_GRANT = 'aliases.base'
ALIASES_ALTDOMAIN_GRANT = 'aliases.altdomain'
ALIASES_PDDALIAS_GRANT = 'aliases.pddalias'
ALIAS_BANK_PHONENUMBER_GRANT = 'aliases.bank_phonenumber'


class BaseAccountAliasView(BaseBundleView, BundleAccountGetterMixin):

    require_track = False
    required_grants = [ALIASES_BASE_GRANT]

    def _process_alias(self):
        raise NotImplementedError()  # pragma: no cover

    def process_request(self, *args, **kwargs):
        self.process_basic_form()
        self.get_account_by_uid(
            self.form_values['uid'],
            enabled_required=False,
        )

        events = {
            'action': 'alias',
            'consumer': self.consumer,
        }

        with UPDATE(self.account, self.request.env, events):
            self._process_alias()


class AccountAliasCreateView(BaseAccountAliasView):

    basic_form = AccountAliasCreateForm

    def _process_alias(self):
        alias = self.form_values['alias']

        self.check_grant(ALIASES_ALTDOMAIN_GRANT)
        if self.account.altdomain_alias and self.account.altdomain_alias.alias:
            raise AliasExistsError()
        self.account.altdomain_alias = AltDomainAlias(
            parent=self.account,
            login=alias,
        )


class AccountPddCreateView(BaseAccountAliasView):

    basic_form = AccountPDDCreateForm

    def _process_alias(self):
        alias = self.form_values['alias']

        self.check_grant(ALIASES_PDDALIAS_GRANT)
        # соответствует ли переданный пользователь требованиям
        if not self.account.is_normal:
            raise AccountInvalidTypeError()

        mailpro_sid = Service.by_slug('mailpro').sid
        if not self.account.has_sid(mailpro_sid):
            raise AccountNotSubscribedError()

        if self.account.pdd_alias:
            raise AliasExistsError()

        # соответствует ли переданный алиас требованиям
        availability_info = self.blackbox.loginoccupation(
            [alias],
            is_pdd=True,
        )
        if availability_info[alias]['status'] != 'free':
            raise AliasNotAvailableError()

        login, domain_name = alias.split('@', 1)
        domain_info = self.blackbox.hosted_domains(domain=domain_name)
        if not domain_info['hosted_domains']:
            raise DomainNotFoundError()

        domain = Domain().parse(domain_info)
        if domain.master_domain:
            raise DomainInvalidTypeError()

        self.account.domain = domain
        self.account.pdd_alias = PddAlias(
            parent=self.account,
            email=alias,
        )


class AccountAliasDeleteView(BaseAccountAliasView):

    basic_form = AccountAliasDeleteForm

    def _process_alias(self):
        alias_type = self.form_values['alias_type']

        if alias_type == 'altdomain':
            self.check_grant(ALIASES_ALTDOMAIN_GRANT)
            self.account.altdomain_alias = None
        elif alias_type == 'pdddomain':
            self.check_grant(ALIASES_PDDALIAS_GRANT)
            if not self.account.is_normal:
                raise AccountInvalidTypeError()
            if not self.account.pdd_alias:
                raise AliasNotFoundError()
            self.account.pdd_alias = None


class BaseAccountPddAliasLoginView(BaseAccountAliasView):

    basic_form = AccountPddAliasLoginForm
    required_grants = [ALIASES_PDDALIAS_GRANT]

    def _process_alias(self):
        if not self.account.is_pdd:
            raise AccountInvalidTypeError()
        self._process_login()


class AccountPddAliasLoginCreateView(BaseAccountPddAliasLoginView):

    def _process_login(self):
        login = self.form_values['login']
        serialized_alias = '%s@%s' % (
            login,
            self.account.domain.domain,
        )
        availability_info = self.blackbox.loginoccupation(
            [serialized_alias],
            is_pdd=True,
        )
        if availability_info[serialized_alias]['status'] != 'free':
            raise AliasExistsError()

        if len(self.account.pdd_alias.additional_logins) >= settings.MAX_PDD_ALIASES_COUNT:
            raise MaxNumbersOfAliasesExceeded()

        self.account.pdd_alias.additional_logins.add(login)


class AccountPddAliasLoginDeleteView(BaseAccountPddAliasLoginView):

    def _process_login(self):
        login = self.form_values['login']
        if self.account.pdd_alias.has_login(login):
            self.account.pdd_alias.remove_login(login)
        else:
            raise AliasNotFoundError()


class AccountBankPhoneNumberAliasCreateView(BaseAccountAliasView):
    basic_form = AccountBankPhoneNumberAliasCreateForm
    phone = None

    @cached_property
    def bind_statbox(self):
        return StatboxLogger(
            uid=self.account.uid,
            mode='bank_alias_bind_phone',
            consumer=self.consumer,
        )

    def check_bank_phonenumber_unique(self, phone_number):
        availability_info = self.blackbox.userinfo(
            login=phone_number,
            sid='bank',
        )
        if availability_info['uid'] is not None:
            raise AliasNotAvailableError()

    def bind_phone(self):
        self.statbox = StatboxLogger(
            mode='account_bank_phonenumber_alias_bind_phone',
            uid=self.account.uid,
            ip=self.client_ip,
        )

        # так как телефона нет и не привязан, то вызовется BindSimplePhone
        phone_saver = SaveSimplePhone(
            account=self.account,
            blackbox=self.blackbox,
            consumer=self.consumer,
            env=self.request.env,
            phone_number=self.form_values['phone_number'],
            statbox=self.statbox,
            yasms=self.yasms,
            washing_enabled=False,
        )
        phone_saver.submit()

        with UPDATE(self.account, self.request.env, {'action': 'create_and_bind_phone', 'consumer': self.consumer}):
            phone_saver.commit()

        self.statbox.log(
            action='phone_bound',
            uid=self.account.uid,
        )
        phone_saver.after_commit()

    def _process_phone_by_id(self):
        phone_id = self.form_values['phone_id']

        if self.account.phones.by_id(phone_id, assert_exists=False):
            phone = self.account.phones.by_id(phone_id)
            self.check_bank_phonenumber_unique(phone.number.digital)
            self.phone = phone
        else:
            raise PhoneForAliasNotBoundError()

    def _process_phone_by_number(self):
        phone_number = self.form_values['phone_number']

        self.check_bank_phonenumber_unique(phone_number.digital)

        if not self.account.phones.by_number(phone_number):
            self.bind_phone()

        self.phone = self.account.phones.by_number(phone_number)

    def _process_alias(self):
        self.account.bank_phonenumber_alias = BankPhoneNumberAlias(
            parent=self.account,
            alias=self.phone.number.digital,
        )

    def process_request(self, *args, **kwargs):
        self.process_basic_form()
        self.get_account_by_uid(
            self.form_values['uid'],
            enabled_required=False,
            need_phones=True,
        )

        # Запрет добавления банковского номера аккаутам созданным админами ПДД
        if self.account.is_pdd or self.account.is_federal:
            raise AccountInvalidTypeError()

        self.check_grant(ALIAS_BANK_PHONENUMBER_GRANT)

        if self.account.bank_phonenumber_alias:
            raise AliasExistsError()

        if self.form_values['phone_id']:
            self._process_phone_by_id()
        else:
            self._process_phone_by_number()

        events = {
            'action': 'alias',
            'consumer': self.consumer,
        }

        with UPDATE(self.account, self.request.env, events):
            self._process_alias()


class AccountBankPhoneNumberAliasDeleteView(BaseAccountAliasView):
    basic_form = AccountBankPhoneNumberAliasDeleteForm

    def _process_alias(self):
        self.check_grant(ALIAS_BANK_PHONENUMBER_GRANT)

        if not self.account.bank_phonenumber_alias:
            raise AliasNotFoundError()

        self.account.bank_phonenumber_alias = None
