# -*- coding: utf-8 -*-
from datetime import datetime

from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    AccountWithoutPasswordError,
    ActionImpossibleError,
    ActionNotRequiredError,
    PasswordRequiredError,
)
from passport.backend.api.views.bundle.headers import AUTHORIZATION_HEADERS
from passport.backend.api.views.bundle.mixins import (
    BundleAccountGetterMixin,
    BundleAccountPropertiesMixin,
)
from passport.backend.core.conf import settings
from passport.backend.core.logging_utils.loggers.statbox import StatboxLogger
from passport.backend.core.mailer.utils import (
    get_tld_by_country,
    login_shadower,
    MailInfo,
    make_email_context,
    send_mail_for_account,
)
from passport.backend.core.models.account import get_preferred_language
from passport.backend.core.runner.context_managers import UPDATE
from passport.backend.core.utils.decorators import cached_property


GRANT = 'account.app_passwords_switch'
MODE = 'app_passwords'


class AppPasswordsSwitch(BaseBundleView,
                         BundleAccountPropertiesMixin,
                         BundleAccountGetterMixin):
    require_track = True
    required_headers = AUTHORIZATION_HEADERS
    required_grants = [GRANT]
    track_type = 'authorize'

    switch_name = None
    switch_position = None

    email_template = None
    email_subject_key = None

    def process_request(self):
        self.read_track()
        self.get_pinned_account_from_session(emails=True)
        self.assert_required_conditions()
        self.switch()
        self.log_statbox()
        self.send_email_notifications()

    def assert_required_conditions(self):
        if not self.account.is_federal:
            if self.is_password_verification_required():
                raise PasswordRequiredError()

            if not self.account.have_password:
                raise AccountWithoutPasswordError()

        if self.account.enable_app_password == self.switch_position:
            raise ActionNotRequiredError()

        if self.account.totp_secret.is_set and not self.switch_position:
            # Кейс недостижимый, поэтому отдаём базовую ошибку
            raise ActionImpossibleError()

    def switch(self):
        events = {
            'action': '.'.join([MODE, self.switch_name]),
            'consumer': self.consumer,
        }

        with UPDATE(self.account, self.request.env, events):
            self.account.enable_app_password = self.switch_position
            if not self.switch_position:
                # Отзываем ПП только при выключении
                self.account.app_passwords_revoked_at = datetime.now()

    def log_statbox(self):
        self.statbox.log(uid=self.account.uid)

    @cached_property
    def statbox(self):
        return StatboxLogger(
            consumer=self.consumer,
            mode=MODE,
            action=self.switch_name,
            track_id=self.track_id,
            yandexuid=self.cookies.get('yandexuid'),
            user_agent=self.user_agent,
            ip=self.client_ip,
            origin=self.track.origin,
        )

    def send_email_notifications(self):
        language = get_preferred_language(self.account)
        template_name, info, context = self.get_email_notification_data(language)
        send_mail_for_account(template_name, info, context, self.account, login_shadower, send_to_native=True)

    def get_email_notification_data(self, language):
        """Возвращает template_name, info, context"""
        translations = settings.translations.NOTIFICATIONS[language]
        template_name = self.email_template
        info = MailInfo(
            subject=translations[self.email_subject_key],
            from_=translations['email_sender_display_name'],
            tld=get_tld_by_country(self.account.person.country),
        )
        context = make_email_context(
            language=language,
            account=self.account,
        )
        return template_name, info, context


class AppPasswordsActivator(AppPasswordsSwitch):
    switch_name = 'activate'
    switch_position = True

    email_template = 'mail/app_passwords_enabled_notification.html'
    email_subject_key = 'app_passwords_enabled.subject'


class AppPasswordsDeactivator(AppPasswordsSwitch):
    switch_name = 'deactivate'
    switch_position = False

    email_template = 'mail/app_passwords_disabled_notification.html'
    email_subject_key = 'app_passwords_disabled.subject'
