# -*- coding: utf-8 -*-

from passport.backend.api.views.bundle.base import BaseBundleView
from passport.backend.api.views.bundle.exceptions import (
    AccountDisabledError,
    AccountInvalidTypeError,
    AccountNotFoundError,
    OAuthTokenValidationError,
)
from passport.backend.api.views.bundle.headers import (
    HEADER_CONSUMER_AUTHORIZATION,
    HEADER_CONSUMER_CLIENT_IP,
)
from passport.backend.api.views.bundle.mixins import BundleAccountGetterMixin
from passport.backend.core.builders.blackbox.constants import BLACKBOX_OAUTH_VALID_STATUS

from .forms import AccountCheckExistsForm


ACCOUNT_CHECK_EXISTS_GRANT = 'account.check_exists'

ACCOUNT_CHECK_EXISTS_BY_EMAIL_SCOPE = 'passport:check-email'


class AccountCheckExistsView(BaseBundleView,
                             BundleAccountGetterMixin):
    """
    Ручка проверки существования аккаунта с заданным email-адресом для WG
    """
    basic_form = AccountCheckExistsForm
    required_grants = [ACCOUNT_CHECK_EXISTS_GRANT]
    required_headers = [HEADER_CONSUMER_CLIENT_IP, HEADER_CONSUMER_AUTHORIZATION]

    def is_account_type_allowed(self):
        return self.account.is_normal

    def process_request(self):
        self.process_basic_form()

        response = self.blackbox.oauth(
            self.oauth_token,
            ip=self.client_ip,
        )
        if response['status'] != BLACKBOX_OAUTH_VALID_STATUS:
            raise OAuthTokenValidationError()

        if ACCOUNT_CHECK_EXISTS_BY_EMAIL_SCOPE not in response['oauth']['scope']:
            raise OAuthTokenValidationError()

        if response['uid'] is not None:
            # Не допускаем персонализированных токенов - для их получения достаточно узнать client_id приложения
            raise OAuthTokenValidationError()

        exists = True
        try:
            self.get_account_by_login(
                login=self.form_values['email'],
                find_by_phone_alias=None,
            )
        except (AccountNotFoundError, AccountDisabledError, AccountInvalidTypeError):
            exists = False

        self.response_values['exists'] = exists
